
#include <iostream>
#include <iomanip>
#include <fstream>
#include <string>
#include <stdlib.h>
#include <math.h>

using namespace std;


#include "constants.hxx"
#include "star.hxx"
#include "planet.hxx"
#include "terrain_relief.hxx"
#include "body.hxx"
#include "options.hxx"
#include "material.hxx"
#include "orbital_solver.hxx"
#include "utilities.hxx"
#include "checks.hxx"
//#include "geology.hxx"


// init the definitions of the main objects - central star, binary or companion star, parent planet and possibly moon

Star central_sun;
Star companion;
Star binary;
Planet planet;
Planet moon;
Body *body_array;
Options options_list;
OrbitalSystem solver;
TerrainRelief relief;

int n_materials;
int id_star, id_companion, id_binary, id_planet, id_moon;
Material materials[20];

#include "parser.hxx"




string commandline_parser (int argc, char *argv[]) 
{ 
string config_file;

if (argc != 2)
	{
	cout <<"Please specify a configuration file!" << endl;
	}
else
	{
	config_file = argv[1];
	cout << "Reading " << config_file << " for configuration." << endl;
	}

return config_file;

}


void assign_body_indices (void)
{

if (options_list.plot_orbit_defined == false) {return;}

for (int i=0; i< options_list.num_bodies;i++)
	{
	if (options_list.plot_orbit_body == body_array[i].get_designation())
		{
		options_list.plot_orbit_body_index = body_array[i].get_ref_index();
		//cout << "Assigning index " << options_list.plot_orbit_body_index << " for plotting object " << options_list.plot_orbit_body << endl;
		break;
		}
	}


if (options_list.plot_orbit_relative)
	{
	for (int i=0; i< options_list.num_bodies;i++)
		{
		if (options_list.plot_orbit_reference == body_array[i].get_designation())
			{
			options_list.plot_orbit_ref_index = body_array[i].get_ref_index();
			//cout << "Assigning index " << options_list.plot_orbit_body_index << " for referencing object " << options_list.plot_orbit_reference << endl;
			break;
			}
		}

	
	}
}



int system_type_mapper (void)
{
if (options_list.planet_orbit_defined == false)
	{
	return -2; // undefined
	}
else if (options_list.moon_orbit_defined == false)
	{
	if ((options_list.planet_orbit_longyear_defined == false) && (options_list.planet_orbit_binary_defined == false))
		{
		id_star = 0; id_planet = 1;
		return 1; // central star and planet
		}
	else if ((options_list.planet_orbit_longyear_defined == true) && (options_list.planet_orbit_binary_defined == false))
		{
		id_star = 0; id_planet = 1; id_companion = 2;
		return 2; // star and planet orbiting a companion
		}
	else if ((options_list.planet_orbit_longyear_defined == false) && (options_list.planet_orbit_binary_defined == true))
		{
		id_star = 0; id_planet = 1; id_binary = 2;
		return 3; // planet orbiting a binary
		}
	else if ((options_list.planet_orbit_longyear_defined == true) && (options_list.planet_orbit_binary_defined == true))
		{
		return -1; // planet orbiting a binary that is in turn orbiting a companion (will be 5)
		}
	}
else if (options_list.moon_orbit_defined == true)
	{
	if ((options_list.planet_orbit_longyear_defined == false) && (options_list.planet_orbit_binary_defined == false))
		{
		id_star = 0; id_planet = 1; id_moon = 2;
		return 4; // central star, planet and moon
		}
	else if ((options_list.planet_orbit_longyear_defined == true) && (options_list.planet_orbit_binary_defined == false))
		{
		id_star = 0; id_planet = 1; id_companion = 2; id_moon = 3;

		if ((options_list.plot_orbit_body == "moon")||(options_list.plot_orbit_body == options_list.moon_name)) {options_list.plot_orbit_body_index = 3;}
		if ((options_list.plot_orbit_reference == "moon")||(options_list.plot_orbit_reference == options_list.moon_name)) {options_list.plot_orbit_ref_index = 3;}
		if ((options_list.plot_body == "moon")||(options_list.plot_body == options_list.moon_name)) {options_list.plot_body_index = 3;}

		return 6; // star, planet and moon orbiting a companion 
		}
	else if ((options_list.planet_orbit_longyear_defined == false) && (options_list.planet_orbit_binary_defined == true))
		{
		id_star = 0; id_planet = 1; id_binary = 2; id_moon = 3;

		if ((options_list.plot_orbit_body == "moon")||(options_list.plot_orbit_body == options_list.moon_name)) {options_list.plot_orbit_body_index = 3;}
		if ((options_list.plot_orbit_reference == "moon")||(options_list.plot_orbit_reference == options_list.moon_name)) {options_list.plot_orbit_ref_index = 3;}
		if ((options_list.plot_body == "moon")||(options_list.plot_body == options_list.moon_name)) {options_list.plot_body_index = 3;}

		return 7; // planet and moon orbiting a binary 
		}
	else if ((options_list.planet_orbit_longyear_defined == true) && (options_list.planet_orbit_binary_defined == true))
		{
		return -1; // planet and moon orbiting a binary that is in turn orbiting a companion (will be 8)
		}

	}

return -1;
	
}



double diurnal_sun_angle (double t, double lat, double lon)
{

double day_correction_angle;

if (options_list.binary_defined)
	{day_correction_angle = solver.get_day_correction_angle(id_star, id_planet);}
else
	{day_correction_angle = 0.0;}

day_correction_angle /= (2.0 * pi);


planet.set_day_fraction(t - solver.get_year_fraction(id_star,id_planet) + day_correction_angle );
//planet.set_day_fraction(t + day_correction_angle);
return rad_to_deg * asin(planet.solar_elevation_angle_sin(lat * deg_to_rad, lon * deg_to_rad));
} 

double diurnal_companion_angle (double t, double lat, double lon)
{
//planet.set_longday_fraction(t - solver.get_year_fraction(2,1));
planet.set_longday_fraction(t );
return rad_to_deg * asin(planet.companion_elevation_angle_sin(lat * deg_to_rad, lon * deg_to_rad));
} 

double diurnal_binary_angle (double t, double lat, double lon)
{
planet.set_day_fraction(t -solver.get_year_fraction_CoG(id_planet) + solver.get_day_correction_angle(id_binary, id_planet) );
return rad_to_deg * asin(planet.binary_elevation_angle_sin(lat * deg_to_rad, lon * deg_to_rad));
} 

double year_averaged_irradiation (double lat)
{
int i,j;
int npoints_year = 300;
int npoints_day = 100;

double res, t_step, t_step_day;

// reset the orbit
solver.set_x(id_planet, options_list.orbit_state_x);
solver.set_y(id_planet, options_list.orbit_state_y);
solver.set_vx(id_planet, options_list.orbit_state_vx);
solver.set_vy(id_planet, options_list.orbit_state_vy);
solver.set_timestep(1000.0);

res = 0.0;

t_step = planet.get_period() / (npoints_year - 1);
t_step_day = planet.get_rotation_period() / (npoints_day - 1);


for (i =0; i < npoints_year; i++)
	{

	double yearfrac = solver.get_true_anomaly(id_star,id_planet)/ (2.0 * pi);
	planet.set_year_fraction(yearfrac);
	double sc_current = central_sun.get_luminosity() / (4.0 * pi * pow(solver.get_distance(id_star,id_planet), 2.0));
	planet.set_solar_constant(sc_current);

	for (j = 0; j < npoints_day; j++)
		{
		double dayfrac = (double) j/ (double) npoints_day;
		res+= diurnal_sun_angle (dayfrac, lat, 0.0) * t_step_day * sc_current;
		}	
	solver.evolve_for(t_step);
	}
return res;
}


void process_atmospheric_heating(int resolution)
{
int n_lambda, n_alt, id_obj;
double lambda_min, lambda_max, lambda_res, lambda;
double dist_factor, dist_factor_companion, dist_factor_binary, intensity;


if (options_list.moon_thermal_defined)
	{
	id_obj = id_moon;
	}
else
	{
	id_obj = id_planet;
	}

n_alt = 1000;


lambda_min = 1.9e-7;
lambda_max = 3.0e-7;

if (options_list.plot_int_range_set)
	{
	lambda_min = options_list.plot_int_range_min;
	lambda_max = options_list.plot_int_range_max;
	}

n_lambda = 100;
lambda_res = (lambda_max - lambda_min)/(n_lambda-1);


dist_factor = pow(central_sun.get_radius() / solver.get_distance(id_star,id_obj),2.0);
if (options_list.binary_defined) {dist_factor_binary = pow(binary.get_radius() / solver.get_distance(id_binary,id_obj),2.0);}
if (options_list.companion_defined) {dist_factor_companion = pow(companion.get_radius() / solver.get_distance(id_companion,id_obj),2.0);}

for (int i=0; i< n_lambda;i++)
	{
	lambda = lambda_min + i * lambda_res;

	if (options_list.plot_star_index == 0)
			{
			intensity = dist_factor * lambda_res * central_sun.spectrum(lambda);
			if (options_list.binary_defined) { intensity += dist_factor_binary * lambda_res * binary.spectrum(lambda);}
			if (options_list.companion_defined) { intensity += dist_factor_companion * lambda_res * companion.spectrum(lambda);}
			}
	else if (options_list.plot_star_index == 1)
			{intensity = dist_factor * lambda_res * central_sun.spectrum(lambda);}
	else if ((options_list.plot_star_index == 2) && (options_list.companion_defined))
			{intensity = dist_factor_companion * lambda_res * companion.spectrum(lambda);}
		else if ((options_list.plot_star_index == 3) && (options_list.binary_defined))
			{intensity = dist_factor_binary * lambda_res * binary.spectrum(lambda);}
		else
			{intensity = 0.0;}

	if (options_list.moon_thermal_defined)
		{
		moon.compute_atmosphere_uv_heating(lambda, intensity , n_alt);
		}
	else
		{
		planet.compute_atmosphere_uv_heating(lambda, intensity , n_alt);
		}

	//spectral_sum+= central_sun.spectrum(lambda) * lambda_res * dist_factor;
	}

//cout << "Integrated spectrum: " << spectral_sum << endl;

}

void process_o3_production(int resolution)
{
int n_lambda, n_alt, alt_res, id_obj;
double lambda_min, lambda_max, lambda_res, lambda;
double dist_factor, dist_factor_companion, dist_factor_binary, intensity, production_sum;
double alt, scale_height;

//double spectral_sum = 0.0;

n_alt = 1000;

if (options_list.moon_thermal_defined)
	{
	moon.init_atmosphere_uv_heating(n_alt);
	id_obj = id_moon;
	scale_height = moon.atmosphere.get_scale_height();
	}
else
	{
	planet.init_atmosphere_uv_heating(n_alt);
	id_obj = id_planet;
	scale_height = planet.atmosphere.get_scale_height();
	}



lambda_min = 2.0e-7;
lambda_max = 3.5e-7;

if (options_list.plot_int_range_set)
	{
	lambda_min = options_list.plot_int_range_min;
	lambda_max = options_list.plot_int_range_max;
	}

n_lambda = 100;
lambda_res = (lambda_max - lambda_min)/(n_lambda-1);


dist_factor = pow(central_sun.get_radius() / solver.get_distance(id_star,id_obj),2.0);
if (options_list.binary_defined) {dist_factor_binary = pow(binary.get_radius() / solver.get_distance(id_binary,id_obj),2.0);}
if (options_list.companion_defined) {dist_factor_companion = pow(companion.get_radius() / solver.get_distance(id_companion,id_obj),2.0);}

for (int i=0; i< n_lambda;i++)
	{
	lambda = lambda_min + i * lambda_res;
	if (options_list.plot_star_index == 0)
		{
		intensity = dist_factor * lambda_res * central_sun.spectrum(lambda);
		if (options_list.binary_defined)
			{
			intensity += dist_factor_binary * lambda_res * binary.spectrum(lambda);
			}
		if (options_list.companion_defined)
			{
			intensity += dist_factor_companion * lambda_res * companion.spectrum(lambda);
			}
		}
	else if (options_list.plot_star_index == 1)
		{
		intensity = dist_factor * lambda_res * central_sun.spectrum(lambda);
		}
	else if ((options_list.plot_star_index == 2) && (options_list.companion_defined))
		{
		intensity = dist_factor_companion * lambda_res * companion.spectrum(lambda);
		}
	else if ((options_list.plot_star_index == 3) && (options_list.binary_defined))
		{
		intensity = dist_factor_binary * lambda_res * binary.spectrum(lambda);
		}
	else
		{
		intensity = 0.0;
		}
	
	if (options_list.moon_thermal_defined)
		{
		moon.compute_atmosphere_o3_profile(lambda, intensity , n_alt);
		}
	else
		{
		planet.compute_atmosphere_o3_profile(lambda, intensity , n_alt);
		}

	}

production_sum = 0.0;
alt_res = 10.0 * scale_height / n_alt;

// Ozone for Earth air column  is 6.24E-3 kg, so we normalize energy input for Earth conditions

for (int i=0; i< n_alt; i++)
	{
	alt = i * alt_res;
	if (options_list.moon_thermal_defined)
		{
		production_sum += moon.get_atmosphere_o3_profile(alt) * alt_res ;
		}
	else
		{
		production_sum += planet.get_atmosphere_o3_profile(alt) * alt_res ;
		}
	}
cout << "Produced O3: " << production_sum << " kg across air column." << endl;

}



double plotfunc (double x, int var_x, int var_y)
{
double spectrum, dist_factor, dist_factor_binary, dist_factor_companion;
double transmission;

if (options_list.moon_orbit_defined)
	{
	dist_factor = pow(central_sun.get_radius() / solver.get_distance(id_star,id_moon),2.0);
	if (options_list.binary_defined) {dist_factor_binary = pow(binary.get_radius() / solver.get_distance(id_binary,id_moon),2.0);}
	if (options_list.companion_defined) {dist_factor_companion = pow(companion.get_radius() / solver.get_distance(id_companion,id_moon),2.0);}
	}
else if (options_list.planet_orbit_defined) 
	{
	dist_factor = pow(central_sun.get_radius() / solver.get_distance(id_star,id_planet),2.0);
	if (options_list.binary_defined) {dist_factor_binary = pow(binary.get_radius() / solver.get_distance(id_binary,id_planet),2.0);}
	if (options_list.companion_defined) {dist_factor_companion = pow(companion.get_radius() / solver.get_distance(id_companion,id_planet),2.0);}
	}
else
	{dist_factor = 1.0;}



if (options_list.plot_body == options_list.star_name) {options_list.plot_body_index = 0;}
else if (options_list.plot_body == options_list.companion_name) {options_list.plot_body_index = 2;}
else if (options_list.plot_body == options_list.binary_name) {options_list.plot_body_index = 2;}

if (var_x == 1)
	{
	if (var_y == 1) 
		{
		if (options_list.plot_star_index == 0)
			{
			spectrum =  dist_factor * central_sun.spectrum(x);
			if (options_list.companion_defined) {spectrum += dist_factor_companion * companion.spectrum(x);}
			if (options_list.binary_defined) {spectrum += dist_factor_binary * binary.spectrum(x);}
			return spectrum;
			}
		else if (options_list.plot_star_index == 1)
			{return dist_factor * central_sun.spectrum(x);}
		else if ((options_list.plot_star_index == 2) && (options_list.companion_defined))
			{return dist_factor_companion * companion.spectrum(x);}
		else if ((options_list.plot_star_index == 3) && (options_list.binary_defined))
			{return dist_factor_binary * binary.spectrum(x);}
		}
	else if (var_y == 4) 
		{
		if (options_list.moon_atmosphere_defined) {return moon.atmosphere.get_transmission(x);}
		else {return planet.atmosphere.get_transmission(x);}
		}
	else if (var_y == 5) 
		{
		if (options_list.moon_atmosphere_defined){return moon.spectrum(x);}
		else {return planet.spectrum(x);}
		}
	else if (var_y == 6) 
		{
		if (options_list.moon_atmosphere_defined){return moon.spectrum_atm_filtered(x);}
		else {return planet.spectrum_atm_filtered(x);}
		}
	else if (var_y == 7) 
		{
		if (options_list.moon_atmosphere_defined){return moon.atmosphere.get_uv_transmission(x);}
		else {return planet.atmosphere.get_uv_transmission(x);}
		}
	else if (var_y == 8) 
		{


		if (options_list.moon_atmosphere_defined)
			{
			if (options_list.par_1 > 0.0)
				{transmission = moon.atmosphere.get_uv_transmission(x, options_list.par_1) * moon.atmosphere.get_transmission(x, options_list.par_1);}
			else	
				{transmission = moon.atmosphere.get_uv_transmission(x) * moon.atmosphere.get_transmission(x);}
			}
		else
			{
			if (options_list.par_1 > 0.0)
				{transmission = planet.atmosphere.get_uv_transmission(x, options_list.par_1) * planet.atmosphere.get_transmission(x, options_list.par_1);}
			else	
				{transmission = planet.atmosphere.get_uv_transmission(x) * planet.atmosphere.get_transmission(x);}
			}
		if (options_list.plot_star_index == 0)
			{
			spectrum =  dist_factor * central_sun.spectrum(x);
			if (options_list.companion_defined) {spectrum += dist_factor_companion * companion.spectrum(x);}
			if (options_list.binary_defined) {spectrum += dist_factor_binary * binary.spectrum(x);}
			return spectrum * transmission;
			}
		else if (options_list.plot_star_index == 1)
			{return dist_factor * central_sun.spectrum(x) * transmission;}
		else if ((options_list.plot_star_index == 2) && (options_list.companion_defined))
			{return dist_factor_companion * companion.spectrum(x) * transmission;}
		else if ((options_list.plot_star_index == 3) && (options_list.binary_defined))
			{return dist_factor_binary * binary.spectrum(x) * transmission;}
		}
	}
else if ((var_x == 2) || (var_x == 4) || (var_x == 5))
	{
	if (var_x == 4)
		{x = (x * 86400.0)/planet.get_rotation_period();}
	else if (var_x == 5) 
		{x = (x * 3600.0)/planet.get_rotation_period();}

	if (var_y == 2)
		{
		if (options_list.plot_body_index == 0)
			{
			if (options_list.companion_defined)
				{
				return diurnal_sun_angle(x + planet.get_longyear_fraction(), options_list.par_1, options_list.par_2);
				}
			else
				{
				return diurnal_sun_angle(x, options_list.par_1, options_list.par_2);
				}

			}
		else if (options_list.plot_body_index == 2)
			{
			if (options_list.companion_defined)
				{return diurnal_companion_angle(x, options_list.par_1, options_list.par_2);}
			else if (options_list.binary_defined)
				{return diurnal_binary_angle(x, options_list.par_1, options_list.par_2);}
			}
	
		}
	}
else if (var_x == 3)
	{
	if (var_y == 3) {return year_averaged_irradiation(x);}

	}
else if (var_x == 6)
	{
	if (var_y == 9)
		{
		if (options_list.moon_atmosphere_defined) {return moon.get_atmosphere_uv_heating(x);}
		else {return planet.get_atmosphere_uv_heating(x);}
		}
	else if (var_y == 10)
		{
		if (options_list.moon_atmosphere_defined){return moon.get_atmosphere_uv_heating_T(x);}
		else {return planet.get_atmosphere_uv_heating_T(x);}
		}
	else if (var_y == 11)
		{
		if (options_list.moon_atmosphere_defined) {return moon.get_atmosphere_o3_profile(x);}
		else {return planet.get_atmosphere_o3_profile(x);}
		}
	else if (var_y == 12)
		{
		if (options_list.moon_geology_defined) {return moon.get_T_at_depth(x);}
		else {return planet.get_T_at_depth(x);}
		}
	else if (var_y == 14)
		{
		if (options_list.moon_geology_defined) {return moon.get_p_at_depth(x);}
		else {return planet.get_p_at_depth(x);}
		}
	}
return 0.0;

}


double recordfunc_global(int index)
{
if (index == 1)
	{
	if (options_list.record_body_index == 0)
		{return planet.get_average_temperature();}
	else
		{return moon.get_average_temperature();}
	}
else if (index == 6)
	{
	if (options_list.record_body_index == 0)
		{return planet.get_average_cloudcover();}
	else
		{return moon.get_average_cloudcover();}
	}
else if (index == 13)
	{
	if (options_list.record_body_index == 0)
		{return planet.get_average_albedo();}
	else
		{return moon.get_average_albedo();}
	}
else if (index == 14)
	{
	if (options_list.record_body_index == 0)
		{return planet.get_year_fraction();}
	else
		{return moon.get_year_fraction();}
	}
else if (index == 15)
	{
	if (options_list.record_body_index == 0)
		{return planet.get_longyear_fraction();}
	else
		{return moon.get_longyear_fraction();}
	}
else if (index == 16)
	{
	if (options_list.record_body_index == 0)
		{return planet.get_solar_declination();}
	else
		{return moon.get_solar_declination();}
	}
else if (index == 17)
	{
	if (options_list.record_body_index == 0)
		{return planet.get_solar_declination_companion();}
	else
		{return moon.get_solar_declination_companion();}
	}
else if (index == 18)
	{
	if (options_list.record_body_index == 0)
		{return planet.get_solar_declination_binary();}
	else
		{return moon.get_solar_declination_binary();}
	}

cout << "Index for global recording not implemented." << endl;
exit(0);

}


double recordfunc(int i, int j, int index)
{


if (index == 1)
	{
	if (options_list.record_body_index == 0)
		{return planet.get_element_temperature(i,j);}
	else
		{return moon.get_element_temperature(i,j);}
	}
else if (index == 2)
	{
	if (options_list.record_body_index == 0)
		{return rad_to_deg * planet.get_element_solar_angle(i,j,options_list.record_ref_index); }
	else
		{return rad_to_deg * moon.get_element_solar_angle(i,j,options_list.record_ref_index); }
	}
else if (index == 3)
	{
	if (options_list.record_body_index == 0)
		{return planet.get_element_emitted_IR_flux(i,j);}
	else
		{return moon.get_element_emitted_IR_flux(i,j);}
	}
else if (index == 4)
	{
	if (options_list.record_body_index == 0)
		{return planet.get_element_reflected_flux(i,j);}
	else
		{return moon.get_element_reflected_flux(i,j);}
	}
else if (index == 5)
	{
	if (options_list.record_body_index == 0)
		{return planet.get_element_absorbed_flux(i,j);}
	else
		{return moon.get_element_absorbed_flux(i,j);}
	}
else if (index == 6)
	{
	if (options_list.record_body_index == 0)
		{return planet.get_element_cloud_cover(i,j);}
	else
		{return moon.get_element_cloud_cover(i,j);}
	}
else if (index == 7)
	{
	if (options_list.record_body_index == 0)
		{return planet.get_element_low_cloud_cover(i,j);}
	else
		{return moon.get_element_low_cloud_cover(i,j);}
	}
else if (index == 8)
	{
	if (options_list.record_body_index == 0)
		{return planet.get_element_mid_cloud_cover(i,j);}
	else
		{return moon.get_element_mid_cloud_cover(i,j);}
	}
else if (index == 9)
	{
	if (options_list.record_body_index == 0)
		{return planet.get_element_high_cloud_cover(i,j);}
	else
		{return moon.get_element_high_cloud_cover(i,j);}
	}
else if (index == 10)
	{
	if (options_list.record_body_index == 0)
		{return planet.get_element_precipitation(i,j);}
	else
		{return moon.get_element_precipitation(i,j);}
	}
else if (index == 11)
	{
	if (options_list.record_body_index == 0)
		{return planet.get_element_snow_thickness(i,j);}
	else
		{return moon.get_element_snow_thickness(i,j);}
	}
else if (index == 12)
	{
	if (options_list.record_body_index == 0)
		{return planet.get_element_ice_thickness(i,j);}
	else
		{return moon.get_element_ice_thickness(i,j);}
	}
else if (index == 19)
	{
	if (options_list.record_body_index == 0)
		{return planet.get_element_convective_energy(i,j);}
	else
		{return moon.get_element_convective_energy(i,j);}
	}


cout << "Index for local recording not implemented." << endl;
exit(0);

}



void plot () 
{
double x, x_step;

int n_points = 100;
string name_result = "result.dat";


if (options_list.plot_filename_defined) {name_result = options_list.plot_filename;}
if (options_list.n_points > 0) {n_points = options_list.n_points;}

if ((options_list.var_y == 9) || (options_list.var_y == 10) || (options_list.var_y == 11))
	{
	cout << "Heating being processed..." << endl;
	process_o3_production(n_points);
	process_atmospheric_heating (n_points);
	}

x_step = (options_list.x_max - options_list.x_min) / (n_points - 1);




ofstream resfile (name_result.c_str());
  if (resfile.is_open())
  {
  
  for (int i = 0; i< n_points; i=i+1)

	{
	x = options_list.x_min + i * x_step;
	resfile << x << " " << plotfunc(x * options_list.plot_x_range_factor, options_list.var_x, options_list.var_y) << endl;

	}


    resfile.close();
    cout << " " << endl;
    cout << "2d dataplot written to file " << name_result << "." << endl;
    

  }
 else cout << "Unable to open file " << name_result << " !" << endl;

planet.atmosphere.consider_only_component("");

}


void plot_orbit ()
{

double t_step;

int n_points = 100;
string name_result = "orbit.dat";

if (options_list.plot_orbit_filename_defined) {name_result = options_list.plot_orbit_filename;}
if (options_list.n_points_orbit > 0) {n_points = options_list.n_points_orbit;}

double timeframe = planet.get_period();

if (options_list.plot_orbit_timeframe > 0)
	{timeframe = options_list.plot_orbit_timeframe;}


t_step = timeframe / (n_points - 1);



if (options_list.orbital_sim_timestep > 0.0) {solver.set_timestep(options_list.orbital_sim_timestep);}

if (t_step < solver.get_timestep())
	{
	solver.set_timestep(t_step);
	}


ofstream resfile (name_result.c_str());
  if (resfile.is_open())
  {
  
  for (int i = 0; i< n_points; i=i+1)

	{
	//t = i * t_step;

	if (options_list.plot_orbit_3d)
		{
		if (options_list.plot_orbit_relative)
			{
			resfile << solver.get_x(options_list.plot_orbit_body_index) - solver.get_x(options_list.plot_orbit_ref_index) << " " << solver.get_y(options_list.plot_orbit_body_index) - solver.get_y(options_list.plot_orbit_ref_index) << " " << solver.get_z(options_list.plot_orbit_body_index) - solver.get_z(options_list.plot_orbit_ref_index) << endl;
			}
		else
			{
			resfile << solver.get_x(options_list.plot_orbit_body_index) << " " << solver.get_y(options_list.plot_orbit_body_index) << " " << solver.get_z(options_list.plot_orbit_body_index)  << endl;

			}
		}
	else
		{
		if (options_list.plot_orbit_relative)
			{
			resfile << solver.get_x(options_list.plot_orbit_body_index) - solver.get_x(options_list.plot_orbit_ref_index) << " " << solver.get_y(options_list.plot_orbit_body_index) - solver.get_y(options_list.plot_orbit_ref_index)  << endl;
			}
		else
			{
			resfile << solver.get_x(options_list.plot_orbit_body_index) << " " << solver.get_y(options_list.plot_orbit_body_index)  << endl;
			}
		}

	solver.evolve_for(t_step);
	}


    resfile.close();

   cout << "Orbital plot written to " << name_result << " !" << endl;

  }
 else cout << "Unable to open file " << name_result << " !" << endl;

}

void plot_surface ()
{

if (options_list.z_min_surface < options_list.z_max_surface )
	{
	//planet.plot_surface_data_mtv (options_list.var_z_surface, options_list.plot_surface_filename, options_list.z_min_surface, options_list.z_max_surface);


	if (options_list.plot_surface_body_index == 2)
		{
		moon.plot_surface_data_gnu (options_list.var_z_surface, options_list.plot_surface_filename);
		}
	else
		{
		planet.plot_surface_data_gnu (options_list.var_z_surface, options_list.plot_surface_filename);
		}
	}
else
	{
	if (options_list.plot_surface_body_index == 2)
		{
		moon.plot_surface_data_gnu (options_list.var_z_surface, options_list.plot_surface_filename);
		}
	else
		{
		//planet.plot_surface_data_mtv (options_list.var_z_surface, options_list.plot_surface_filename);
		planet.plot_surface_data_gnu (options_list.var_z_surface, options_list.plot_surface_filename);
		}
	}
}


void output_eclipse_data (string name_star, string name_ec, double time)
{

if (options_list.use_eclipse_verbose == false) {return;}

if  ((time > 86400.0) && (time < 31536000.0))
	{
	cout << "Eclipse of " << name_star << " by " << name_ec << " on day " << time / 86400.0 << endl;
	}
else if (time > 31536000.0)
	{
	cout << "Eclipse of " << name_star << " by " << name_ec << " in year " << time / 31536000.0 << endl;
	}
else
	{
	cout << "Eclipse of " << name_star << " by " << name_ec << " in second " << time  << endl;
	}

}

void output_eclipse_end(double time)
{

if (options_list.use_eclipse_verbose == false) {return;}

if  ((time > 86400.0) && (time < 31536000.0))
	{
	cout << "Eclipse ends on day " << time / 86400.0 << endl;
	}
else if (time > 31536000.0)
	{
	cout << "Eclipse ends in year  " << time / 31536000.0 << endl;
	}
else
	{
	cout << "Eclipse ends in second " << time  << endl;
	}


}

int find_body_by_name(std::string name)
{
int id = -1;

if ((name == planet.get_designation()) || (name == "planet")) {id = id_planet;}
else if ((name == moon.get_designation()) || (name == "moon")) {id = id_moon;}
else if ((name == central_sun.get_designation()) || (name == "star")) {id = id_star;}
else if ((name == binary.get_designation()) || (name == "binary")) {id = id_binary;}
else if ((name == companion.get_designation()) || (name == "companion")) {id = id_companion;}
else
	{
	for (int i=0; i< options_list.num_bodies; i++)
		{
		if (name == body_array[i].get_designation()) {id = body_array[i].get_ref_index(); break;}
		}
	}
if (id == -1) {cout << "Orbital summary report warning: body " << name << " could not be found in list, exiting..." << endl;}
return id;
}


void orbit_summary_report()
{
int id, id_ref;

if (options_list.report_orbit_reference != "")
	{
	id_ref = find_body_by_name(options_list.report_orbit_reference);
	if (id_ref == -1) {exit(0);}
	}
else
	{
	id_ref = find_body_by_name("star");
	}

cout << endl;

cout << left << setw(12) << "Body" << right;

for (int i=0; i< options_list.report_orbit_num_var; i++)
	{
	cout << setw(15) << options_list.report_orbit_variable[i];
	}
cout << endl;

for (int i=0; i< options_list.report_orbit_num_src; i++)
	{
	id = find_body_by_name(options_list.report_orbit_source[i]);
	if (id == -1) {exit(0);}
	
	cout << left << setw(12) << options_list.report_orbit_source[i] + ":"  << right;
	
	for (int j=0; j< options_list.report_orbit_num_var; j++)
		{
		if (options_list.report_orbit_variable[j] == "distance")
			{
			cout << setw(8) << setprecision(4) <<  solver.get_distance(id, id_ref)/1e9 << setw(6) << " [Mkm] ";
			}
		else if  (options_list.report_orbit_variable[j] == "rel_velocity")
			{
			cout << setw(8) << setprecision(4) << solver.get_rel_velocity(id, id_ref)/1e3 << setw(6) << " [km/s] ";
			}
		else if (options_list.report_orbit_variable[j] == "semimajor_au")
			{
			cout << setw(8) << setprecision(4) << solver.get_semimajor(id, id_ref)/AU << setw(6) << " [au] ";
			}
		else if (options_list.report_orbit_variable[j] == "semimajor")
			{
			cout << setw(8) << setprecision(4) << solver.get_semimajor(id, id_ref)/1e9 << setw(6) << " [Mkm] ";
			}
		}
	cout << endl;
	}

}


int main (int argc, char *argv[] ) 
{
string config_file;

bool eclipse_flag_star = false;
bool eclipse_flag_binary = false;
bool eclipse_flag_companion = false;
bool eclipse_flag_pstar = false;
bool eclipse_flag_pbinary = false;
bool eclipse_flag_pcompanion = false;

double eclipse_factor_star, eclipse_factor_companion, eclipse_factor_binary, eclipse_factor_pstar, eclipse_factor_pbinary, eclipse_factor_pcompanion;

ofstream recordfile, scatterfile;
std::string name_record, name_scatter;

int system_type;

  config_file = commandline_parser (argc, argv);
  file_parser(config_file);

  srand(options_list.random_seed);

  system_type = system_type_mapper();

  if (system_type == -1)
	{
	cout << "The specified star system type is not implemented and cannot be simulated." << endl;
	}
  else if (system_type == -2)
	{
	cout << "Only star definition found, no planetary system will be initialized." << endl;
	}


  if (options_list.relief_defined)
	{
	relief.evolve();
	}


  if (options_list.record_defined)
	{
	if (options_list.record_filename_defined)
		{name_record = options_list.record_filename;}
	else
		{name_record = "record.dat";}

	recordfile.open(name_record.c_str());
	if (recordfile.is_open())
	  	{
		if (options_list.record_global)
			{
			cout << "Global record initialized." << endl;
			}
		else
			{
			cout << "Record initialized for element at lat " << rad_to_deg * planet.get_element_latitude(options_list.record_index_x, options_list.record_index_y) << " and longitude " << rad_to_deg * planet.get_element_longitude(options_list.record_index_x, options_list.record_index_y) << endl;
			}
		cout << "Recording to file " << name_record << "..." << endl;

		}
	 else 
		{cout << "Unable to open file " << name_record << "!" << endl;}
	}

  if (options_list.scatter_plot_defined)
	{
	if (options_list.scatter_filename_defined)
		{name_scatter = options_list.scatter_filename;}
	else
		{name_scatter = "scatter_plot.dat";}

	scatterfile.open(name_scatter.c_str());
	if (scatterfile.is_open())
		{cout << "Scatter plot to file " << name_scatter << "..." << endl;}
	 else 
		{cout << "Unable to open file " << name_scatter << "!" << endl;}
	}


  if (options_list.planet_orbit_defined)
	{
	if (system_type == 2)
		if (planet.has_lagrange_orbit())
			{solver.initialize(central_sun, companion, planet, 3);}
		else 	
			{solver.initialize(central_sun, companion, planet, 1);}
	else if (system_type == 3)
		{solver.initialize(central_sun, binary, planet, 2);}
	else if (system_type == 1)
		{solver.initialize(central_sun, planet);}
	else if (system_type == 4)
		{solver.initialize(central_sun, planet, moon);}
	else if (system_type == 6)
		{solver.initialize(central_sun, companion, planet, moon, 1);}
	else if (system_type == 7)
		{solver.initialize(central_sun, binary, planet, moon, 2);}
	solver.set_timestep(planet.get_period()/1e6);
	options_list.orbit_state_x = solver.get_x(id_planet);
	options_list.orbit_state_y = solver.get_y(id_planet);
	options_list.orbit_state_vx = solver.get_vx(id_planet);
	options_list.orbit_state_vy = solver.get_vy(id_planet);

	if (options_list.bodies_defined)
		{
		for (int i=0; i< options_list.num_bodies; i++)
			{
			int index;

			if ((options_list.body_attach_to[i] == planet.get_designation()) || (options_list.body_attach_to[i] == "planet"))
				{
				index = solver.add_body(body_array[i], id_planet);
				}
			else if ((options_list.body_attach_to[i] == central_sun.get_designation()) || (options_list.body_attach_to[i] == "star"))
				{
				index = solver.add_body(body_array[i], id_star);
				}
			else if ((options_list.body_attach_to[i] == moon.get_designation()) || (options_list.body_attach_to[i] == "moon"))
				{
				index = solver.add_body(body_array[i], id_moon);
				}
			else if ((options_list.body_attach_to[i] == binary.get_designation()) || (options_list.body_attach_to[i] == "binary"))
				{
				index = solver.add_body(body_array[i], id_binary);
				}
			else if ((options_list.body_attach_to[i] == companion.get_designation()) || (options_list.body_attach_to[i] == "companion"))
				{
				index = solver.add_body(body_array[i], id_companion);
				}
			else
				{
				for (int j=0; j< i; j++)
					{
					if (options_list.body_attach_to[i] == body_array[j].get_designation())
						{
						index = solver.add_body(body_array[i], body_array[j].get_ref_index());
						}
					}
				}

			body_array[i].set_ref_index(index);
			}
		assign_body_indices();
		}

	if (options_list.orbit_resume_flag)
		{
		solver.resume(options_list.orbit_filename);
		}


	}


	if (options_list.roche_check)
		{
		cout << endl;
		cout << "Consistency checks for Roche limit: " << endl;
    		 check_roche(planet.get_radius(),  options_list.star_mass * M_solar, planet.get_mass(), planet.get_periapsis(), options_list.star_name, options_list.planet_name);

		if (options_list.binary_defined)
			{
 			check_roche(planet.get_radius(),  options_list.binary_mass * M_solar, planet.get_mass(), planet.get_periapsis(), options_list.binary_name, options_list.planet_name);

			if (options_list.star_mass > options_list.binary_mass)
				{
				check_roche(binary.get_radius(),  central_sun.get_mass(), binary.get_mass(), options_list.planet_semimajor_binary, options_list.star_name, options_list.binary_name);
				}
			else
				{
				check_roche(central_sun.get_radius(),  binary.get_mass(), central_sun.get_mass(), options_list.planet_semimajor_binary, options_list.binary_name, options_list.star_name);
				}
			}

		if (options_list.moon_orbit_defined)
			{
			check_roche(moon.get_radius(),  planet.get_mass(), moon.get_mass(), moon.get_periapsis(), options_list.planet_name, options_list.moon_name);
			}

		cout << endl;
		}

	if (options_list.hill_sphere_check)
		{
		cout << endl;
		cout << "Consistency checks for 3-body orbital stability: " << endl;
		if (options_list.moon_orbit_defined)
			{
			check_hill(options_list.planet_semimajor, central_sun.get_mass(), planet.get_mass(), options_list.planet_eccentricity, options_list.moon_semimajor, options_list.planet_name, options_list.moon_name);
		cout << endl;
			}
		if (options_list.companion_defined)
			{
			check_hill(options_list.planet_semimajor_longyear, companion.get_mass(), central_sun.get_mass(), options_list.planet_eccentricity_longyear, options_list.planet_semimajor, options_list.star_name, options_list.planet_name);
			}
		}

	if (options_list.tidal_locking_check)
		{
		cout << endl;
		cout << "Consistency checks for tidal locking of rotation: " << endl;
		check_locking(options_list.planet_semimajor, planet.get_radius(), planet.get_mass(), central_sun.get_mass(), options_list.star_name, options_list.planet_name);
		
		if (options_list.moon_orbit_defined)
			{
			check_locking(options_list.moon_semimajor, moon.get_radius(), moon.get_mass(), planet.get_mass(), options_list.planet_name, options_list.moon_name);
			}

		cout << endl;
		}

	if (options_list.atmosphere_phase_check)
		{
		if (options_list.planet_atmosphere_defined)
			{
			planet.atmosphere.check_phases();
			}
		else if (options_list.moon_atmosphere_defined)
			{
			moon.atmosphere.check_phases();
			}
		else	
			{
			cout << "Consistency checks for atmosphere gas phases: No atmosphere defined." << endl;
			cout << endl;
			}
		}





  if (options_list.planet_orbit_defined && options_list.orbit_evolve)
	{


	double timestep = planet.get_period()/1e6;
	if (options_list.orbital_sim_timestep > 0.0) {timestep = options_list.orbital_sim_timestep;}
	if (options_list.orbit_evolve_timestep > 0.0) {timestep = options_list.orbit_evolve_timestep;}

	double evolve_time_display = options_list.orbit_evolve_time/86400.0;
	string evolve_unit = " days with ";
	if (evolve_time_display > 365.25) {evolve_time_display /= 365.25; evolve_unit = " years with ";} 
	cout << "Evolving orbit for " << evolve_time_display << evolve_unit << timestep <<" s timestep... " << endl;
	solver.set_timestep(timestep);	
	solver.evolve_for(options_list.orbit_evolve_time);
	options_list.orbit_state_x = solver.get_x(id_planet);
	options_list.orbit_state_y = solver.get_y(id_planet);
	options_list.orbit_state_vx = solver.get_vx(id_planet);
	options_list.orbit_state_vy = solver.get_vy(id_planet);
	cout << "Finished!" << endl;

	if (options_list.orbit_save_flag)
		{
		solver.save(options_list.orbit_filename);
		}
	

	planet.set_year_fraction (solver.get_year_fraction(id_star, id_planet));
	//cout << "year fraction: " << solver.get_year_fraction(0, 1) << endl;
	planet.set_distance_to_star (solver.get_distance(id_star, id_planet), central_sun.get_luminosity());
	planet.set_day_fraction (options_list.orbit_day_fraction);
	
	cout << "Current distance to star [Mkm]:      " << solver.get_distance(id_star, id_planet)/1e9 << endl;
	cout << "True anomaly [deg]:                  " << solver.get_true_anomaly(id_star, id_planet) * rad_to_deg << endl;

	if (options_list.companion_defined)
		{

		planet.set_longyear_fraction (solver.get_year_fraction(id_companion, id_planet));
		cout << "Current distance to companion [Mkm]: " << solver.get_distance(id_companion, id_planet)/1e9 << endl;
		cout << "True anomaly [deg]:                  " << solver.get_true_anomaly(id_companion, id_planet) * rad_to_deg << endl;
		}
	else if (options_list.binary_defined)
		{
		planet.set_day_correction_angle_star(solver.get_day_correction_angle(id_star, id_planet));
		planet.set_day_correction_angle_binary(solver.get_day_correction_angle(id_binary, id_planet));

		planet.set_elevation_correction_angle_star(solver.get_elevation_correction_angle(id_star, id_planet));
		planet.set_elevation_correction_angle_binary(solver.get_elevation_correction_angle(id_binary, id_planet));
		
		cout << "Current distance to binary [Mkm]: " << solver.get_distance(id_binary, id_planet)/1e9 << endl;
		}

	cout << "Angular size of star in sky [deg]:   " << 360.0/pi * central_sun.get_radius() / solver.get_distance(id_star,id_planet) << endl;
	if (options_list.binary_defined)
		{
		cout << "Angular size of binary in sky [deg]: " << 360.0/pi * binary.get_radius() / solver.get_distance(id_binary,id_planet) << endl;
		}
	else if (options_list.companion_defined)
		{
		cout << "Angular size of companion [deg]:     " << 360.0/pi * companion.get_radius() / solver.get_distance(id_companion,id_planet) << endl;
		}

	if (options_list.report_orbit_defined)
		{
		orbit_summary_report();
		}


	if (options_list.orbital_sim_timestep > 0.0) {solver.set_timestep( options_list.orbital_sim_timestep);}
	}

  if (options_list.thermal_evolve)
	{
	cout << "Evolving thermal balance for " << options_list.thermal_evolve_time/86400.0 << " days...";
  	planet.set_surface_irradiation();
	planet.evolve_temperature(options_list.thermal_evolve_time);
	cout << " finished." << endl;
	planet.compute_average_temperature();
	}



  if (options_list.full_evolve)
	{
	//int n_steps;
	double evolution_timestep = 1000.0;
	double orbital_timestep;
	double record_timer = 0.0;
	double scatter_timer = 0.0;
	
	if (options_list.full_evolve_timestep > 0.0) {evolution_timestep = options_list.full_evolve_timestep;}
	if (evolution_timestep < options_list.orbital_sim_timestep) 
		{solver.set_timestep(evolution_timestep); orbital_timestep = evolution_timestep;}
	else 
		{solver.set_timestep(options_list.orbital_sim_timestep); orbital_timestep = options_list.orbital_sim_timestep; }

	double time = 0.0;
	double day_fraction = 0.0;
	double moon_day_fraction = 0.0;
	double year_fraction,longyear_fraction;
	double fractional_step = evolution_timestep/planet.get_sidereal_rotation_period();

	//n_steps = (int) options_list.full_evolve_time/evolution_timestep;

	if (options_list.full_evolve_timestep > 0.0) {evolution_timestep = options_list.full_evolve_timestep;}
	cout << "Evolving system for " << options_list.full_evolve_time/86400.0 << " days with " << evolution_timestep << " s timestep and " << orbital_timestep << " s orbital timestep..." << endl;
	//cout << " " << endl;




	while (time < options_list.full_evolve_time)
		{
		solver.evolve_for(evolution_timestep);


		if (options_list.binary_defined)
			{year_fraction = solver.get_year_fraction_CoG(id_planet);}
		else
			{year_fraction = solver.get_year_fraction(id_star, id_planet);}

		planet.set_year_fraction (year_fraction);
		if ((options_list.binary_defined == false) && (options_list.companion_defined == false))
			{planet.set_distance_to_star (solver.get_distance(id_star, id_planet), central_sun.get_luminosity());}
		day_fraction += fractional_step;

		if (day_fraction > 1.0) {day_fraction -= 1.0;}


		planet.set_day_fraction (day_fraction - year_fraction );
	
		eclipse_factor_star = 1.0;
		eclipse_factor_binary = 1.0;
		eclipse_factor_companion = 1.0;
		eclipse_factor_pstar = 1.0;
		eclipse_factor_pbinary = 1.0;
		eclipse_factor_pcompanion = 1.0;

		if (options_list.companion_defined)
			{

			//eclipse_factor_star = 1.0;
			//eclipse_factor_companion  = 1.0;
			if (options_list.use_eclipse_finder)
				{

				eclipse_factor_star -= solver.get_eclipse_factor (id_star,id_companion,id_planet);	
				eclipse_factor_companion -= solver.get_eclipse_factor (id_companion,id_planet,id_star);

				
				if ((eclipse_factor_star < 1.0) && (eclipse_flag_star == false))
					{
					output_eclipse_data(options_list.star_name, options_list.companion_name, time);
					eclipse_flag_star = true;
					}

				if ((eclipse_factor_star == 1.0) && (eclipse_flag_star == true))
					{
					eclipse_flag_star = false;
					output_eclipse_end(time);
					}

				if ((eclipse_factor_companion < 1.0) && (eclipse_flag_companion == false))
					{
					output_eclipse_data(options_list.companion_name, options_list.star_name, time);
					eclipse_flag_companion = true;
					}
				if ((eclipse_factor_companion == 1.0) && (eclipse_flag_companion == true))
					{
					eclipse_flag_companion = false;
					output_eclipse_end(time);
					}

				}

			longyear_fraction = solver.get_year_fraction(id_companion, id_planet);
			planet.set_longyear_fraction (longyear_fraction);

			planet.set_distance_to_companion (solver.get_distance(id_companion, id_planet), eclipse_factor_companion *  companion.get_luminosity());
			planet.set_distance_to_star (solver.get_distance(id_star, id_planet), eclipse_factor_star * central_sun.get_luminosity());


			planet.set_longday_fraction (day_fraction - longyear_fraction );
			}
		else if (options_list.binary_defined)
			{

			//eclipse_factor_star = 1.0;
			//eclipse_factor_binary  = 1.0;
			if (options_list.use_eclipse_finder)
				{
			
				eclipse_factor_star -= solver.get_eclipse_factor (id_star,id_binary,id_planet);	
				eclipse_factor_binary -= solver.get_eclipse_factor (id_binary,id_star,id_planet);

				
				if ((eclipse_factor_star < 1.0) && (eclipse_flag_star == false))
					{
					output_eclipse_data(options_list.star_name, options_list.binary_name, time);
					eclipse_flag_star = true;
					}

				if ((eclipse_factor_star == 1.0) && (eclipse_flag_star == true))
					{
					eclipse_flag_star = false;
					output_eclipse_end(time);
					}

				if ((eclipse_factor_binary < 1.0) && (eclipse_flag_binary == false))
					{
					output_eclipse_data(options_list.binary_name, options_list.star_name, time);
					eclipse_flag_binary = true;
					}
				if ((eclipse_factor_binary == 1.0) && (eclipse_flag_binary == true))
					{
					eclipse_flag_binary = false;
					output_eclipse_end(time);
					}
				

				}			

			planet.set_distance_to_binary (solver.get_distance(id_binary, id_planet), eclipse_factor_binary * binary.get_luminosity());
			planet.set_distance_to_star (solver.get_distance(id_star, id_binary), eclipse_factor_star * central_sun.get_luminosity());

			planet.set_day_correction_angle_star(solver.get_day_correction_angle(id_star, id_planet));
			planet.set_day_correction_angle_binary(solver.get_day_correction_angle(id_binary, id_planet));

			planet.set_elevation_correction_angle_star(solver.get_elevation_correction_angle(id_star, id_planet));
			planet.set_elevation_correction_angle_binary(solver.get_elevation_correction_angle(id_binary, id_planet));

			
			}



  		planet.set_surface_irradiation();
		planet.evolve_temperature_onestep(evolution_timestep);


		
		if (options_list.moon_defined)
			{


			if (options_list.use_eclipse_finder)
				{
				eclipse_factor_pstar -= solver.get_eclipse_factor (id_star,id_planet,id_moon);


				if ((eclipse_factor_pstar < 1.0) && (eclipse_flag_pstar == false))
					{
					output_eclipse_data(options_list.star_name, options_list.planet_name, time);
					eclipse_flag_pstar = true;
					}
				if ((eclipse_factor_pstar == 1.0) && (eclipse_flag_pstar == true))
					{
					eclipse_flag_pstar = false;
					output_eclipse_end(time);
					}
				}





			if (options_list.binary_defined)
				{
				if (options_list.use_eclipse_finder)
					{
					eclipse_factor_pbinary -= solver.get_eclipse_factor (id_binary,id_planet,id_moon);
	
					if ((eclipse_factor_pbinary < 1.0) && (eclipse_flag_pbinary == false))
						{
						output_eclipse_data(options_list.binary_name, options_list.planet_name, time);
						eclipse_flag_pbinary = true;
						}
					if ((eclipse_factor_pbinary == 1.0) && (eclipse_flag_pbinary == true))
						{
						eclipse_flag_pbinary = false;
						output_eclipse_end(time);
						}
					}

				moon.set_distance_to_binary (solver.get_distance(id_binary, id_moon), eclipse_factor_binary * eclipse_factor_pbinary * binary.get_luminosity());
				moon.set_day_correction_angle_star(solver.get_day_correction_angle(id_star, id_moon));
				moon.set_day_correction_angle_binary(solver.get_day_correction_angle(id_binary, id_moon));

				moon.set_elevation_correction_angle_star(solver.get_elevation_correction_angle(id_star, id_moon));
				moon.set_elevation_correction_angle_binary(solver.get_elevation_correction_angle(id_binary, id_moon));

				year_fraction = solver.get_year_fraction_CoG(id_moon);

				}
			else if (options_list.companion_defined)
				{
				if (options_list.use_eclipse_finder)
					{
					eclipse_factor_pcompanion -= solver.get_eclipse_factor (id_companion,id_planet,id_moon);

				

					if ((eclipse_factor_pcompanion < 1.0) && (eclipse_flag_pcompanion == false))
						{
						output_eclipse_data(options_list.companion_name, options_list.planet_name, time);
						eclipse_flag_pcompanion = true;
						}
					if ((eclipse_factor_pcompanion == 1.0) && (eclipse_flag_pcompanion == true))
						{
						eclipse_flag_pcompanion = false;
						output_eclipse_end(time);
						}
					}

				longyear_fraction = solver.get_year_fraction(id_companion, id_moon);
				moon.set_longyear_fraction (longyear_fraction);

				moon.set_distance_to_companion (solver.get_distance(id_companion, id_moon), eclipse_factor_companion * eclipse_factor_pcompanion *  companion.get_luminosity());

				}


			moon.set_year_fraction (year_fraction);
			moon_day_fraction += evolution_timestep/moon.get_sidereal_rotation_period();
			if (moon_day_fraction > 1.0) {moon_day_fraction -=1.0;}


			//cout << "Day fraction moon: " <<  moon_day_fraction - year_fraction << endl;
			moon.set_day_fraction (moon_day_fraction - year_fraction );
			if (options_list.companion_defined) {moon.set_longday_fraction (moon_day_fraction - longyear_fraction );}

			moon.set_distance_to_star (solver.get_distance(id_star, id_moon), eclipse_factor_star * eclipse_factor_pstar * central_sun.get_luminosity());
			moon.set_surface_irradiation();	
		
			if (options_list.use_indirect_irradiation)
				{
				// what's the planet's phase?
				double planet_dayside_fraction = solver.get_phase_area(id_star,id_planet,id_moon);
				double planetary_constant_current = planet_dayside_fraction * (planet.get_dayside_flux() + planet.get_reflected_flux());
				planetary_constant_current += (1.0 - planet_dayside_fraction) * planet.get_nightside_flux();
				planetary_constant_current *= pow(planet.get_radius()/solver.get_distance(id_planet,id_moon),2.0);
				moon.set_planetary_constant(planetary_constant_current);
			
				// which part of the moon faces the planet?
				moon.set_day_fraction (moon_day_fraction - solver.get_year_fraction(id_planet,id_moon));

				moon.add_indirect_surface_irradiation();

				// reset the sun angle for correct recording
				moon.set_day_fraction (moon_day_fraction - year_fraction );
				moon.set_sun_angle();

				}



			moon.evolve_temperature_onestep(evolution_timestep);
			}
		


		if (options_list.record_defined)
			{
			if (time > options_list.record_delay)
				{
				if (record_timer == 0.0)
					{
					
					recordfile << time/options_list.record_x_format_factor;				

					for (int i=0; i< options_list.record_num_columns;i++)
						{
						if (options_list.record_global)
							{
							recordfile << " " << recordfunc_global(options_list.record_var_y[i]);
							}
						else
							{
							recordfile << " " << recordfunc(options_list.record_index_x, options_list.record_index_y, options_list.record_var_y[i]);
							}
						
						}
					
					recordfile << endl;

					}
					
				record_timer += evolution_timestep;
				if (record_timer > options_list.record_interval) {record_timer = 0.0;}
				}
			}


		if (options_list.scatter_plot_defined)
			{
			if (time > options_list.scatter_delay)
				{
				if (scatter_timer == 0.0)
					{
					if (options_list.moon_defined)
						{
						int num_storms = moon.get_num_storms();

						for (int i=0; i< num_storms; i++)
							{
							if (options_list.scatter_filters_defined)
								{
								if ((moon.get_storm_category(i) > options_list.scatter_filter_low) && (moon.get_storm_category(i) < options_list.scatter_filter_high))
									{
									scatterfile << moon.get_storm_lon(i) << " " << moon.get_storm_lat(i) << endl;
									}
								}
							else
								{
								scatterfile << moon.get_storm_lon(i) << " " << moon.get_storm_lat(i) << endl;
								}
							}
						}
					else
						{
						int num_storms = planet.get_num_storms();

						for (int i=0; i< num_storms; i++)
							if (options_list.scatter_filters_defined)
								{
								if ((planet.get_storm_category(i) > options_list.scatter_filter_low) && (planet.get_storm_category(i) < options_list.scatter_filter_high))
									{
									scatterfile << planet.get_storm_lon(i) << " " << planet.get_storm_lat(i) << endl;
									}
								}
						else
								{
								scatterfile << planet.get_storm_lon(i) << " " << planet.get_storm_lat(i) << endl;
								}
						}
					}
				scatter_timer += evolution_timestep;
				if (scatter_timer > options_list.scatter_interval) {scatter_timer = 0.0;}
				}

			}

		time = time + evolution_timestep;
		}




	cout << "Finished." << endl;


//cout << "Coord Object" << endl;
//CoordinateObject coords;

//coords.initialize(solver.get_x(0), solver.get_y(0), solver.get_z(0), solver.get_x(2), solver.get_y(2), solver.get_z(2));
//coords.tilt_by(-10.0 * deg_to_rad, 0.0 * deg_to_rad);


	if (options_list.record_defined)
		{
		cout << "Record written to " << name_record << "." << endl;
		recordfile.close();
		}

	if (options_list.scatter_plot_defined)
		{
		cout << "Scatter plot written to " << name_scatter << "." << endl;
		scatterfile.close();
		}


	cout << "Current distance to star [Mkm]:      " << solver.get_distance(0, 1)/1e9 << endl;
	if (options_list.binary_defined)
		{
		cout << "Current distance to binary [Mkm]:    " << solver.get_distance(2, 1)/1e9 << endl;
		}
	else if (options_list.companion_defined)
		{
		cout << "Current distance to companion [Mkm]: " << solver.get_distance(2, 1)/1e9 << endl;
		}

	cout << "True anomaly [deg]:                  " << solver.get_true_anomaly(0, 1) * rad_to_deg << endl;


	cout << "Angular size of star in sky [deg]:   " << 360.0/pi * central_sun.get_radius() / solver.get_distance(0,1) << endl;
	if (options_list.binary_defined)
		{
		cout << "Angular size of binary in sky [deg]: " << 360.0/pi * binary.get_radius() / solver.get_distance(2,1) << endl;
		}
	else if (options_list.companion_defined)
		{
		cout << "Angular size of companion [deg]:     " << 360.0/pi * companion.get_radius() / solver.get_distance(2,1) << endl;
		}

        if (options_list.moon_defined)
		{
		moon.compute_average_temperature();
		moon.compute_average_cloudcover();
		if (options_list.weather_storms)
			{moon.list_storms(time);}
		}
	else
		{
		planet.compute_average_temperature();
		planet.compute_average_cloudcover();
		if (options_list.weather_storms)
			{planet.list_storms(time);}
		}

	if (options_list.orbital_sim_timestep > 0.0) {solver.set_timestep( options_list.orbital_sim_timestep);}
	}
  

  if (options_list.plot_defined) { plot(); }
 
  if (options_list.plot_orbit_defined) {plot_orbit();}

  if (options_list.plot_surface_defined) {plot_surface();}

  return 0;
}
