// weather.hxx -- class containing simple weather effect simulation routines
//
// Written by Thorsten Renk, started 2022
//
// Copyright (C) 2022  Thorsten Renk - thorsten@science-and-fiction.org 
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301

#ifndef weather_H
#define weather_H

class Cyclone {
	
	double lat;
	double lon;
	double size;
	double lifetime_s;
	double max_lifetime_s;
	double R_planet;
	double movement_speed;

	bool is_expired;


	public:
	
	Cyclone();
	Cyclone(double, double, double, double, double);
	double get_lat();
	double get_lon();
	double get_size();
	double get_age_factor();

	bool get_expired();

	void move(double);
	void init_age(double);

};


class Storm {

	int category;
	double lat;
	double lon;
	double size;
	double size_modifier;
	double size_base;
	double strength;
	double coriolis_bending;
	double gradient_direction;
	double gradient_acceleration;
	double gradient_acceleration_factor;
	double lifetime_s;
	double movement_speed_lon;
	double movement_speed_lat;
	double R_planet;

	double energy_consumption;
	double base_energy_consumption;

	bool is_decayed;
	bool is_decaying;

	int cat_max;

	public:

	Storm();
	Storm(double, double, double);

	int get_category();
	int get_category_max();

	double get_lat();
	double get_lon();
	double get_size();
	double get_energy_need();
	double get_lifetime();
	double get_age_factor();

	bool get_decayed();

	void randomize(double, double, double);
	void set_motion_parameters(double, double, double, double);
	void move(double);
	void evolve(double);
	void set_energy_input(double);
	void set_gradient_dir(double);
	void set_decaying();
	void set_active();
	


};


class Weather {

	double omega;
	double R;
	double cell_size_deg;
	double cyclone_max_size;
	double cyclone_min_size;
	double cyclone_min_lifetime;
	double cyclone_max_lifetime;
	double cyclone_spawn_probability;
	double cyclone_average_number;
	double midlevel_min;
	double midlevel_max;
	double midlevel_grid[18][36];
	double highlevel_min;
	double highlevel_max;
	double highlevel_grid[18][36];
	double convection_rn_cloud_grid[18][36];
	double convection_rn_rain_grid[18][36];
	double lat_to_m;
	double m_to_lat;
	double jetstream_pos_deg[10];
	double ridge_center_deg[10];
	double ridge_width_deg;
	double ridge_migration_deg;
	double ridge_suppression_factor;
	double polar_suppression_factor;
	double midlevel_timer;
	double precipitation_factor;
	double humidity_factor;
	double humidity_rain_factor;
	double humidity_cloud_factor;
	double humidity_high_cloud_factor;
	double humidity_mid_cloud_factor;
	double mean_atmosphere_temperature;
	double mean_windspeed;
	double grad_motion_factor;

	double medium_gas_constant;
	double medium_latent_heat;

	double simulation_timestep;

	double cloud_T_feedback;
	double mid_cloud_T_feedback;
	double high_cloud_T_feedback;
	double precipitation_T_feedback;

	int num_cells;
	int num_jetstreams;
	int num_cyclones;
	int num_storms;


	
	int circulation_model;
	int circulation_model_preset;

	bool symmetric_circulation;
	bool compute_storms;
	
	Cyclone cyclone_array[50];
	Storm storm_array[50];


	void cleanup();
	void create_cyclone(double);
	void create_midlevel_clouds();
	void create_highlevel_clouds();
	void update_rn_grids();
	void clausius_clapeyron(double);
	void set_circulation_parameters(int);
	double coriolis_radius (double, double);
	double rnd(double, double);

	// debug stuff
	int num_storms_total;
	int num_storms_cat[16];
	double storm_average_lifetime;

	public:

	//Weather(double, double );
	


	void init(double, double, int);
	void init_circulation();
	void evolve(double);
	void set_random(int);
	void set_midlevel(double, double);
	void set_highlevel(double, double);
	void set_ridge_migration_deg (double);
	void set_precipitation_factor(double);
	void set_atmosphere_temperature(double);
	void set_parameter(std::string, double);
	void set_storm_energy_usage(double, double, double, double, double, int);

	void check_storm_generation(double, double, double, double, double);

	double get_low_cloudcover(double, double, double, double);
	double get_mid_cloudcover(double, double);
	double get_high_cloudcover(double, double);
	double get_storm_cloudcover(double, double, double, double);
	double get_convection_rn_cloud(double, double);
	double get_convection_rn_precipitation(double, double);
	double get_precipitation(double, double, double, double);
	double get_ridge_deg_low(int);
	double get_ridge_deg_high(int);
	double get_ridge_suppression();
	double get_storm_energy_usage(double, double, double, double);
	double get_storm_lat(int);
	double get_storm_lon(int);

	int get_num_storms(void);
	int get_storm_category(int);
	
	void list_storms (double);
	  
};


#endif

