// weather.cxx -- class containing simple weather effect simulation routines
//
// Written by Thorsten Renk, started 2022
//
// Copyright (C) 2022  Thorsten Renk - thorsten@science-and-fiction.org 
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301

#include <iostream>
#include <fstream>
#include <string>
#include <stdlib.h>
#include <math.h>
//#include <tgmath.h>

#include "weather.hxx"
#include "constants.hxx"


using namespace std;

Cyclone::Cyclone(double lat_in, double lon_in, double size_in,  double max_lifetime, double radius)
{
lat = lat_in * deg_to_rad;
lon = lon_in * deg_to_rad;
R_planet = radius;
size = size_in;
lifetime_s = 0.0;
max_lifetime_s = max_lifetime;

movement_speed = 5.5;
is_expired = false;
}

Cyclone::Cyclone()
{
lat = 0.0;
lon = 0.0;
size = 300000.0;
lifetime_s = 0.0;
R_planet = 6700000.0;
max_lifetime_s = 864000;

movement_speed = 5.5;
is_expired = false;
}

void Cyclone::init_age(double rnd)
{
lifetime_s = rnd * max_lifetime_s;
}

double Cyclone::get_lat()
{
return lat;
}

double Cyclone::get_lon()
{
return lon;
}

double Cyclone::get_size()
{
return size;
}


double Cyclone::get_age_factor()
{
double rel_age;

rel_age = lifetime_s/max_lifetime_s;

if (rel_age < 0.2)
	{
	return (rel_age / 0.2);
	}
else if (rel_age < 0.6)
	{
	return 1.0;
	}
else	
	{
	return (1.0 - (rel_age - 0.6) /0.4);
	}
}

bool Cyclone::get_expired()
{
return is_expired;
}

void Cyclone::move(double time)
{
if (is_expired) {return;}

lon += time * movement_speed /R_planet * cos(lat * deg_to_rad);
if (lon > 2.0 * pi) {lon -= 2.0*pi;}
else if (lon < 0.0) {lon += 2.0 * pi;}
lifetime_s += time;
if (lifetime_s > max_lifetime_s) {is_expired = true;}
}



Storm::Storm()
{
category = 0;
cat_max = 0;
lat = 0.0;
lon = 0.0;
size = 200000.0;
strength = 20.0;
lifetime_s = 0.0;
R_planet = 6700000.0;
size_base = 300000.0;
size_modifier = 1.0;
movement_speed_lat = 0.0;
movement_speed_lon = 5.5;
coriolis_bending = 0.003;
gradient_direction = 0.0;
gradient_acceleration = 0.35 * 1.15e-5;
gradient_acceleration_factor = 1.0;

base_energy_consumption = 40000.0;
energy_consumption = base_energy_consumption;

is_decayed = false;
is_decaying = false;
}


Storm::Storm(double lat_in, double lon_in, double radius)
{
category = 0;
cat_max = 0;
lat = lat_in;
lon = lon_in;
size = 200000.0;
strength = 20.0;
lifetime_s = 0.0;
R_planet = radius;
size_base = 300000.0;
size_modifier = 1.0;
movement_speed_lat = 0.0;
movement_speed_lon = 5.5;
coriolis_bending = 0.003;
gradient_direction = 0.0;
gradient_acceleration = 0.35 * 1.15e-5;
gradient_acceleration_factor = 1.0;

base_energy_consumption = 40000.0;
energy_consumption = base_energy_consumption;

is_decayed = false;
is_decaying = false;
}

void Storm::randomize(double speed_mod, double bending_mod, double size_mod)
{
movement_speed_lat *= speed_mod;
coriolis_bending *= bending_mod;

size_modifier *= size_mod;
}

void Storm::set_motion_parameters(double drift_h, double drift_v, double coriolis, double gradient_factor)
{

movement_speed_lon = drift_h;
movement_speed_lat = drift_v;

coriolis_bending = coriolis;

gradient_acceleration_factor = gradient_factor;



}

void Storm::move(double timestep)
{
double speed_factor;

if (is_decayed) {return;}

//total_speed = sqrt(movement_speed_lon * movement_speed_lon + movement_speed_lat * movement_speed_lat);

speed_factor = 1.0;

//if (total_speed > 5.0) { speed_factor = 0.0;}

if (gradient_direction >= 0.0)
	{
	movement_speed_lon -= sin(gradient_direction) * timestep * gradient_acceleration * gradient_acceleration_factor * speed_factor;
	movement_speed_lat += cos(gradient_direction) * timestep * gradient_acceleration * gradient_acceleration_factor * speed_factor;
	}
else
	{
	movement_speed_lon *= (1.0 - (0.001 * timestep/1000.0));
	movement_speed_lat *= (1.0 - (0.001 * timestep/1000.0));
	}


lon -= timestep * movement_speed_lon /R_planet * cos(lat * deg_to_rad);

if (lat > 0.5*pi) {lon = lon + pi; movement_speed_lat = - movement_speed_lat;}
else if (lat < -0.5*pi) {lon = lon + pi; movement_speed_lat = - movement_speed_lat;}

if (lon > 2.0 * pi) {lon -= 2.0*pi;}
else if (lon < 0.0) {lon += 2.0 * pi;}



if ((lat > 0.0) && (movement_speed_lon > 0.0))
	{
	movement_speed_lat += coriolis_bending * timestep/1000.0;
	movement_speed_lon -= coriolis_bending * timestep/1000.0;
	}
else
	{
	movement_speed_lat -= coriolis_bending * timestep/1000.0;
	movement_speed_lon -= coriolis_bending * timestep/1000.0;
	}



lat += timestep * movement_speed_lat/R_planet;

}



void Storm::evolve(double timestep)
{
//int category_before;

//category_before = category;

lifetime_s += timestep;

if (is_decaying)
	{
	strength *= (1.0 - 0.004 * timestep/1000.0);
	}
else
	{
	strength *= (1.0 + 0.001 * timestep/1000.0);
	}

if (strength < 18.0) {is_decayed = true;} 

size = (size_base + (strength -20.0) * 5000.0) * size_modifier;

energy_consumption = base_energy_consumption * strength * strength * size * size /(size_base * size_base) / 400.0;



if (strength > 33.0) {category = 1;}
if (strength > 43.0) {category = 2;}
if (strength > 50.0) {category = 3;}
if (strength > 58.0) {category = 4;}
if (strength > 70.0) {category = 5;}
if (strength > 85.0) {category = 6;}
if (strength > 100.0) {category = 7;}
if (strength > 120.0) {category = 8;}
if (strength > 145.0) {category = 9;}
if (strength > 175.0) {category = 10;}
if (strength > 205.0) {category = 11;}
if (strength > 240.0) {category = 12;}
if (strength > 275.0) {category = 13;}
if (strength > 315.0) {category = 14;}
if (strength > 360.0) {category = 15;}

/*if (category_before != category) 
	{
	if (category > 4){cout << "*** Storm changed from category " << category_before << " to " << category << " on day " << lifetime_s/86400.0 << endl;}
	}
*/

if (category > cat_max)
	{
	cat_max = category;
	}

}

double Storm::get_age_factor()
{
if (is_decayed)
	{
	if (strength < 20.0) {return (strength - 18.0)/2.0;}
	else {return 1.0;}
	}
else
	{
	if (strength < 22.0) {return (strength - 20.0)/2.0;}
	else {return 1.0;}
	}

}

void Storm::set_gradient_dir(double value)
{
gradient_direction = value;
}

void Storm::set_decaying()
{
is_decaying = true;
}

void Storm::set_active()
{
is_decaying = false;
}

bool Storm::get_decayed()
{
return is_decayed;
}

double Storm::get_lat()
{
return lat;
}

double Storm::get_lon()
{
return lon;
}

double Storm::get_size()
{
return size;
}

double Storm::get_energy_need()
{

return energy_consumption;
}

double Storm::get_lifetime()
{
return lifetime_s;
}

int Storm::get_category()
{
return category;
}

int Storm::get_category_max()
{
return cat_max;
}




void Weather::init(double omega_in, double R_in, int circulation_model_in)
{

double coriolis_deg;
int n_cyclone;

//cout << "Weather init!" << endl;

omega = omega_in;
R = R_in;

lat_to_m = 2.0 * pi * R / 360.0;
m_to_lat = 1.0/lat_to_m;
ridge_migration_deg = 0.0;
cyclone_min_lifetime = 691200;
cyclone_max_lifetime = 1382400;

simulation_timestep = 1000.0;

coriolis_deg = coriolis_radius (130.0, 0.0);

compute_storms = false;

//cout << "Circulation model is now: " << circulation_model_in << endl;

if (circulation_model_in == -1)
	{
	if (coriolis_deg >= 45.0) // single cell convection
		{
		circulation_model = 1;
		}
	else if ((coriolis_deg < 45.0) && (coriolis_deg >= 20.0)) // two Hadley cells and a jetstream 
		{
		circulation_model = 2;
		}
	else if ((coriolis_deg < 20.0) && (coriolis_deg >= 14.0)) // two Hadley cells, two jetstreams and a counter-rotating belt
		{
		circulation_model = 3;
		}
	else if ((coriolis_deg < 14.0) && (coriolis_deg >= 8.0)) // two Hadley cells, three jetstreams and two counter-rotating belts
		{
		circulation_model = 4;
		}
	else 
		{
		cout << "No solution for this global circulation pattern, exiting..." << endl;
		}
	}
else
	{
	circulation_model = circulation_model_in;
	}

set_circulation_parameters(circulation_model);

cout << "Weather initialized with " << num_cells << " convection cells";
if (symmetric_circulation) {cout << " per hemisphere." << endl;}
else {cout << " in total." << endl;}


cell_size_deg = 90.0/ num_cells;

cyclone_max_size = 1.2 * cell_size_deg * deg_to_rad * R;
cyclone_min_size = 0.25 * cyclone_max_size;
cyclone_average_number = 2.0 * num_jetstreams * 0.707 * 2.0 * pi * R/ 7075000.0;
ridge_suppression_factor = 0.1;
polar_suppression_factor = 0.5;

if (num_jetstreams > 0)
	{
	cout << "Average number of cyclones per band is " << cyclone_average_number / (2.0 *num_jetstreams) << endl;
	cout << "Average cyclone size is " << cyclone_max_size * 0.625/1000. << " km" << endl;
	}

cyclone_spawn_probability = cyclone_average_number/(0.5 * (cyclone_max_lifetime + cyclone_min_lifetime));
precipitation_factor = 1.0;
humidity_factor = 1.0;
humidity_rain_factor = 1.0;
humidity_cloud_factor = 1.0;
humidity_high_cloud_factor = 1.0;
humidity_mid_cloud_factor = 1.0;
mean_atmosphere_temperature = 288.0;
midlevel_timer = 0.0;
precipitation_T_feedback = 1.0;
cloud_T_feedback = 0.0;
high_cloud_T_feedback = 0.0;
mid_cloud_T_feedback = 0.0;

storm_average_lifetime = 0.0;

num_cyclones = 0;
num_storms = 0;
num_storms_total = 0;


for (int i=0; i<16; i++)
{
num_storms_cat[i] = 0;
}

medium_latent_heat = L_vap_h2o;
medium_gas_constant = R_h2o;

n_cyclone = static_cast<int>(cyclone_average_number  / (2.0 *num_jetstreams));





for (int i=0; i< n_cyclone ; i++)
	{
	for (int j=0; j< num_jetstreams; j++)
		{
		if (symmetric_circulation)
			{
			create_cyclone(jetstream_pos_deg[j]);
			create_cyclone(-jetstream_pos_deg[j]);
			}
		else
			{
			create_cyclone(jetstream_pos_deg[j]);
			}
		}
	}

for (int i=0; i< num_cyclones; i++)
	{
	cyclone_array[i].init_age(rnd(0.0, 1.0));
	}
create_midlevel_clouds();
create_highlevel_clouds();
update_rn_grids();
}


void Weather::set_circulation_parameters(int model_id)
{

if (model_id == 1)
	{
	circulation_model = 1;
	num_cells = 1;
	num_jetstreams = 0;
	ridge_center_deg[0] = 0.0;
	ridge_width_deg = 0.0;
	mean_windspeed = 1.5;
	symmetric_circulation = true;
	}
else if (model_id == 2)
	{
	circulation_model = 2;
	num_cells = 3;
	num_jetstreams = 1;
	jetstream_pos_deg[0] = 45.0;
	ridge_center_deg[0] = 30.0;
	ridge_width_deg = 15.0;
	mean_windspeed = 5.5;
	symmetric_circulation = true;
	}
else if (model_id == 3)
	{
	circulation_model = 3;
	num_cells = 5;
	num_jetstreams = 2;
	jetstream_pos_deg[0] = 27.0;
	jetstream_pos_deg[1] = 63.0;
	ridge_center_deg[0] = 18.0;
	ridge_width_deg = 9.0;
	mean_windspeed = 5.5;
	symmetric_circulation = true;
	}
else if (model_id == 4)
	{
	circulation_model = 4;
	num_cells = 7;
	num_jetstreams = 3;
	jetstream_pos_deg[0] = 15.0;
	jetstream_pos_deg[1] = 45.0;
	jetstream_pos_deg[2] = 75.0;
	ridge_center_deg[0] = 10.0;
	ridge_width_deg = 5.0;
	mean_windspeed = 5.5;
	symmetric_circulation = true;
	}
else if (model_id == 10)
	{
	circulation_model = 10;
	num_cells = 5;
	num_jetstreams = 2;
	jetstream_pos_deg[0] = 30.0;
	jetstream_pos_deg[1] = -30.0;
	ridge_center_deg[0] = 45.0;
	ridge_center_deg[1] = -15.0;
	ridge_width_deg = 15.0;
	mean_windspeed = 5.5;
	symmetric_circulation = false;
	}
else
	{
	cout << "Circulation model " << model_id << " not implemented, exiting..." << endl;
	exit(0);
	}


}



double Weather::rnd(double min, double max)
{
double average = 0.5 * (max + min);
double variance = max - average;

return average + variance * 2.0 * ( rand()/(RAND_MAX +1.0) - 0.5);
}

void Weather::create_midlevel_clouds()
{
for (int i=0; i< 18; i++)
	{
	for (int j=0; j< 36; j++)
		{
		midlevel_grid[i][j] = rnd(midlevel_min, midlevel_max);
		}

	}
}

void Weather::create_highlevel_clouds()
{
for (int i=0; i< 18; i++)
	{
	for (int j=0; j< 36; j++)
		{
		highlevel_grid[i][j] = rnd(highlevel_min, highlevel_max);
		}

	}
}

void Weather::update_rn_grids()
{
for (int i=0; i< 18; i++)
	{
	for (int j=0; j< 36; j++)
		{
		convection_rn_cloud_grid[i][j] = rnd(0.0, 1.0);
		convection_rn_rain_grid[i][j] = rnd(0.0, 1.0);
		}

	}
}

double Weather::get_mid_cloudcover(double lat, double lon)
{
int i,j;

i = static_cast<int>(18.0 * (lat + 0.5 * pi) / pi);
j = static_cast<int>(36.0 * lon / (2.0 *pi));

return midlevel_grid[i][j] * humidity_mid_cloud_factor;

}


double Weather::get_high_cloudcover(double lat, double lon)
{
int i,j;

i = static_cast<int>(18.0 * (lat + 0.5 * pi) / pi);
j = static_cast<int>(36.0 * lon / (2.0 *pi));

return highlevel_grid[i][j] * humidity_high_cloud_factor;

}

double Weather::get_convection_rn_cloud(double lat, double lon)
{
int i,j;

i = static_cast<int>(18.0 * (lat + 0.5 * pi) / pi);
j = static_cast<int>(36.0 * lon / (2.0 *pi));

return convection_rn_cloud_grid[i][j];
}

double Weather::get_convection_rn_precipitation(double lat, double lon)
{
int i,j;

i = static_cast<int>(18.0 * (lat + 0.5 * pi) / pi);
j = static_cast<int>(36.0 * lon / (2.0 *pi));

return convection_rn_rain_grid[i][j];
}


void Weather::create_cyclone(double track)
{

double size, lat, lon, max_lifetime;

size = rnd(cyclone_min_size, cyclone_max_size);
lat = track - 0.5 * cell_size_deg + rnd(0, cell_size_deg) + ridge_migration_deg;
lon = rnd(0.0, 360.0);
max_lifetime = rnd(cyclone_min_lifetime, cyclone_max_lifetime);

Cyclone new_cyclone(lat, lon, size, max_lifetime, R);
cyclone_array[num_cyclones] = new_cyclone;
num_cyclones++;

}

void Weather::clausius_clapeyron(double T)
{
double medium_ratio;

medium_ratio = medium_latent_heat/ medium_gas_constant;
// normalize approx. by value for 288 K, i.e.  2.783468
humidity_factor = exp(medium_ratio * ((1.0/273.15) - (1.0/T))) / 2.5;


humidity_rain_factor = (1.0 - precipitation_T_feedback) + humidity_factor * precipitation_T_feedback;
humidity_cloud_factor = (1.0 - cloud_T_feedback) + humidity_factor * cloud_T_feedback;
humidity_high_cloud_factor = (1.0 - high_cloud_T_feedback) + humidity_factor * high_cloud_T_feedback;
humidity_mid_cloud_factor = (1.0 - mid_cloud_T_feedback) + humidity_factor * mid_cloud_T_feedback;

//cout << "Cloud : " << humidity_cloud_factor << " Rain: " << humidity_rain_factor << endl;
}

void Weather::cleanup()
{
int remove_counter;

if ((num_cyclones == 0)&&(num_storms == 0)) {return;}

remove_counter = 0;

for (int i=0; i< num_cyclones; i++)
	{
	if (cyclone_array[i].get_expired())
		{
		remove_counter ++;
		}
	}

if (remove_counter > 0)
	{
	Cyclone* tmp_array;
	int counter;


	counter = 0;
	tmp_array = new Cyclone[num_cyclones - remove_counter];

	for (int i=0; i< num_cyclones; i++)
		{
		if (cyclone_array[i].get_expired() == false)
			{
			tmp_array[counter] = cyclone_array[i];
			counter++;
			}
		}

	num_cyclones -= remove_counter;
	//cout << "Cleanup - num_cyclones is " << num_cyclones << endl;

	for (int i=0; i< num_cyclones; i++)
		{
		cyclone_array[i] = tmp_array[i];
		}
	}
if ((compute_storms) && (num_storms > 0))
	{
	remove_counter = 0;


	for (int i=0; i< num_storms; i++)
		{
		if (storm_array[i].get_decayed())
			{
			remove_counter ++;
			storm_average_lifetime += storm_array[i].get_lifetime();
			num_storms_cat[storm_array[i].get_category_max()]++;
			}
		}


	if (remove_counter > 0)
		{
		Storm* tmp_storm_array;
		int counter;

		counter = 0;
		tmp_storm_array = new Storm[num_storms - remove_counter];

		for (int i=0; i< num_storms; i++)
			{
			if (storm_array[i].get_decayed() == false)
				{
				tmp_storm_array[counter] = storm_array[i];
				counter++;
				}
			}

		num_storms -= remove_counter;
		//cout << "Cleanup - num_storms is " << num_storms << " average lifetime is " << storm_average_lifetime / num_storms_total / 86400. << endl;

		for (int i=0; i< num_storms; i++)
			{
			storm_array[i] = tmp_storm_array[i];
			}
		}



	}


}

void Weather::evolve(double timestep)
{
int track_id;
double rn_track, track;


simulation_timestep = timestep;

cleanup();

if (rnd(0.0, 1.0) < (cyclone_spawn_probability * timestep))
	{
	rn_track = rnd(0.0, num_jetstreams);
	track_id = static_cast<int>(floor(rn_track));


	track= jetstream_pos_deg[track_id];
	if (symmetric_circulation)
		{
		if (rnd(0.0, 1.0) < 0.5) {track= - jetstream_pos_deg[track_id];}
		}

	create_cyclone(track);
	//cout << "Creation: " << num_cyclones << endl;
	}

for (int i=0; i< num_cyclones; i++)
	{
	cyclone_array[i].move(timestep);
	}


for (int i=0; i < num_storms; i++)
	{
	storm_array[i].evolve(timestep);
	storm_array[i].move(timestep);
	}


midlevel_timer += timestep;
if (midlevel_timer > 28800.0)
	{
	midlevel_timer -=28800.0;
	create_midlevel_clouds();
	create_highlevel_clouds();
	update_rn_grids();
	}

}

void Weather::set_random(int seed)
{
if (seed != -1) {srand(seed); }
else {srand(time(0)); }
}

void Weather::set_ridge_migration_deg(double value)
{
ridge_migration_deg = value;
}


void Weather::set_midlevel(double min, double max)
{
midlevel_min = min;
midlevel_max = max;
}

void Weather::set_highlevel(double min, double max)
{
highlevel_min = min;
highlevel_max = max;
}

void Weather::set_precipitation_factor(double value)
{
precipitation_factor = value;
}


void Weather::set_parameter(std::string name, double value)
{

if (name == "precipitation_feedback")
	{
	precipitation_T_feedback = value;
	}
else if (name == "cloud_feedback")
	{
	cloud_T_feedback = value;
	}
else if (name == "high_cloud_feedback")
	{
	high_cloud_T_feedback = value;
	}
else if (name == "mid_cloud_feedback")
	{
	mid_cloud_T_feedback = value;
	}
else if (name == "ridge_suppression_factor")
	{
	ridge_suppression_factor = value;
	}
else if (name == "grad_motion_factor")
	{
	grad_motion_factor = value;
	}

if (name == "storms")
	{
	if (value == 1.0) {compute_storms = true;} else {compute_storms = false;}
	}

}

void Weather::set_atmosphere_temperature(double value)
{
mean_atmosphere_temperature = value;
clausius_clapeyron(mean_atmosphere_temperature);
}

double Weather::get_low_cloudcover(double lat, double lon, double res_lat, double res_lon)
{
double delta_lat, delta_lon, delta_ang, res_cell, size;
double cloudcover, occupation_factor, distance_factor, lat_factor;

cloudcover = 0.0;
res_cell = 0.5 * (res_lat + res_lon);



lat_factor = polar_suppression_factor * (1.0 - ((abs(lat) - 1.22173045) / 0.34906));

for (int i=0; i< num_cyclones; i++)
	{

	if (cyclone_array[i].get_expired()) {continue;}

	delta_lat = (lat - cyclone_array[i].get_lat()) ;
	delta_lon = (lon - cyclone_array[i].get_lon()) ;

	if (delta_lon > pi) {delta_lon -= 2.0 * pi;}
	if (delta_lon < -pi) {delta_lon += 2.0 * pi;}

	delta_lat/= res_cell;
	delta_lon/= res_cell;

	delta_ang = sqrt(pow(delta_lat, 2.0) + pow(delta_lon, 2.0));

	
	size = cyclone_array[i].get_size() * m_to_lat * deg_to_rad / res_cell;



	occupation_factor = 1.0;

	if (size < 1.0) 
		{
		occupation_factor = size * size;
		}


	distance_factor = exp(- pow(delta_ang/size, 2.0));

	if (delta_ang < (size -0.5))
		{
		cloudcover  +=occupation_factor * distance_factor * cyclone_array[i].get_age_factor();
		}
	else if (delta_ang < (size + 0.5))
		{
		cloudcover  +=occupation_factor * (delta_ang - size + 0.5) * distance_factor * cyclone_array[i].get_age_factor();
		}
	}

for (int i=0; i< num_storms; i++)
	{

	if (storm_array[i].get_decayed()) {continue;}

	delta_lat = (lat - storm_array[i].get_lat()) ;
	delta_lon = (lon - storm_array[i].get_lon()) ;

	if (delta_lon > pi) {delta_lon -= 2.0 * pi;}
	if (delta_lon < -pi) {delta_lon += 2.0 * pi;}

	delta_lat/= res_cell;
	delta_lon/= res_cell;

	delta_ang = sqrt(pow(delta_lat, 2.0) + pow(delta_lon, 2.0));

	
	size = storm_array[i].get_size() * m_to_lat * deg_to_rad / res_cell;

	occupation_factor = 1.0;

	if (size < 1.0) 
		{
		occupation_factor = size * size;
		}

	distance_factor = exp(- pow(delta_ang/size, 2.0));

	if (delta_ang < (size -0.5))
		{
		cloudcover  +=occupation_factor * distance_factor * storm_array[i].get_age_factor();
		}
	else if (delta_ang < (size + 0.5))
		{
		cloudcover  +=occupation_factor * (delta_ang - size + 0.5) * distance_factor * storm_array[i].get_age_factor();
		}
	}


cloudcover *= lat_factor * humidity_cloud_factor;

if (cloudcover > 1.0) {cloudcover = 1.0;}

return cloudcover;

}


double Weather::get_storm_cloudcover(double lat, double lon, double res_lat, double res_lon)
{
double delta_lat, delta_lon, delta_ang, res_cell, size;
double cloudcover, occupation_factor, distance_factor;

cloudcover = 0.0;
res_cell = 0.5 * (res_lat + res_lon);


for (int i=0; i< num_storms; i++)
	{

	if (storm_array[i].get_decayed()) {continue;}

	delta_lat = (lat - storm_array[i].get_lat()) ;
	delta_lon = (lon - storm_array[i].get_lon()) ;

	if (delta_lon > pi) {delta_lon -= 2.0 * pi;}
	if (delta_lon < -pi) {delta_lon += 2.0 * pi;}

	delta_lat/= res_cell;
	delta_lon/= res_cell;

	delta_ang = sqrt(pow(delta_lat, 2.0) + pow(delta_lon, 2.0));

	
	size = storm_array[i].get_size() * m_to_lat * deg_to_rad / res_cell;
	
	//cout << storm_array[i].get_size() << " " << size << " " << humidity_cloud_factor << endl;

	occupation_factor = 1.0;

	if (size < 1.0) 
		{
		occupation_factor = size * size;
		}

	distance_factor = exp(- pow(delta_ang/size, 2.0));
	//distance_factor = 1.0;

	if (delta_ang < (size -0.5))
		{
		cloudcover  +=occupation_factor * distance_factor * storm_array[i].get_age_factor();
		}
	else if (delta_ang < (size + 0.5))
		{
		cloudcover  +=occupation_factor * (delta_ang - size + 0.5) * distance_factor * storm_array[i].get_age_factor();
		}
	}


cloudcover *= humidity_cloud_factor;

if (cloudcover > 1.0) {cloudcover = 1.0;}

return cloudcover;

}

double Weather::get_precipitation(double lat, double lon, double res_lat, double res_lon)
{

double delta_lat, delta_lon, delta_ang, res_cell, size;
double precipitation, occupation_factor, distance_factor;

precipitation = 0.0;
res_cell = 0.5 * (res_lat + res_lon);


for (int i=0; i< num_cyclones; i++)
	{

	if (cyclone_array[i].get_expired()) {continue;}

	delta_lat = (lat - cyclone_array[i].get_lat()) ;
	delta_lon = (lon - cyclone_array[i].get_lon()) ;

	if (delta_lon > pi) {delta_lon -= 2.0 * pi;}
	if (delta_lon < -pi) {delta_lon += 2.0 * pi;}

	delta_lat/= res_cell;
	delta_lon/= res_cell;

	delta_ang = sqrt(pow(delta_lat, 2.0) + pow(delta_lon, 2.0));

	size = cyclone_array[i].get_size() * m_to_lat * deg_to_rad / res_cell;

	occupation_factor = 1.0;

	if (size < 1.0) 
		{
		occupation_factor = size * size;
		}

	distance_factor = exp(- pow(delta_ang/(0.5 *size), 2.0));

	if (delta_ang < (size -0.5))
		{
		precipitation  +=occupation_factor * distance_factor * cyclone_array[i].get_age_factor() * 5.0 * (cyclone_array[i].get_size() / 1000000.0) / 86400.0;
		}
	else if (delta_ang < (size + 0.5))
		{
		precipitation  +=occupation_factor * (delta_ang - size + 0.5) * distance_factor * cyclone_array[i].get_age_factor() * 5.0 * (cyclone_array[i].get_size() / 1000000.0) / 86400.0;
		}
	}


for (int i=0; i< num_storms; i++)
	{

	if (storm_array[i].get_decayed()) {continue;}

	delta_lat = (lat - storm_array[i].get_lat()) ;
	delta_lon = (lon - storm_array[i].get_lon()) ;

	if (delta_lon > pi) {delta_lon -= 2.0 * pi;}
	if (delta_lon < -pi) {delta_lon += 2.0 * pi;}

	delta_lat/= res_cell;
	delta_lon/= res_cell;

	delta_ang = sqrt(pow(delta_lat, 2.0) + pow(delta_lon, 2.0));

	size = storm_array[i].get_size() * m_to_lat * deg_to_rad / res_cell;

	occupation_factor = 1.0;

	if (size < 1.0) 
		{
		occupation_factor = size * size;
		}

	distance_factor = exp(- pow(delta_ang/(0.5 *size), 2.0));

	if (delta_ang < (size -0.5))
		{
		precipitation  +=occupation_factor * distance_factor * storm_array[i].get_age_factor() * 5.0 * (storm_array[i].get_size() / 100000.0) / 86400.0;
		}
	else if (delta_ang < (size + 0.5))
		{
		precipitation  +=occupation_factor * (delta_ang - size + 0.5) * distance_factor * storm_array[i].get_age_factor() * 5.0 * (storm_array[i].get_size() / 100000.0) / 86400.0;
		}
	}



return precipitation * precipitation_factor * humidity_rain_factor;


}


double Weather::coriolis_radius (double v, double lat_in)
{

double vx, vy, vx_tmp;
double x, y, lat;
double dt = 0.1;
double f;

vx = 0; vy = v;
x = 0; y = lat_in/R;
lat = lat_in;

while ((vy > 0.0) && (lat < 1.1))
	{
	x += vx * dt;
	y += vy * dt;

	lat = lat_in + y/R;
	f = 2.0 * omega * sin(lat);
	vx_tmp = vx;
	
	vx += f * vy * dt;
	vy -= f * vx_tmp * dt;


	}

return (y/R - lat_in) * rad_to_deg;

}



double Weather::get_ridge_deg_low(int hemisphere)
{
if (symmetric_circulation)
	{
	if (hemisphere == 1)
		{return ridge_center_deg[0] - ridge_width_deg;}
	else
		{return -ridge_center_deg[0] - ridge_width_deg;}
	}
else
	{
	if (hemisphere == 1)
		{return ridge_center_deg[0] - ridge_width_deg;}
	else	
		{return ridge_center_deg[1] - ridge_width_deg;}
	}
}

double Weather::get_ridge_deg_high(int hemisphere)
{
if (symmetric_circulation)
	{
	if (hemisphere == 1)
		{return ridge_center_deg[0] + ridge_width_deg;}
	else
		{return -ridge_center_deg[0] + ridge_width_deg;}
	}
else
	{
	if (hemisphere == 1)
		{return ridge_center_deg[0] + ridge_width_deg;}
	else	
		{return ridge_center_deg[1] + ridge_width_deg;}
	}
}

double Weather::get_ridge_suppression()
{
return ridge_suppression_factor;
}


void Weather::check_storm_generation(double lat, double lon, double delta_lat, double delta_lon, double convective_energy)
{
double P_storm;

P_storm = 0.0;

if (convective_energy > 1.7e8) {P_storm = 5.0 * 4.5e-5 * simulation_timestep/1000.0;}
else if (convective_energy > 1.2e8) {P_storm = 5.0 * 2.5e-5 * simulation_timestep/1000.0;}
else if (convective_energy > 0.7e8) {P_storm = 5.0 * 1.5e-5 * simulation_timestep/1000.0;}

if (num_storms > 49) {return;}

if (rnd(0.0, 1.0) < P_storm)
	{
	lat -= - rnd(-0.5, 0.5) * delta_lat;
	lon -= - rnd(-0.5, 0.5) * delta_lon;

	for (int i=0; i< num_storms; i++)
		{
		double dist_lat = lat - storm_array[i].get_lat();
		double dist_lon = lon - storm_array[i].get_lon();

		if (sqrt(dist_lat * dist_lat + dist_lon * dist_lon) < 1.0)
			{
			//cout << "Storm distance blocking..." << endl; 
			return;
			}

		}

	Storm new_storm(lat  , lon , R);
	storm_array[num_storms] = new_storm;
	storm_array[num_storms].set_motion_parameters(mean_windspeed, 0.0, omega * 41.1409764125069, grad_motion_factor);
	storm_array[num_storms].randomize(rnd(0.5,2.0), rnd(0.0, 2.0), rnd(0.8, 2.5));
	num_storms++;
	num_storms_total++;
	
	//cout << "Storm " << num_storms_total << " spawned on " << storm_array[num_storms-1].get_lat() * rad_to_deg << " " << storm_array[num_storms-1].get_lon() * rad_to_deg << endl;
	}
}

double Weather::get_storm_energy_usage(double lat_min, double lat_max, double lon_min, double lon_max)
{
double lat_s, lon_s;

for (int i=0; i< num_storms; i++)
	{
	lat_s = storm_array[i].get_lat();
	lon_s = storm_array[i].get_lon();

	if ((lat_s >= lat_min) && (lat_s <= lat_max) && (lon_s >= lon_min) && (lon_s <= lon_max))
		{
		return storm_array[i].get_energy_need();
		}
	}

return 0.0;

}

void Weather::set_storm_energy_usage(double lat_min, double lat_max, double lon_min, double lon_max, double dir, int flag)
{

double lat_s, lon_s;

for (int i=0; i< num_storms; i++)
	{
	lat_s = storm_array[i].get_lat();
	lon_s = storm_array[i].get_lon();

	if ((lat_s >= lat_min) && (lat_s <= lat_max) && (lon_s >= lon_min) && (lon_s <= lon_max))
		{
		if (flag == 0) 
			{
			storm_array[i].set_decaying();
			}
		else
			{
			storm_array[i].set_active();
			}
		if (dir >= 0.0)
			{
			storm_array[i].set_gradient_dir(dir);
			}
		
		}
	}


}

int Weather::get_num_storms(void)
{
return num_storms;
}

double Weather::get_storm_lat(int index)
{
if (index > num_storms-1) {return 0.0;}

return storm_array[index].get_lat();

}

double Weather::get_storm_lon(int index)
{
if (index > num_storms-1) {return 0.0;}

return storm_array[index].get_lon();

}

int Weather::get_storm_category(int index)
{
if (index > num_storms-1) {return -1;}

return storm_array[index].get_category();
}


void Weather::list_storms(double total_evolution_time)
{
cout << endl;
cout << "Storm generation: " << endl;
cout << "----------------- " << endl;
cout << "per year:         " << num_storms_total / total_evolution_time * 31536000.0 << endl;
cout << "lifetime [d]:     " << storm_average_lifetime / num_storms_total / 86400.0 << endl;
for (int i=0; i< 16; i++)
	{
	cout << "category: ";
	if (i < 10) {cout << " ";}
	cout << i << "      " << num_storms_cat[i] /total_evolution_time * 31536000.0 << endl;;
	} 

}
