// terrain_relief.cxx -- routines to assign and compute terrain relief
//
// Written by Thorsten Renk, started 2025
//
// Copyright (C) 2025  Thorsten Renk - thorsten@science-and-fiction.org 
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301

#include <iostream>
#include <stdlib.h>
#include <math.h>
#include <fstream>

#include "terrain_relief.hxx"

using namespace std;


void TerrainRelief::set_random_seed(int seed)
{
srand(seed);
}


double TerrainRelief::rnd(double min, double max)
{
double average = 0.5 * (max + min);
double variance = max - average;

return average + variance * 2.0 * ( rand()/(RAND_MAX +1.0) - 0.5);
}

double TerrainRelief::pseudo_rnd(double x, double y)
{
double intpart;

return  modf(10. + 24.5*sin((cos(x) * 129.98 + cos(y) * 776.233) * 457338.5453),&intpart);

}


double TerrainRelief::clamp_01(double x)
{
if (x < 0.0) {return 0.0;}
if (x > 1.0) {return 1.0;}
return x;
}

double TerrainRelief::smoothstep(double edge1, double edge2, double x)
{

//if (edge1 == edge2) {cout << "Edges coincide" << edge1 << " " << edge2 << endl;}

if (x< edge1) {return edge1;}
if (x> edge2) {return edge2;}

x =  (x - edge1) / (edge2 - edge1);

return x * x * (3.0- 2.0*x);

}


double TerrainRelief::perlin_noise(double x, double y, double scale, double weight)
{
double x0, y0, x1, y1;
double frac_x, frac_y;
double int1, int2;

frac_x = modf(x/scale, &x0);
frac_y = modf(y/scale, &y0);


x1 = x0 + 1.0;
y1 = y0 + 1.0;


int1 = smoothstep(pseudo_rnd(x0, y0), pseudo_rnd(x1, y0), frac_x);
int2 = smoothstep(pseudo_rnd(x0, y1), pseudo_rnd(x1, y1), frac_x);


return weight * smoothstep(int1, int2, frac_y);

}


void TerrainRelief::set_general_parameters(int lat, int lon, double scale, double time, double timestep_in, string file)
{
res_lat = lat;
res_lon = lon;
base_scale = scale;
evolution_time = time;
timestep = timestep_in;
plotfile = file;

if ((base_scale > 0.0) && (res_lat > 10) && (res_lon > 10))
	{
	create_surface();
	}
else
	{
	cout << "Terrain surface resolution is poorly defined, check parameters - exiting..." << endl;
	exit(0);
	}

do_impacts = false;
do_flow_erosion = false;
do_tectonics = false;


}

void TerrainRelief::set_impact_parameters(double rate, double distribution, double density, double gravity)
{

impact_rate = rate;
impact_mass_distribution = distribution;
impact_atmosphere_density = density;
impact_gravity = gravity;

if (rate == 0.0) {return;}

do_impacts = true;

}


void TerrainRelief::set_flow_erosion_parameters(double rain, double rain_var, double ev, double hard)
{
fe_rainfall = rain;
fe_rainfall_var = rain_var;
fe_evaporation  = ev;
fe_hardness = hard;


if (fe_rainfall == 0.0) {return;}

do_flow_erosion = true;
}


void TerrainRelief::create_surface()
{

// dynamical memory allocation for surface element array

terrain_elements = new TerrainElement *[res_lat];

for (int i=0; i< res_lat; i++)
	{
	terrain_elements[i] = new TerrainElement [res_lon];
	}

for (int i=0; i<res_lat; i++)
	{
	for (int j=0; j< res_lon; j++)
		{
		double corr;

		corr = 0.0 * (j-25) * (j-25);

		/*if ((i > 20) && (i< 26) && (j >35) && (j<55))
			{
			corr-=100.0;
			}*/

		/*corr = 0.0;

		if ((i > 4) && (i<8) && (j> 3) && (j<9))
			{
			corr = -40.0;
			}
		else if ((i > 7) && (i<11) && (j> 5) && (j<9))
			{
			corr = -55.0;
			}

		if ((i==7) && (j==6))
			{
			corr+=0.0;
			}*/

		
		corr += perlin_noise(i*base_scale, j*base_scale, 32.0 * base_scale, 20.0);
		corr += perlin_noise(i*base_scale, j*base_scale, 16.0 * base_scale, 20.0);
		corr += perlin_noise(i*base_scale, j*base_scale, 8.0 * base_scale, 10.0);
		corr += perlin_noise(i*base_scale, j*base_scale, 4.0 * base_scale, 5.0);
		//cout << corr << endl;
		//corr += pseudo_rnd(i * base_scale,j* base_scale) * 50.0;
	
		terrain_elements[i][j].set_elevation(rnd(0.0, base_scale * 5.0)+ i * 100./res_lat + corr);
		 //terrain_elements[i][j].set_elevation(rnd(0.0, base_scale * 50.0)+ i * 100./res_lat + corr);
		}
	}






}



void TerrainRelief::create_meteor_impact()
{

int nx, ny;
double size, depth, wall_height;


nx = static_cast<int>(rnd(0, 1.0 * res_lon));
ny = static_cast<int>(rnd(0, 1.0 * res_lat));

size = 5.0 + (1.0 - pow(rnd(0.0, 1.0),(0.005 + impact_mass_distribution * 0.11))) * 150.0;
depth = size/7.0 * 1000.0;
wall_height = 0.5 * depth;

meteor_impact (nx, ny, size, wall_height, depth); 
	

}

void TerrainRelief::meteor_impact(int pos_x, int pos_y, double radius, double wall_height, double depth)
{

double wall_radius_outer, wall_radius_inner,  wall_height_var;
double distance, current_elevation, tgt_elevation;
double sample_elevation;

int sample_range, sample_pos_x, sample_pos_y;

wall_radius_outer = 0.3 * radius;
wall_radius_inner = 0.1 * radius;
wall_height_var = 0.1 * wall_height;

// sample impact elevation

sample_range = static_cast<int>(0.5* radius/base_scale);

sample_elevation = terrain_elements[pos_x][pos_y].get_elevation();

sample_pos_x = pos_x + sample_range;
if (sample_pos_x > (res_lat-1)) {sample_pos_x = res_lat -1;}
sample_elevation += terrain_elements[sample_pos_x][pos_y].get_elevation();

sample_pos_y = pos_y + sample_range;
if (sample_pos_y > (res_lon-1)) {sample_pos_y = res_lon -1;}
sample_elevation += terrain_elements[pos_x][sample_pos_y].get_elevation();

sample_pos_x = pos_x - sample_range;
if (sample_pos_x < 0) {sample_pos_x = 0;}
sample_elevation += terrain_elements[sample_pos_x][pos_y].get_elevation();

sample_pos_y = pos_y - sample_range;
if (sample_pos_y < 0) {sample_pos_y = 0;}
sample_elevation += terrain_elements[pos_x][sample_pos_y].get_elevation();

sample_elevation /=5.0;

for (int i=0; i<res_lat; i++)
	{
	for (int j=0; j< res_lon; j++)
		{
		distance = base_scale * sqrt(1.0 * (pos_x - i) * (pos_x- i) + (pos_y - j) * (pos_y - j));


		if (distance < (radius - wall_radius_inner))
			{
			current_elevation = terrain_elements[i][j].get_elevation();

			terrain_elements[i][j].set_elevation(sample_elevation-depth);
			}
		else if (distance < radius)
			{
			current_elevation = terrain_elements[i][j].get_elevation();
			tgt_elevation = (wall_height) * (distance - (radius - wall_radius_inner)) / wall_radius_inner; 
			tgt_elevation += rnd(-wall_height_var, wall_height_var);
			terrain_elements[i][j].set_elevation(current_elevation + tgt_elevation);
			}
		else if (distance < radius + wall_radius_outer)
			{
			current_elevation = terrain_elements[i][j].get_elevation();
			tgt_elevation = (wall_height ) * (1.0 - (distance -radius) / wall_radius_outer); 
			tgt_elevation += rnd(-wall_height_var, wall_height_var);
			terrain_elements[i][j].set_elevation(current_elevation  + tgt_elevation);

			}

		}
	}

}



void TerrainRelief::create_rainfall()
{

double elevation, elevation_test, delta_e, delta_e_test, gradient_test, gradient, dist;
int xtest, ytest, xetest, yetest;

xtest = 0;
ytest = 0;



for (int i=0; i<res_lat; i++)
	{
	for (int j=0; j< res_lon; j++)
		{
		terrain_elements[i][j].reset_flow();
		terrain_elements[i][j].set_rainfall(fe_rainfall * timestep);
		terrain_elements[i][j].set_water_level(0.0);
		terrain_elements[i][j].unset_lake();
		gradient= -10000.0;
		delta_e = -10000.0;
		

		// water may always flow off the edges
		if ((i==0) || (j==0) || (i == (res_lat-1)) || (j == (res_lon -1))) 
			{
			terrain_elements[i][j].set_sink();
			}
		// find the flow direction
		else
			{
			elevation = terrain_elements[i][j].get_elevation();

			for (int k=-1; k<2; k++)	
				{
				for (int l=-1; l<2; l++)
					{

					if ((k==0) && (l==0)) {continue;}

					elevation_test = terrain_elements[i+k][j+l].get_elevation();

					if ((k!=0)&&(l!=0)) {dist = 1.414 * 1000.0 * base_scale;} else {dist = 1000.0 * base_scale;}

					gradient_test = (elevation - elevation_test)/dist;

					delta_e_test = elevation - elevation_test;
					


					if (gradient_test >= gradient) 	
						{
						gradient = gradient_test;
						xtest = k; ytest = l;
						}

					if (delta_e_test >= delta_e)
						{
						delta_e = delta_e_test;
						xetest = k, yetest = l;
						}

 					}	
				}
			terrain_elements[i][j].set_gradient(gradient);
			if (gradient > 0.0)
				{
				terrain_elements[i][j].set_flow_dir(xtest, ytest);
				}
			else 
				{
				terrain_elements[i][j].set_flow_dir(xetest, yetest);
				
				terrain_elements[i][j].set_barrier(-delta_e);
				terrain_elements[i][j].set_minimum();
				//cout << "Local minimum at " << i << " " << j << " elevation " << terrain_elements[i][j].get_water_elevation() << endl;
				}

			}

		}
	}
}


void TerrainRelief::trace_flow ()
{
bool termination_flag;
double flow;
int dx, dx_tmp, dy;


for (int i=0; i<res_lat; i++)
	{
	for (int j=0; j< res_lon; j++)
		{
		dx = 0;
		dy = 0;


		flow = terrain_elements[i][j].get_rainfall();
		flow += terrain_elements[i][j].get_outflow();
		termination_flag = false;


		while (termination_flag == false)
			{
			termination_flag = terrain_elements[i+dx][j+dy].get_terminating();

			if (termination_flag)
				{
				terrain_elements[i+dx][j+dy].add_to_inflow(flow);
				}
			else 	
				{terrain_elements[i+dx][j+dy].add_to_flow(flow);}
			
			dx_tmp = dx;

			dx+= terrain_elements[i+dx][j+dy].get_grad_x();
			dy+= terrain_elements[i+dx_tmp][j+dy].get_grad_y();

			}
		}
	}
}



void TerrainRelief::flood_fill(int i, int j, double level)
{
double barrier_height;

//cout << "Flood_fill: " << i << " " << j << " " << terrain_elements[i][j].get_elevation();

if (terrain_elements[i][j].get_elevation() > level) 
	{
	barrier_height = terrain_elements[i][j].get_elevation() - level;
	if (barrier_height < fe_search_barrier_level)
		{
		fe_search_barrier_level = barrier_height;
		fe_search_x = i; fe_search_y = j;
		}
	//cout << " edge " << barrier_height << endl;
	return;
	}
if (terrain_elements[i][j].check_fill_id(fe_search_id)) {	/*cout << " filled "  << endl;*/ return;}

terrain_elements[i][j].set_water_to_level(level);
terrain_elements[i][j].set_fill_id(fe_search_id);
terrain_elements[i][j].set_lake();
fe_search_lake_size++;
if (terrain_elements[i][j].get_inflow() > 0.0)
	{
	fe_search_inflow += terrain_elements[i][j].get_inflow();
	if (fe_search_final_flag) {terrain_elements[i][j].set_inflow(0.0);}
	}
if (terrain_elements[i][j].get_outflow() > 0.0)
	{
	fe_search_inflow += terrain_elements[i][j].get_outflow(); //cout << i << " " << j << " " << terrain_elements[i][j].get_outflow() << endl;
	//if (fe_search_final_flag) {terrain_elements[i][j].set_outflow(0.0);}
	}

if (terrain_elements[i][j].get_sink()) {/*cout << " sink " << endl;*/ return;}

//if ((i==0) || (j==0)) {cout << "Border issue!" << endl;}
//cout << "Flood " << i << " " << j << " " << level - terrain_elements[i][j].get_water_elevation() << endl;

//cout << " flood " << endl;

flood_fill(i-1, j, level);
flood_fill(i, j-1, level);
flood_fill(i+1, j, level);
flood_fill(i, j+1, level);

flood_fill(i-1, j-1, level);
flood_fill(i+1, j-1, level);
flood_fill(i+1, j+1, level);
flood_fill(i-1, j+1, level);

}


bool TerrainRelief::flood_check_barrier(int i, int j, double level)
{
bool continuation_flag;
int xnext, ynext;
double elevation, elevation_test;

continuation_flag = false;

//cout << "Barrier check " << i << " " << j << endl;


// any terrain out of bounds can drain water
if ((i<0) || (j<0) || (i>res_lat -1) || (j>res_lon-1)) {/*cout << "Passed" << endl;*/ return true;}

// any edge sink can drain water
if (terrain_elements[i][j].get_sink()) {/*cout << "Passed, contact to sink" << endl;*/ return true;}


elevation = terrain_elements[i][j].get_water_elevation();


for (int k=-1; k<2; k++)
	{
	for (int l=-1; l<2; l++)
		{
		// the barrier can't drain itself
		if ((k==0) && (l==0)) {continue;}

		// the barrier can't drain into terrain that are the lake to be drained
		if (terrain_elements[i+k][j+l].check_fill_id(fe_search_id)) {continue;}

		// the barrier can't drain if it is the outflow of a different lake
		if (terrain_elements[i+k][j+l].get_outflow() > 0.0) {continue;}
		
		elevation_test = terrain_elements[i+k][j+l].get_water_elevation();

		//cout << i+k << " " << j+l << " " << elevation_test << " "  << elevation << " " << level << endl;
		//cout << elevation_test - (elevation-0.1) << endl;

		//if ((elevation_test < (elevation-0.1)) && (elevation_test != level))
		if ((elevation_test < elevation))
			{
			//cout << "Passed: Elevation: " << elevation <<  " " << elevation_test << " " << level << " at " << i+k << " " << j+l << endl;
			return true;
			}
		if ((abs(elevation_test - elevation) < 0.1) && (elevation_test != level))
			{
			continuation_flag = true;
			xnext = i+k;
			ynext = j+l;
			} 

			
		}
	}
if (continuation_flag)
	{
	//cout << "Continuing barrier check at " << xnext << " " << ynext << endl;
	return flood_check_barrier(xnext, ynext, level);
	}

//cout << "Failed " << endl;
return false;

}

void TerrainRelief::reset_fill_parameters()
{
fe_search_barrier_level = 1.0e6;
fe_search_inflow = 0;
fe_search_lake_size = 0;
}


void TerrainRelief::flood_from(int i, int j)
{
double tgt_level, barrier;
bool flag;

reset_fill_parameters();
fe_search_final_flag = false;


barrier = terrain_elements[i][j].get_barrier();
if (barrier == 0) {barrier+=1.0;}

tgt_level = 0.99 * barrier + terrain_elements[i][j].get_elevation() ;


//if (terrain_elements[i][j].get_water_elevation() > tgt_level) // we're filling an existing lake
//	{
//	tgt_level = terrain_elements[i][j].get_water_elevation() + 1.0;
//	if ((i==3) && (j==6)) {cout << "Adding..." << endl;}
//	}



//cout << "================" << endl;
//cout << "Filling " << i << " " << j << endl;
//cout << "================" << endl;
//cout << "Filling to level " << tgt_level << endl;





fe_search_id++;
flood_fill(i,j, tgt_level);

if (fe_search_barrier_level == 1.0e6) {cout << "Something is wrong with flooding..." << endl; return;}
if (fe_search_lake_size > 10) {cout << "Fill at " << i << " " << j << " caused large lake " <<endl;}

//cout << "Flood fill ende, next barrier is " << fe_search_barrier_level << " at " << fe_search_x << " " << fe_search_y << endl;


flag = flood_check_barrier (fe_search_x, fe_search_y, tgt_level);

//if (flag== false) {cout << "More flooding needed, barrier " << fe_search_barrier_level << endl; }

while (flag == false)
	{
	tgt_level += 1.0* fe_search_barrier_level;
	//cout << "Filling to level " << tgt_level << endl;
	reset_fill_parameters();
	fe_search_id++;
	flood_fill(i,j, tgt_level);
	if (fe_search_barrier_level == 1.0e6) {cout << "Something is wrong with flooding..." << endl; return;}
	//if (fe_search_lake_size > 10) {cout << "Fill at " << i << " " << j << " caused large lake size "<<  fe_search_lake_size<<endl;}
	//cout << "Flood fill ende, next barrier is " << fe_search_barrier_level << " at " << fe_search_x << " " << fe_search_y << endl;
	//cout << "Lake size is " << fe_search_lake_size << endl;
	flag = flood_check_barrier (fe_search_x,fe_search_y, tgt_level);
	}

// fill to remaining barrier
tgt_level += fe_search_barrier_level;

reset_fill_parameters();
fe_search_final_flag = true;



//cout << "Final flood to level " << tgt_level << endl;
fe_search_id++;
flood_fill(i,j, tgt_level - 0.01);
//if (fe_search_lake_size > 10) {cout << "Fill at " << i << " " << j << " caused large lake " <<endl;}
fe_search_final_flag = false;
terrain_elements[fe_search_x][fe_search_y].set_outflow(fe_search_inflow);
//cout << "Lake size is " << fe_search_lake_size << " outflow: " << fe_search_inflow << endl;
//cout << "Flood fill ende, next barrier is " << fe_search_barrier_level << " at " << fe_search_x << " " << fe_search_y << endl;


}


void TerrainRelief::flood()
{

fe_search_id = 0;

for (int i=0; i<res_lat; i++)
	{
	for (int j=0; j< res_lon; j++)
		{
		//if (terrain_elements[i][j].get_inflow()> 0.0)
		if (terrain_elements[i][j].get_minimum())
			{
			//cout << "Flood check " << i << " " << j <<  " " << terrain_elements[i][j].get_inflow() << endl;
			flood_from(i,j);
			}
		}

	}

}


void TerrainRelief::erode ()
{
double abrasion, gradient;

for (int i=0; i<res_lat; i++)
	{
	for (int j=0; j< res_lon; j++)
		{
		gradient = terrain_elements[i][j].get_gradient();
		if (gradient > 0.5) {gradient = 0.5;}
		abrasion = 1e-3 * terrain_elements[i][j].get_flow() * gradient * fe_hardness;
		if (abrasion < 0.0) {cout << "Adding terrain elevation, there is an issue..." << endl;}
		terrain_elements[i][j].set_elevation(terrain_elements[i][j].get_elevation() - abrasion);
		}

	}
}




void TerrainRelief::reset_lakes ()
{
for (int i=0; i<res_lat; i++)
	{
	for (int j=0; j< res_lon; j++)
		{
		terrain_elements[i][j].unset_lake();
		terrain_elements[i][j].set_water_level(0.0);
		terrain_elements[i][j].set_outflow(0.0);
		terrain_elements[i][j].set_fill_id(-1);
		}
	}
}

void TerrainRelief::clean_up_lakes ()
{
for (int i=0; i<res_lat; i++)
	{
	for (int j=0; j< res_lon; j++)
		{
		terrain_elements[i][j].unset_minimum();
		}
	}
}


void TerrainRelief::evolve()
{
double time;
int iteration;

time = 0.0;
iteration = 0;


/*for (int i=0; i<res_lat; i++)
	{
	cout << i <<  " " << endl;
	for (int j=0; j< res_lon; j++)
		{
		cout << j << " " << terrain_elements[i][j].get_elevation() << " ";
		}
	cout << endl;
	}*/


while (time < evolution_time)
	{

	if (do_impacts)
		{
		if (rnd(0.0, 1.0) < (1.0 - exp(-timestep *impact_rate)))
			{
			create_meteor_impact();
			}
		}
	if (do_flow_erosion)
		{
		create_rainfall(); cout << "Rainfall done" << endl;
		trace_flow(); cout << "Flow done" << endl;
		reset_lakes();
		flood(); cout << "Flooding done" << endl;
		clean_up_lakes();
		erode(); cout << "Erosion done" << endl;

		}

	iteration++;
	if (iteration == 100) {iteration = 0; cout << "100 iterations done..." << endl;}
	time+= timestep;
	}

cout << "Terrain evolution finished." << endl;
plot_surface_data_gnu(1, plotfile);
plot_surface_data_gnu(2, "flow.dat");
plot_surface_data_gnu(3, "lakes.dat");
plot_surface_data_gnu(4, "inflow.dat");
plot_surface_data_gnu(5, "outflow.dat");

}


void TerrainRelief::plot_surface_data_gnu (int index, string filename)
{

int xsize, ysize;
double x, y;

xsize = res_lat;
ysize = res_lon;



ofstream resfile (filename.c_str());
  if (resfile.is_open())
  {
  

for(int j=0; j<ysize; j++)
	{
	for (int i = 0; i< xsize; i=i+1)
		{
		x = i * base_scale;
		y =  j * base_scale;
		
		//x = i;
		//y = j;

		resfile << y << " " << x << " ";

		if (index == 1) {resfile << terrain_elements[i][j].get_elevation() << endl;}
		else if (index == 2) {resfile << terrain_elements[i][j].get_flow() << endl;}
		else if (index == 3) {resfile << terrain_elements[i][j].get_water_elevation() << endl;}
		else if (index == 4) {resfile << terrain_elements[i][j].get_inflow() << endl;}
		else if (index == 5) {resfile << terrain_elements[i][j].get_outflow() << endl;}
		
		}
	resfile << " " << endl;
	}


    resfile.close();
    cout << "Surface plot written to " << filename << "." << endl;

  }
 else cout << "Unable to open file " << "surface_plot.mtv" << " !" << endl;

}
