// terrain_elements.cxx -- routines to assign and compute terrain relief surface elements
//
// Written by Thorsten Renk, started 2025
//
// Copyright (C) 2025  Thorsten Renk - thorsten@science-and-fiction.org 
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301

#include <stdlib.h>
#include <math.h>

#include "terrain_elements.hxx"

using namespace std;

TerrainElement::TerrainElement()
{
elevation = 0.0;
rainfall = 0.0;
flow = 0.0;
water_level = 0.0;
inflow = 0.0;
outflow = 0.0;
gradient = 0.0;

grad_x = 0;
grad_y = 0;

fill_id = -1;

is_sink = false;
is_lake = false;
is_minimum = false;
}


void TerrainElement::reset_flow()
{
flow = 0.0;
grad_x = 0;
grad_y = 0;
barrier_height = 0.0;
}

void TerrainElement::add_to_flow(double in)
{
flow += in;
}

void TerrainElement::add_to_inflow(double in)
{
if (!is_sink) {inflow += in;}
}

void TerrainElement::add_to_water_level(double amount)
{
if (is_lake)
	{water_level+= amount;}
}

void TerrainElement::set_water_level(double level)
{
water_level = level;
}


void TerrainElement::set_water_to_level(double level)
{
if (elevation  < level)
	{
	water_level = level - elevation;
	is_lake = true;
	}
else
	{
	water_level = 0.0;
	is_lake = false;
	}

}

void TerrainElement::set_elevation(double elev)
{
elevation = elev;
}

void TerrainElement::set_rainfall(double rain)
{
rainfall = rain;
}

void TerrainElement::set_gradient(double grad)
{
gradient = grad;
}

void TerrainElement::set_barrier(double bar)
{
barrier_height = bar;
}

void TerrainElement::set_outflow(double flow)
{
outflow = flow;
}

void TerrainElement::set_inflow(double flow)
{
if (is_sink) {inflow = 0.0;} else {inflow = flow;}

}

void TerrainElement::set_flow_dir(int x, int y)
{
grad_x = x;
grad_y = y;
}

void TerrainElement::set_fill_id(int id)
{
fill_id = id;
}

void TerrainElement::set_sink()
{
is_sink = true;
}

void TerrainElement::set_lake()
{
is_lake = true;
}

void TerrainElement::set_minimum()
{
is_minimum = true;
}

void TerrainElement::unset_lake()
{
is_lake = false;
}

void TerrainElement::unset_minimum()
{
is_minimum = false;
}

double TerrainElement::get_elevation()
{
return elevation;
}

double TerrainElement::get_water_elevation()
{
return elevation + water_level;
}

double TerrainElement::get_flow()
{
return flow;
}

double TerrainElement::get_water_level()
{
return water_level;
}

double TerrainElement::get_rainfall()
{
return rainfall;
}

double TerrainElement::get_gradient()
{
return gradient;
}

double TerrainElement::get_barrier()
{
return barrier_height;
}

double TerrainElement::get_outflow()
{
return outflow;
}

double TerrainElement::get_inflow()
{
return inflow;
}

int TerrainElement::get_grad_x()
{
return grad_x;
}

int TerrainElement::get_grad_y()
{
return grad_y;
}

int TerrainElement::get_fill_id()
{
return fill_id;
}


bool TerrainElement::get_terminating()
{
bool flag;

if ((is_lake) || (is_sink)||(is_minimum)) {flag = true;} else {flag = false;}

return flag;
}

bool TerrainElement::get_sink()
{
return is_sink;
}

bool TerrainElement::get_lake()
{
return is_lake;
}

bool TerrainElement::get_minimum()
{
return is_minimum;
}

bool TerrainElement::check_fill_id(int id)
{
if (fill_id == id) {return true;}
else {return false;}
}
