// surface.hxx -- class to instance one surface element of a planet
//
// Written by Thorsten Renk, started 2018
//
// Copyright (C) 2018  Thorsten Renk - thorsten@science-and-fiction.org 
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301


class Surface {
	  double lat;
	  double lon;
	  double area;
	  double albedo;
	  double effective_ground_albedo;
	  double effective_total_albedo;
	  double temperature;
	  double radiative_flux;
	  double internal_flux;
	  double C_V;
	  double V_diurnal;
	  double E_stored;
	  double E_in;
	  double E_out;
	  double E_rad_total;
	  double E_transport;
	  double E_transport_sum;
	  double infrared_blocking;
	  double solar_elevation_angle;
	  double companion_elevation_angle;
	  double binary_elevation_angle;
	  double timestep;

	  int albedo_dynamical;
	  int transport_counter;

	  double forest_fire_cloud_max;
	  double forest_fire_T_min;
	  double forest_fire_T_max;
	  double fire_smoke_level;
	  double fire_highlevel_smoke;
	  double fire_factor;

	  bool compute_weather;
	  bool compute_storms;
	  bool compute_fires;

	  double convective_energy;
	  double convective_potential;
	  double convective_energy_bias;
	  double convective_cloudcover;
	  double convection_factor;
	  double convective_precipitation_threshold;
	  double convective_precipitation_factor;
	  double convective_reservoir_fraction;
	  double convective_reservoir_fraction_max;
	  double synoptic_cloudcover;
	  double local_convection_factor;
	  double low_cloudcover;
	  double mid_cloudcover;
	  double high_cloudcover;
	  double storm_cloudcover;
	  double reflective_cloudcover;
	  double cloud_albedo;
	  double low_cloud_IR_blocking;
	  double mid_cloud_IR_blocking;
	  double high_cloud_IR_blocking;

	  double convective_cloud_rn;
	  double convective_precipitation_rn;

	  bool liquid_flag;
	  double ice_factor;
	  double ice_thickness;
	  double freezing_point;
	  double ice_buildup_factor;
	  double albedo_ice;
	  double albedo_snow;
	  double current_precipitation;
	  double snow_thickness;

	  void compute_convection();
	  void compute_freezing();
	  void compute_snowcover();

	  void set_initial_parameters();

	  double p_sample_tri(double, double, double);


	public:
	  double transport_factor_lat;
	  double transport_factor_lon;
	  Surface (double, double, double, double, double);
	  Surface ();
	  void list_coords(void);
	  void list_instantaneous_temperature();
	  double instantaneous_eq_temperature ();

	  void set_radiative_flux (double set_flux) {radiative_flux = set_flux;}
	  void set_internal_flux (double set_flux) {internal_flux = set_flux;}
	  void set_albedo (double); 
	  void set_albedo_dynamical (int); 
	  void set_thermal_properties (double, double, double);
	  void set_forest_fire(double, double, double);
	  void set_fire_factor(double);
	  void set_atmosphere(double set_IR_blocking) {infrared_blocking = set_IR_blocking;}
	  void set_elevation_angle (double, int);
	  void set_total_energy (double energy) {E_stored = energy / area;}
	  void set_transport_energy (double energy);
	  void set_weather (bool flag) {compute_weather = flag;}
	  void set_synoptic_cloudcover(double);
	  void set_midlevel_cloudcover(double);
	  void set_highlevel_cloudcover(double);
	  void set_storm_cloudcover(double);
	  void set_highlevel_smoke(double);
	  void set_precipitation(double);
	  void set_convection (double);
	  void set_local_convection(double);
	  void set_weather_rn (double val1, double val2) {convective_cloud_rn = val1; convective_precipitation_rn = val2;}
	  void set_liquid (bool flag) {liquid_flag = flag;}
	  void set_parameter(std::string, double);
	  void set_ice_buildup(double value) {ice_buildup_factor = value;}

	  void evolve_temperature(double);
	  void use_convective_energy(double);

	  double get_lat(void) {return lat;}
	  double get_lon(void) {return lon;}
	  double get_area(void) {return area;}
	  double get_albedo (void) {return albedo;}
	  double get_ground_albedo (void) {return effective_ground_albedo;}
	  double get_total_albedo (void) {return effective_total_albedo;}
	  int get_albedo_dynamical (void);
	  double get_temperature (void) {return temperature;}
	  double get_C_V (void) {return C_V;}
	  double get_V_diurnal (void) {return V_diurnal;}
	  double get_transport_lat (void) {return transport_factor_lat;}
	  double get_transport_lon (void) {return transport_factor_lon;}
	  double get_radiative_flux(void) {return radiative_flux;}
	  double get_total_energy (void) {return E_stored * area;}
	  double get_absorbed_energy (void) {return E_in * area;}
	  double get_emitted_energy (void) {return E_out * area;}
	  double get_transported_energy (void) {return E_transport_sum / transport_counter * area;}
	  double get_absorbed_energy_flux (void) {return E_in/timestep;}
	  double get_emitted_energy_flux (void) {return E_out/timestep;}
	  double get_transported_energy_flux (void) {return E_transport_sum/(timestep * transport_counter);}
	  double get_current_transport_energy_flux (void) {return E_transport/timestep;}
	  double get_reflected_energy_flux (void) {return radiative_flux * albedo;}
	  double get_total_irradiation (void) {return E_rad_total;}
	  double get_smoke_level(void) {return fire_smoke_level;}
	  double get_solar_elevation_angle (int);
	  double get_cloudcover(void) {return reflective_cloudcover;}
	  double get_high_cloudcover(void) {return high_cloudcover;}
	  double get_low_cloudcover(void) {return low_cloudcover;}
	  double get_mid_cloudcover(void) {return mid_cloudcover;}
	  double get_storm_cloudcover(void){return storm_cloudcover;}
	  double get_ice_thickness(void) {return ice_thickness;}
	  double get_snow_thickness(void) {return snow_thickness;}
	  double get_precipitation(void) {return current_precipitation;}
	  double get_convective_energy(void) {return convective_energy + convective_potential;}
	  bool get_liquid (void) {return liquid_flag;}


};
