// surface.cxx -- class to instance one surface element of a planet
//
// Written by Thorsten Renk, started 2018
//
// Copyright (C) 2018  Thorsten Renk - thorsten@science-and-fiction.org 
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301

#include <iostream>
#include <stdlib.h>
#include <math.h>
#include <iomanip>

#include "constants.hxx"
#include "surface.hxx"
#include "atmosphere.hxx"

using namespace std;

Surface::Surface (double latitude, double longitude, double size_lat_in, double size_lon_in, double radius)
{
double size_lat = size_lat_in * deg_to_rad;
double size_lon = size_lon_in * deg_to_rad;


lat = latitude * deg_to_rad;
lon = longitude * deg_to_rad;

double lat_to_m = 2.0 * radius * pi / 360.0;

double lon_to_m_north =  cos((lat + 0.5 * size_lat) ) * lat_to_m;
double lon_to_m_south =  cos((lat - 0.5 * size_lat) ) * lat_to_m;




double h = lat_to_m * size_lat * rad_to_deg;
double a = lon_to_m_north * size_lon * rad_to_deg;
double b = lon_to_m_south * size_lon * rad_to_deg;

transport_factor_lon = lat_to_m;
transport_factor_lat = lon_to_m_south;
if (lon_to_m_north > lon_to_m_south) {transport_factor_lat = lon_to_m_north;}

area = 0.5 * (a + b) * h;
albedo = 0.3;

set_initial_parameters();
}

Surface::Surface ()
{
lat = 0;
lon = 0;
area = 0;
albedo = 0;

set_initial_parameters();
}

void Surface::set_initial_parameters()
{
E_rad_total = 0.0;
albedo_dynamical = 0;
E_transport = 0.0;
E_transport_sum = 0.0;

transport_counter = 0;
compute_weather = false;
compute_fires = false;
compute_storms = false;
convective_energy = 0.0;
convective_cloudcover = 0.0;
convection_factor = 1.0;
convective_potential = 0.0;
local_convection_factor = 1.0;
convective_precipitation_threshold = 0.4;
convective_precipitation_factor = 1.0;
convective_reservoir_fraction = 0.0;
convective_reservoir_fraction_max = 0.5;
synoptic_cloudcover = 0.0;
low_cloudcover = 0.0;
mid_cloudcover = 0.0;
high_cloudcover = 0.0;
storm_cloudcover = 0.0;
reflective_cloudcover = 0.0;
low_cloud_IR_blocking = 0.6;
mid_cloud_IR_blocking = 0.3;
high_cloud_IR_blocking = 0.1;
cloud_albedo = 0.8;
convective_energy_bias = 1.0;
liquid_flag = false;
ice_factor = 0.0;
ice_thickness = 0.0;
freezing_point = 273.15;
albedo_ice = 0.6;
albedo_snow = 0.8;
snow_thickness = 0.0;
current_precipitation = 0.0;
ice_buildup_factor = 1.0;
forest_fire_cloud_max = 0.0;
forest_fire_T_min = 0.0;
forest_fire_T_max = 0.0;
fire_smoke_level = 0.0;
fire_highlevel_smoke = 0.0;
convective_cloud_rn = 0.5;
convective_precipitation_rn = 0.5;
}



double Surface::p_sample_tri(double center, double width, double rn)
{
double res;

if (rn > 0.5)
	{
	rn -= 0.5;
	res = 2.0 - sqrt(1.0 - 2.0 * rn);
	}
else	
	{
	res = sqrt(2.0 * rn);
	}
res -= 1.0;
res *=width;
res += center;

return res;
}


double Surface::instantaneous_eq_temperature ()
{
double I_in = radiative_flux * (1.0 - albedo) + internal_flux;

return pow(I_in  / sigma_SB, 0.25);
}



void Surface::set_thermal_properties (double C_V_set, double V_diurnal_set, double T)
{
C_V = C_V_set * 1000.0 * 2600;
V_diurnal = V_diurnal_set;
temperature = T;

E_stored = C_V * V_diurnal * temperature;



convective_reservoir_fraction = C_V_set * V_diurnal* 0.12;

if (convective_reservoir_fraction > convective_reservoir_fraction_max) 
	{
	convective_reservoir_fraction = convective_reservoir_fraction_max;
	}


}

void Surface::set_forest_fire(double cloud_max, double T_min, double T_max)
{
if (compute_weather == false) {cout << "Forest fires need weather simulation to be computed." << endl; return;}

forest_fire_cloud_max = cloud_max;
forest_fire_T_min = T_min;
forest_fire_T_max = T_max;

compute_fires = true;
//cout << "Forest fire simulation on" << endl;
}

void Surface::set_transport_energy(double energy)
{
E_transport = energy/ area; 
E_transport_sum += E_transport; 
transport_counter++;
}

void Surface::set_albedo(double albedo_in)
{
albedo = albedo_in;
//cout << albedo << endl;
}

void Surface::set_fire_factor(double value)
{
fire_factor = value;
//cout << "Fire factor: " << fire_factor << endl;
}

void Surface::set_synoptic_cloudcover(double value)
{
synoptic_cloudcover = value;
}

void Surface::set_midlevel_cloudcover(double value)
{
double T_eff;

mid_cloudcover = value;
if (compute_fires)
	{
	if (temperature > forest_fire_T_min)
		{
		T_eff = temperature;
		if (T_eff > forest_fire_T_max) {T_eff = forest_fire_T_max;}
		fire_smoke_level = forest_fire_cloud_max * fire_factor * (T_eff - forest_fire_T_min)/(forest_fire_T_max - forest_fire_T_min);
		mid_cloudcover += fire_smoke_level;
		if (mid_cloudcover > 1.0) {mid_cloudcover = 1.0;}
		}
	}
}

void Surface::set_highlevel_cloudcover(double value)
{
high_cloudcover = value + fire_highlevel_smoke;
if (high_cloudcover > 1.0) {high_cloudcover = 1.0;}
}

void Surface::set_storm_cloudcover(double value)
{
storm_cloudcover = value;
}

void Surface::set_highlevel_smoke(double value)
{
fire_highlevel_smoke = value;
}

void Surface::set_precipitation(double value)
{
current_precipitation = value;
}


void Surface::set_albedo_dynamical(int flag)
{
albedo_dynamical = flag;
}

void Surface::set_convection(double ce_bias)
{
convective_energy_bias = ce_bias; 
}

void Surface::set_local_convection(double value)
{
local_convection_factor = value;
}


void Surface::set_parameter(std::string name, double value)
{

if (name == "snow_albedo")
	{
	albedo_snow = value; 
	}
else if (name == "precipitation_threshold")
	{
	convective_precipitation_threshold = value;
	}
else if (name == "precipitation_factor")
	{
	convective_precipitation_factor = value;
	}
else if (name == "convection_factor")
	{
	convection_factor = value;
	}
else if (name == "storms")
	{
	if (value == 1.0) {compute_storms = true;} else {compute_storms = false;}
	}


}

void Surface::use_convective_energy(double energy)
{
convective_potential -= energy;
//cout << "Using energy..." << endl;
if (convective_potential < 0.0) {convective_potential = 0.0;}

}

void Surface::evolve_temperature (double timestep_in)
{
double IR_open_fraction;
double ice_fraction, snow_fraction;


timestep = timestep_in;

ice_fraction = ice_thickness/100.0;
if (ice_fraction > 1.0) {ice_fraction = 1.0;}
effective_ground_albedo = (1.0 - ice_fraction) * albedo + ice_fraction * albedo_ice;


snow_fraction = snow_thickness / 10.0;
if (snow_fraction > 1.0) {snow_fraction = 1.0;}
effective_ground_albedo = (1.0 - snow_fraction) * albedo + snow_fraction * albedo_snow;


reflective_cloudcover = 1.0 - ((1.0 - low_cloudcover) * (1.0 - mid_cloudcover) * (1.0 - 0.5 * high_cloudcover));
IR_open_fraction = 1.0 - (high_cloudcover * high_cloud_IR_blocking + mid_cloud_IR_blocking * mid_cloudcover + low_cloud_IR_blocking * low_cloudcover); 

effective_total_albedo = (1.0 - reflective_cloudcover) * effective_ground_albedo + reflective_cloudcover * cloud_albedo;

E_in = (radiative_flux * (1.0 - ((1.0 - reflective_cloudcover) * effective_ground_albedo + reflective_cloudcover * cloud_albedo)) + internal_flux) * timestep;
E_out = sigma_SB * pow(temperature, 4.0) * timestep * IR_open_fraction * (1.0 -  infrared_blocking );


E_rad_total += E_in;
E_stored = E_stored + E_in - E_out;

temperature = E_stored / (C_V * V_diurnal);

if (compute_weather)
	{
	compute_convection();
	if (liquid_flag) {compute_freezing();}
	compute_snowcover();	
	}

}



void Surface::compute_convection()
{

double convection_factor_actual, cloudcover_factor_actual;



convection_factor_actual = p_sample_tri(convection_factor * local_convection_factor, 0.7 * convection_factor * local_convection_factor, convective_cloud_rn);



if (compute_storms)
	{
	convective_energy+= (1.0 - convective_reservoir_fraction) * E_in * convective_energy_bias;
	convective_potential+= convective_reservoir_fraction * E_in * convective_energy_bias;
	}
else
	{
	convective_energy+= E_in * convective_energy_bias;
	}



convective_cloudcover = 1000.0/ timestep * convection_factor_actual * convective_energy / 8e6;
if (convective_cloudcover > 1.0) {convective_cloudcover = 1.0;}

cloudcover_factor_actual = p_sample_tri(convective_cloudcover, 0.35, convective_precipitation_rn);
if (cloudcover_factor_actual < 0.0) {cloudcover_factor_actual = 0.0;}

if (cloudcover_factor_actual > convective_precipitation_threshold)
	{
	current_precipitation += (cloudcover_factor_actual - convective_precipitation_threshold) * convective_precipitation_factor * 0.005;
	}

low_cloudcover = convective_cloudcover + synoptic_cloudcover;
if (low_cloudcover > 1.0) {low_cloudcover = 1.0;}
else if (low_cloudcover < 0.0) {low_cloudcover = 0.0;}


if (compute_storms)
	{
	convective_potential -= 120.0 / (convection_factor * local_convection_factor) * timestep;
	if (convective_potential < 0.0) {convective_potential = 0.0;}
	}

convective_energy -= ((convective_cloudcover  * 1200.0) + 150.0) / (convection_factor * local_convection_factor) * timestep;
	

if (convective_energy < 0.0) {convective_energy = 0.0;}

}


void Surface::compute_freezing()
{
double delta_T, penalty;

if (temperature < freezing_point - 10.0)
	{
	delta_T = (freezing_point - 10.0) - temperature;
	ice_factor += 1e-6 * timestep * (1.0 + delta_T/10.0);
	

	if (ice_factor > 1.0)
		{
		ice_factor = 1.0;

		penalty = 50.0/ice_thickness * ice_buildup_factor;
		if (penalty > 1.0) {penalty = 1.0;}
		ice_thickness += 1.5e-5 * penalty * timestep * (1.0 + delta_T/10.0);
		}

	}

if (temperature > freezing_point + 5.0)
	{
	delta_T = temperature -  (freezing_point + 5.0) ;
	ice_thickness -= 2.5e-5 * timestep * (1.0 + delta_T/10.0);


	if (ice_thickness < 0.0)	
		{
		ice_thickness = 0.0;
		ice_factor -= 1e-6 * timestep * (1.0 + delta_T/10.0);
		if (ice_factor < 0.0){ice_factor = 0.0;}
		}

	}



}

void Surface::compute_snowcover()
{

if ((liquid_flag == false) || ((ice_factor == 1.0) && (ice_thickness > 1.0)))
	{

	if (temperature < freezing_point)
		{snow_thickness += current_precipitation * timestep;}
	else
		{snow_thickness -= 1e-5 * timestep * (temperature-freezing_point);}
	}

if ((liquid_flag == true) && (ice_thickness < 1.0))
	{
	snow_thickness = 0.0;
	}

if (snow_thickness < 0.0) {snow_thickness = 0.0;}

}


void Surface::list_coords (void)
{
cout << "(" << setw(5) << lat * rad_to_deg << ", " << setw(3) << lon * rad_to_deg << ")  ";
}

void Surface::list_instantaneous_temperature ()
{
double T = instantaneous_eq_temperature ();

cout << setw(5) << setiosflags(ios::fixed) << setprecision(1) << T << " ";
}

void Surface::set_elevation_angle(double angle_sin, int body_index)
{
if (body_index == 0)
	{
	solar_elevation_angle = asin(angle_sin);
	}
else if (body_index == 1)
	{
	companion_elevation_angle = asin(angle_sin);
	}
else if (body_index == 2)
	{
	binary_elevation_angle = asin(angle_sin);
	}


}

double Surface::get_solar_elevation_angle(int body_index)
{
//if (body_index != 0.0) {cout << body_index << endl;}

if (body_index == 0)
	{
	return solar_elevation_angle;
	}
else if (body_index == 1)
	{
	return companion_elevation_angle;
	}
else if (body_index == 2)
	{
	return binary_elevation_angle;
	}

return 0.0;
}

int Surface::get_albedo_dynamical()
{
//cout << albedo_dynamical << endl;
return albedo_dynamical;
}
