// star.cxx -- routines to assign and compute stellar properties
//
// Written by Thorsten Renk, started 2018
//
// Copyright (C) 2018  Thorsten Renk - thorsten@science-and-fiction.org 
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301


#include <iostream>
#include <fstream>
#include <string>
#include <stdlib.h>
#include <math.h>

#include "constants.hxx"
#include "star.hxx"

using namespace std;


Star::Star (double M, double T)
{
mass_solar = M;
mass = mass_solar * M_solar;
temperature = T;
luminosity_solar = compute_luminosity(M);

compute_spectral_fractions();
}

Star::Star ()
{
mass_solar = 1.0;
mass = mass_solar * M_solar;
temperature = 5700.0;
luminosity_solar = compute_luminosity(mass_solar);
compute_spectral_fractions();
}

void Star::assign_properties (double M, double T)
{
mass_solar = M;
mass = mass_solar * M_solar;
temperature = T;
luminosity_solar = compute_luminosity(M);
luminosity = luminosity_solar * L_solar;
radius = sqrt(luminosity/(sigma_SB * pow(T, 4.0) * 4.0 * pi) );
density = mass / (4.0/3.0 * pi * pow(radius,3.0));
compute_spectral_fractions();
}

void Star::assign_properties (double M, double T, double L, double R)
{
mass_solar = M;
mass = mass_solar * M_solar;
temperature = T;
luminosity_solar = L;
luminosity = luminosity_solar * L_solar;
radius = R_solar * R;
density = mass / (4.0/3.0 * pi * pow(radius,3.0));
compute_spectral_fractions();
}

void Star::set_xyz (double set_x, double set_y, double set_z)
{
x = set_x; y = set_y; z = set_z;
}

double Star::compute_luminosity (double M)
{

double L;

if (M < 0.43) {L = 0.23 * pow(M, 2.3);}
else if ((M >= 0.43) && (M < 2.0)) {L = pow(M, 4.0);}
else if ((M >= 2.0) && (M < 20.0)) { L = 1.5 * pow(M, 3.5);}
else if (M>=20.0) {L = 3200.0 * M;}
else {cout << "No luminosity for " << M << " solar masses." << endl; L=0;}

//cout << "Luminosity: " << L << " L_sun" << endl;

return L;
}


double Star::spectrum (double lambda)
{
double nu;

nu = c_light/lambda;
return pi * nu/lambda * 2.0 * h_planck * pow(nu,3.0) / pow(c_light,2.0) * 1.0/(exp((h_planck * nu)/(k_B * temperature)) -1.0);
}

void Star::list_properties ()
{
cout << endl;
if (designation_set)
	{cout << designation << endl;}
else
	{cout << "Central star" << endl;}

cout << "------------" << endl;
cout << "Mass [m_sun]:          " << mass_solar << endl;
cout << "Surface T [K]:         " << temperature << endl;
cout << "Luminosity [L_sun]:    " << luminosity_solar << endl;
cout << "Radius [Mkm]:          " << radius / 1e9 << endl;
cout << "Mean density [g/cm^3]: " << density /1000.0 << endl;
cout << "Spectral fraction IR   " << ir_fraction << endl;
cout << "Spectral fraction vis  " << vis_fraction << endl;
cout << "Spectral fraction UV   " << uv_fraction << endl;
}


void Star::compute_spectral_fractions()
{
double lambda;
double B_IR, B_vis, B_UV, B_norm;

B_IR = 0.0;
B_vis = 0.0;
B_UV = 0.0;

for (int i=1; i< 1000; i++)
	{
	lambda = i * 5e-9;
	//cout << lambda << " " << spectrum(lambda) << endl;
	if (lambda < 4e-7)
		{
		B_UV += spectrum (lambda);
		}
	else if (lambda < 7e-7)
		{
		B_vis += spectrum (lambda);
		}
	else
		{
		B_IR += spectrum (lambda);
		}

	}

B_norm = B_IR + B_vis + B_UV;
ir_fraction = B_IR /B_norm;
vis_fraction = B_vis / B_norm;
uv_fraction = B_UV/B_norm;

}
