// planet.hxx -- routines to assign and compute planetary properties
//
// Written by Thorsten Renk, started 2018
//
// Copyright (C) 2018  Thorsten Renk - thorsten@science-and-fiction.org 
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301


#include "surface.hxx"
#include "atmosphere.hxx"
#include "weather.hxx"
#include "geology.hxx"

class Material;


class Planet {
	  int n_stars;
	  int resolution_lat;
	  int resolution_lon;
	  int atmosphere_heating_profile_size;

	  bool binary_defined;
	  bool companion_defined;
	  bool lagrange_orbit;
	  bool lagrange_leading;
	  double mass_earth;
	  double mass;
	  double radius_earth;
	  double radius;
	  double density;
	  double surface_gravity;
	  double semimajor;
	  double eccentricity;
	  double apoapsis;
	  double periapsis;
	  double period;
	  double distance_to_star;
	  double semimajor_longyear;
	  double eccentricity_longyear;
	  double apoapsis_longyear;
	  double periapsis_longyear;
	  double period_longyear;
	  double speed_periapsis_longyear;
	  double speed_apoapsis_longyear;
	  double inclination_longyear;
	  double lan_longyear;
	  double distance_to_companion;
	  double semimajor_binary;
	  double eccentricity_binary;
	  double apoapsis_binary;
	  double periapsis_binary;
	  double period_binary;
	  double speed_periapsis_binary;
	  double speed_apoapsis_binary;
	  double inclination_binary;
	  double lan_binary;
	  double inclination_planet;
	  double lan_planet;
	  double distance_to_binary;
	  double inclination;
	  double inclination_axis_companion;
 	  double rotation_period;
	  double sidereal_rotation_period;
	  double speed_periapsis;
	  double speed_apoapsis;
	  double solar_constant_min[5];
	  double solar_constant_max[5];
	  double solar_constant_current[5];
	  double planetary_constant_current;
	  double internal_heat_flux;
	  double dayside_flux;
	  double nightside_flux;
	  double reflected_flux;
	  double albedo;
	  double CV;
	  double diurnal_depth;
	  double max_diurnal_depth;
	  double x;
	  double y;
	  double z;
	  double vx;
	  double vy;
	  double vz;
	  double solar_declination;
	  double solar_declination_offset;
	  double companion_declination;
	  double companion_declination_offset;
	  double binary_declination;
	  double binary_declination_offset;
	  double day_fraction;
	  double year_fraction;
	  double day_correction_angle_star;
	  double day_correction_angle_binary;
	  double elevation_correction_angle_star;
	  double elevation_correction_angle_binary;
	  double longyear_fraction;
	  double longday_fraction;
	  double temperature;
	  double T_init;
	  double T_rad_av;
	  double T_rad_av_periapsis;
	  double T_rad_av_apoapsis;

	  double Ldphi;
	  double LdR;
	  double Ldz;
	  double LdvT;
	  double LdvR;
	  double Ldvz;
	
	  double hydrosphere_transport_coefficient;
	  double hydrosphere_ice_factor;
	  double ridge_migration_factor;
	  double smoke_stratosphere_transfer_coefficient;


	  bool weather_defined;
	  bool geology_defined;
	  bool geology_depth_profile;
	  bool compute_smoke_transfer;
	  bool compute_storms;

	  Surface** surface_elements;
	  double** DE_array;
	  double** atmosphere_heating_profile;
	  double** atmosphere_o3_profile;

	  std::string designation;
	  std::string lagrange_point;

	  void compute_average_albedo();
	  void compute_thermal_flux();
	  void compute_reflected_flux();
	  double albedo_water(double);
	  double search_ce_maximum(int, int);

	public:

	  Atmosphere atmosphere;
	  Weather weather;
	  Geology geo;

	  //constructor
	  Planet ();
	  
	  //general property assignment
	  void assign_properties(double, double);
	  void assign_rotation (double, double, double);
	  void assign_companion_dec_offset(double offset) {companion_declination_offset = offset;}
	  void assign_companion_tilt_axis(double tilt) {inclination_axis_companion = tilt;}
	  void assign_orbit (double, double, double);
	  void assign_orbit_longyear (double, double, double, double);
	  void assign_orbit_lagrange(bool);
	  void assign_lagrange_corrections(double, double, double, double, double, double);
	  void assign_orbit_binary (double, double, double, double);
	  void tilt_orbit_longyear(double, double);
	  void tilt_orbit_binary(double, double);
	  void tilt_orbit_planet(double, double);
	  void register_orbit_binary();
	  void register_orbit_companion();
	  void assign_atmosphere_basic(double, double);
	  void assign_hydrosphere_basic(double, double);
	  void assign_materials_map(std::string file, Material materials[]);
	  void create_surface_elements(int, int);
	  void assign_name (std::string set_name) {designation = set_name; }
	  void set_albedo (double a) {albedo = a;}
	  void set_thermal_properties (double c, double d) {CV = c; diurnal_depth = d;} 
	  void set_max_diurnal_depth (double depth) {max_diurnal_depth = depth;}
	  void set_T_init (double T) {T_init = T;}
	  void init_weather(int, int, double, double, double, double, double);
	  void init_geology(double, double, double, double, std::string, std::string, std::string);
	  void init_geology(double, double, double, std::string, std::string);
	  void init_geology(double, double, double, std::string);
	  void init_geology_undifferentiated(double, double, double, std::string, std::string);
	  void geology_override_density(double, std::string);

	  void set_weather_par(std::string, double);
	  void set_geo_par(std::string, double);
	  void init_forest_fire_simulation(double, double, double);

	  //generic properties
	  double get_mass(void) {return mass;}
	  double get_radius(void) {return radius;}
	  double get_surface_gravity(void) {return surface_gravity;}

	  //updates for orbital position changes
	  void set_xyz (double, double, double);
	  void set_distance_to_star(double, double );
	  void set_distance_to_companion(double, double );
	  void set_distance_to_binary(double, double );
	  void set_day_correction_angle_star(double);
	  void set_day_correction_angle_binary(double);
	  void set_elevation_correction_angle_star(double);
	  void set_elevation_correction_angle_binary(double);
	  void set_internal_heat (double set_flux) {internal_heat_flux = set_flux;}
	  void set_solar_constants (double, double);
	  void set_solar_constants (double, double, int);
	  void set_solar_constant (double s) {solar_constant_current[0] = s;}
	  void set_solar_constant (double s, int index) {solar_constant_current[index] = s;}
	  void set_planetary_constant (double p) {planetary_constant_current = p;}
	  void set_surface_irradiation ();
	  void add_indirect_surface_irradiation();
	  void set_sun_angle();
	  void set_year_fraction(double);
	  void set_longyear_fraction(double);

	  std::string get_designation() {return designation;}
	  std::string get_lagrange_point() {return lagrange_point;}


	  //thermal properties
	  void compute_average_thermal_properties ();
	  void compute_IR_retention ();
	  void compute_atmosphere_transport();
	  void compute_average_temperature();
	  void compute_average_cloudcover();

	  double get_average_temperature();
	  double get_average_albedo();
	  double get_average_cloudcover();
	  double get_average_smoke();

	  //rotational state access
	  void set_daytime(double);
	  void set_day_fraction(double);
	  void set_longday_fraction(double);
	  void set_binaryday_fraction(double);

	  double solar_elevation_angle_sin(double, double);
	  double companion_elevation_angle_sin(double, double);
	  double binary_elevation_angle_sin(double, double);

	  double get_year_fraction(void) {return year_fraction;}
	  double get_longyear_fraction () {return longyear_fraction;}

	  //orbital property access
	  bool has_lagrange_orbit(void) {return lagrange_orbit;}	

 	  double get_periapsis(void) {return periapsis;}
	  double get_periapsis_longyear(void) {return periapsis_longyear;}
	  double get_apoapsis(void) {return apoapsis;}
	  double get_semimajor(void) {return semimajor;}
	  double get_eccentricity(void) {return eccentricity;}
	  double get_apoapsis_longyear(void) {return apoapsis_longyear;}
	  double get_speed_periapsis(void) {return speed_periapsis;}
	  double get_speed_periapsis_longyear(void) {return speed_periapsis_longyear;}
	  double get_inc_longyear(void) {return inclination_longyear;}
	  double get_lan_longyear(void) {return lan_longyear;}
	  double get_inc_binary(void) {return inclination_binary;}
	  double get_lan_binary(void) {return lan_binary;}
	  double get_inc_planet(void) {return inclination_planet;}
	  double get_lan_planet(void) {return lan_planet;}
	  double get_speed_apoapsis(void) {return speed_apoapsis;}
	  double get_speed_apoapsis_longyear(void) {return speed_apoapsis_longyear;}
	  double get_periapsis_binary(void) {return periapsis_binary;}
	  double get_apoapsis_binary(void) {return apoapsis_binary;}
	  double get_speed_periapsis_binary(void) {return speed_periapsis_binary;}
	  double get_speed_apoapsis_binary(void) {return speed_apoapsis_binary;}
	  double get_period (void) {return period;}
	  double get_rotation_period (void) {return rotation_period;}
	  double get_sidereal_rotation_period (void) {return sidereal_rotation_period;}
	  double get_day_fraction (void) {return day_fraction;}
 	  double get_solar_declination(void) {return solar_declination;}
 	  double get_solar_declination_companion(void) {return companion_declination;}
 	  double get_solar_declination_binary(void) {return binary_declination;}
	  double get_Ldphi (void) {return Ldphi;} 
	  double get_LdR (void) {return LdR;} 
	  double get_Ldz (void) {return Ldz;} 
	  double get_LdvT (void) {return LdvT;} 
	  double get_LdvR (void) {return LdvR;} 
	  double get_Ldvz (void) {return Ldvz;} 

	  //emissive properties

	  double get_dayside_flux(void) {return dayside_flux + internal_heat_flux;}
	  double get_nightside_flux(void) {return nightside_flux + internal_heat_flux;}
	  double get_reflected_flux(void) {return reflected_flux;}

	  //spectral properties
	  double spectrum (double);
	  double spectrum_atm_filtered (double);

	  //stratosphere and Chapman mechanism
	  double get_atmosphere_uv_heating(double);
	  double get_atmosphere_o3_profile(double);
	  double get_atmosphere_uv_heating_T(double);
	  void compute_atmosphere_uv_heating(double, double, int);
	  void compute_atmosphere_o3_profile(double, double, int);
	  void init_atmosphere_uv_heating(int);
	  void dump_atmosphere_heating_profile(void);

	  // time evolution
	  void evolve_temperature (double);
	  void evolve_temperature_onestep (double);
	  void atmosphere_transport(double);

	  //state lists
	  void list_properties (void);
	  void list_rotation (void);
	  void list_orbit (void);
	  void list_orbit_longyear (void);
	  void list_orbit_binary (void);
	  void list_thermal_properties (void);
	  void list_albedo_dynamical(void);
	  void list_storms(double);

	  //surface element lists
	  void list_surface_elements(int);

	  //surface element access
	  double get_element_temperature(int, int);
	  double get_element_latitude(int, int);
	  double get_element_longitude(int, int);
	  double get_element_solar_angle(int, int, int);
	  double get_element_absorbed_energy(int, int);
	  double get_element_emitted_energy(int, int);
	  double get_element_absorbed_flux(int, int);
	  double get_element_emitted_IR_flux(int, int);
	  double get_element_reflected_flux(int, int);
	  double get_element_transported_flux (int, int);
	  double get_element_cloud_cover (int, int);
	  double get_element_low_cloud_cover (int, int);
	  double get_element_mid_cloud_cover (int, int);
	  double get_element_high_cloud_cover (int, int);
	  double get_element_precipitation (int, int);
	  double get_element_ice_thickness (int, int);
	  double get_element_snow_thickness (int, int);
	  double get_element_convective_energy(int, int);

	  //weather access
	  int get_num_storms(void);
	  int get_storm_category(int);
	  double get_storm_lat(int);
	  double get_storm_lon(int);

	  //geo access
	  double get_T_at_depth(double);
	  double get_p_at_depth(double);
	  std::string get_phase_at_depth(double);


	  //state plots
	  void plot_surface_data_gnu(int, std::string);
	  void plot_surface_data_mtv(int, std::string);
	  void plot_surface_data_mtv(int, std::string, double, double);
	  void plot_diurnal_evolution(double, double, int, int);
};
