// planet.cxx -- routines to assign and compute planetary properties
//
// Written by Thorsten Renk, started 2018
//
// Copyright (C) 2018  Thorsten Renk - thorsten@science-and-fiction.org 
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301


#include <iostream>
#include <stdlib.h>
#include <fstream>
#include <math.h>

#include "planet.hxx"
#include "weather.hxx"
#include "constants.hxx"
#include "material.hxx"
#include "utilities.hxx"

using namespace std;

Planet::Planet ()
{
mass_earth = 1.0;
mass = mass_earth * M_earth;
max_diurnal_depth = 0.0;
T_init = 0.0;
binary_defined = false;
companion_defined = false;
weather_defined = false;
geology_defined = false;
compute_storms = false;
lagrange_orbit = false;
lagrange_leading = false;
geology_depth_profile = false;

day_correction_angle_star = 0.0;
day_correction_angle_binary = 0.0;

elevation_correction_angle_star = 0.0;
elevation_correction_angle_binary = 0.0;

hydrosphere_transport_coefficient = 0.0;
ridge_migration_factor = 0.5;

smoke_stratosphere_transfer_coefficient = 0.0;
compute_smoke_transfer = false;
}

double Planet::spectrum (double lambda)
{
double nu;


nu = c_light/lambda;
return pi * nu/lambda * 2.0 * h_planck * pow(nu,3.0) / pow(c_light,2.0) * 1.0/(exp((h_planck * nu)/(k_B * temperature)) -1.0);
}

double Planet::spectrum_atm_filtered (double lambda)
{
return spectrum(lambda) * atmosphere.get_transmission(lambda);
}



void Planet::assign_properties (double set_mass, double set_radius)
{
mass_earth = set_mass;
mass = mass_earth * M_earth;
radius_earth = set_radius;
radius = radius_earth * R_earth; 

density = mass / (4.0/3.0 * pi * pow(radius, 3.0));

surface_gravity = G * mass / pow(radius, 2.0);
}

void Planet::assign_rotation (double set_period, double inc, double dec_offset)
{
rotation_period = set_period;
sidereal_rotation_period = rotation_period/(1.0 + rotation_period/period);
inclination = inc;
solar_declination_offset = dec_offset;
year_fraction = 0.0;
solar_declination = inclination * sin(2.0 * pi * year_fraction + solar_declination_offset);
}

void Planet::set_year_fraction (double fraction)
{
year_fraction = fraction;
solar_declination = inclination * sin(2.0 * pi * year_fraction + solar_declination_offset);
}

void Planet::set_longyear_fraction (double fraction)
{
longyear_fraction = fraction;
companion_declination = inclination_axis_companion * sin(2.0 * pi * longyear_fraction + companion_declination_offset);
//cout << designation << " " << companion_declination * rad_to_deg << " " << inclination_axis_companion * rad_to_deg << " " << companion_declination_offset * rad_to_deg << endl;
}

void Planet::assign_orbit (double a, double ecc, double M_star)
{

semimajor = a;
eccentricity = ecc;

apoapsis = semimajor * (1.0 + eccentricity);
periapsis = semimajor * (1.0 - eccentricity);

//double M_red = (mass * M_star)/(mass + M_star);

speed_periapsis = sqrt(G * (M_star + mass) * (2.0/periapsis - 1.0/semimajor));
speed_apoapsis = sqrt(G * (M_star + mass) * (2.0/apoapsis - 1.0/semimajor));

period = 2.0 * pi * sqrt(pow(semimajor, 3.0) / (G * (M_star + mass)));

n_stars += 1;

}

void Planet::assign_orbit_longyear (double a, double ecc, double M_companion, double M_star)
{

semimajor_longyear = a;
eccentricity_longyear = ecc;

apoapsis_longyear = semimajor_longyear * (1.0 + eccentricity_longyear);
periapsis_longyear = semimajor_longyear * (1.0 - eccentricity_longyear);


speed_periapsis_longyear = sqrt(G * (M_companion + M_star + mass) * (2.0/periapsis_longyear - 1.0/semimajor_longyear));
speed_apoapsis_longyear = sqrt(G * (M_companion + M_star + mass) * (2.0/apoapsis_longyear - 1.0/semimajor_longyear));

period_longyear = 2.0 * pi * sqrt(pow(semimajor_longyear, 3.0) / (G * (M_star + M_companion)));


companion_defined = true;


n_stars += 1;
}

void Planet::assign_orbit_lagrange(bool is_leading)
{
semimajor = semimajor_longyear;
eccentricity = eccentricity_longyear;

apoapsis = apoapsis_longyear;
periapsis = periapsis_longyear;

speed_apoapsis = speed_apoapsis_longyear;
speed_periapsis = speed_periapsis_longyear;

period = period_longyear;

inclination_planet = inclination_longyear;
lan_planet = lan_longyear;

lagrange_orbit = true;
if (is_leading)
	{
	lagrange_leading = true;
	lagrange_point = "L4";
	}
else
	{
	lagrange_leading = false;
	lagrange_point = "L5";
	}

n_stars += 1;
}

void Planet::assign_lagrange_corrections(double p1, double p2, double p3, double p4, double p5, double p6)
{
Ldphi = p1;
LdR = p2;
Ldz = p3;
LdvT = p4;
LdvR = p5;
Ldvz = p6;
}

void Planet::tilt_orbit_planet(double inc, double lan)
{
inclination_planet = inc;
lan_planet = lan;
}


void Planet::tilt_orbit_longyear (double inc, double lan)
{
inclination_longyear = inc;
lan_longyear = lan;
}

void Planet::assign_orbit_binary (double a, double ecc, double M_binary, double M_star)
{

semimajor_binary = a;
eccentricity_binary = ecc;

apoapsis_binary = semimajor_binary * (1.0 + eccentricity_binary);
periapsis_binary = semimajor_binary * (1.0 - eccentricity_binary);


speed_periapsis_binary = sqrt(G * (M_binary + M_star) * (2.0/periapsis_binary - 1.0/semimajor_binary));
speed_apoapsis_binary = sqrt(G * (M_binary + M_star) * (2.0/apoapsis_binary - 1.0/semimajor_binary));

period_binary = 2.0 * pi * sqrt(pow(semimajor_binary, 3.0) / (G * (M_star + M_binary)));

binary_defined = true;

n_stars += 1;

}


void Planet::tilt_orbit_binary (double inc, double lan)
{
inclination_binary = inc;
lan_binary = lan;
}



void Planet::register_orbit_binary () // needs to be done for a moon which doesn't manage the orbit but still does illumination
{

binary_defined = true;
n_stars +=1;
}

void Planet::register_orbit_companion() // needs to be done for a moon which doesn't manage the orbit but still does illumination
{

companion_defined = true;
n_stars +=1;
}

void Planet::assign_atmosphere_basic(double set_infrared_blocking, double set_transport_coefficient)
{
atmosphere.set_infrared_blocking (set_infrared_blocking);
atmosphere.set_transport_coeff (set_transport_coefficient);
}

void Planet::assign_hydrosphere_basic(double value1, double value2)
{
hydrosphere_transport_coefficient = value1;
hydrosphere_ice_factor = value2;
}

void Planet::init_weather(int seed, int circulation, double cloudcover_min, double cloudcover_max, double high_min, double high_max, double precip_factor)
{


weather.set_random(seed);
weather.set_midlevel(cloudcover_min, cloudcover_max);
weather.set_highlevel(high_min, high_max);
weather.init(2.0 * pi /sidereal_rotation_period, radius, circulation);
weather.set_precipitation_factor(precip_factor);
weather_defined = true;
}


void Planet::init_geology(double age, double iso_abundancy, double m_host, double R_core, string mat1, string mat2, string mat3)
{
geo.set_isotope_abundancy(iso_abundancy);
geo.initialize(mass, radius, age, R_core, T_rad_av, mat1, mat2, mat3);
geo.set_tidal(m_host, period, eccentricity, semimajor);
geo.finish_init();
if (geology_depth_profile) {geo.show_profile();}
geo.list();
geology_defined = true;
} 


void Planet::init_geology(double age, double iso_abundancy, double m_host, string mat1, string mat2)
{
geo.set_isotope_abundancy(iso_abundancy);
geo.initialize(mass, radius, age, T_rad_av, mat1, mat2);
geo.set_tidal(m_host, period, eccentricity, semimajor);
geo.finish_init();
if (geology_depth_profile) {geo.show_profile();}
geo.list();
geology_defined = true;
} 

void Planet::init_geology(double age, double iso_abundancy, double m_host, string mat)
{
geo.set_isotope_abundancy(iso_abundancy);
geo.initialize(mass, radius, age, T_rad_av, mat);
geo.set_tidal(m_host, period, eccentricity, semimajor);
geo.finish_init();
if (geology_depth_profile) {geo.show_profile();}
geo.list();
geology_defined = true;
} 

void Planet::init_geology_undifferentiated(double age, double iso_abundancy, double m_host, string mat1, string mat2)
{
geo.set_isotope_abundancy(iso_abundancy);
geo.initialize_undifferentiated(mass, radius, age, T_rad_av, mat1, mat2);
geo.set_tidal(m_host, period, eccentricity, semimajor);
geo.finish_init();
if (geology_depth_profile) {geo.show_profile();}
geo.list();
geology_defined = true;
} 


void Planet::geology_override_density(double rho, string layer)
{
geo.override_density(rho, layer);
}

void Planet::init_forest_fire_simulation(double cloud, double T_min, double T_max)
{

for (int i=0; i< resolution_lat; i++)
	{
	for (int j=0; j< resolution_lon; j++)
		{
		surface_elements[i][j].set_forest_fire(cloud, T_min, T_max);
		}
	}

}

void Planet::set_xyz (double set_x, double set_y, double set_z)
{
x = set_x; y = set_y; z = set_z;
}

void Planet::set_daytime (double time)
{
day_fraction = time/rotation_period;
if (day_fraction > 1.0) {day_fraction = day_fraction - 1.0;}
}


void Planet::set_day_fraction (double time)
{
day_fraction = time;

if (day_fraction > 1.0) {day_fraction = day_fraction - 1.0;}
if (day_fraction < 0.0) {day_fraction = day_fraction + 1.0;}
}

void Planet::set_longday_fraction (double time)
{
longday_fraction = time;
if (longday_fraction > 1.0) {longday_fraction = longday_fraction - 1.0;}
}

void Planet::set_solar_constants (double max, double min)
{
//cout << "Solar constant is now: " << max << endl;

solar_constant_max[0] = max;
solar_constant_min[0] = min;

}

void Planet::set_solar_constants (double smax, double smin, int index)
{

solar_constant_max[index] = smax;
solar_constant_min[index] = smin;

}

void Planet::set_geo_par(std::string name, double value)
{
if (name == "depth_resolution") {geology_depth_profile = true;}
geo.set_parameter(name, value);
}

void Planet::set_weather_par(std::string name, double value)
{


for (int i=0; i< resolution_lat; i++)
	{
	for (int j=0; j< resolution_lon; j++)
		{
		if (name == "snow_albedo") {surface_elements[i][j].set_parameter("snow_albedo", value);}
		else if (name == "precipitation_threshold_convection") {surface_elements[i][j].set_parameter("precipitation_threshold", value);}
		else if (name == "precipitation_factor_convection") {surface_elements[i][j].set_parameter("precipitation_factor", value);}
		else if (name == "cloud_factor_convection") {surface_elements[i][j].set_parameter("convection_factor", value);}
		else if (name == "storms") {surface_elements[i][j].set_parameter("storms", value);}
		}

	}

if (name == "precipitation_feedback")
	{
	weather.set_parameter("precipitation_feedback", value);
	}
else if (name == "cloud_feedback")
	{
	weather.set_parameter("cloud_feedback", value);
	}
else if (name == "mid_cloud_feedback")	
	{
	weather.set_parameter("mid_cloud_feedback", value);
	}
else if (name == "high_cloud_feedback")	
	{
	weather.set_parameter("high_cloud_feedback", value);
	}
else if (name == "ridge_migration_factor")
	{
	ridge_migration_factor = value;
	}
else if (name == "ridge_suppression_factor")
	{
	weather.set_parameter("ridge_suppression_factor", value);
	}
else if (name == "grad_motion_factor")
	{
	weather.set_parameter("grad_motion_factor", value);
	}
else if (name == "storms")
	{
	weather.set_parameter("storms", value);
	if (value == 1.0) {compute_storms = true;}
	}
else if (name == "smoke_stratosphere_transfer")
	{
	smoke_stratosphere_transfer_coefficient = value;
	if (value > 0.0) {compute_smoke_transfer = true;}
	}

}


void Planet::compute_average_thermal_properties()
{
double I_max = 0.0;
double I_min = 0.0;

double T_emit_periapsis;
double T_emit_apoapsis;

for (int i = 0; i < n_stars; i++)
	{
	I_max +=solar_constant_max[i];
	I_min +=solar_constant_min[i];
	}

I_max *= (1.0 -albedo); 
I_min *= (1.0 - albedo);

I_max += internal_heat_flux;
I_min += internal_heat_flux;

I_max *= 0.25;
I_min *= 0.25;

T_rad_av_periapsis = pow(I_max / (sigma_SB * (1.0 - atmosphere.get_infrared_blocking())), 0.25);
T_rad_av_apoapsis = pow(I_min /(sigma_SB * (1.0 - atmosphere.get_infrared_blocking())), 0.25);

T_emit_periapsis = pow(I_max / sigma_SB , 0.25);
T_emit_apoapsis = pow(I_min /sigma_SB , 0.25);

temperature = 0.5 * T_rad_av_periapsis + 0.5 * T_rad_av_apoapsis;
T_rad_av = temperature;

atmosphere.set_mean_surface_temperature(0.5 * T_rad_av_periapsis + 0.5 * T_rad_av_apoapsis);
atmosphere.set_mean_radiative_temperature(0.5 * T_emit_periapsis + 0.5 * T_emit_apoapsis);
atmosphere.set_minimum_surface_temperature(T_rad_av_apoapsis);
}


void Planet::compute_IR_retention()
{
double lambda_min = 1e-6;
double lambda_max = 100e-6;

int n = 1000;

double step, lambda;
step = (lambda_max - lambda_min)/(n - 1); 

double sum_surface = 0.0;
double sum_space = 0.0;

for (int i=0; i<n; i++)
	{
	lambda = lambda_min + i * step;
	sum_surface += spectrum(lambda);
	sum_space += spectrum_atm_filtered(lambda);
	}

cout << "Atmospheric IR retention:    " << 1.0 - sum_space/sum_surface << endl;

}

void Planet::compute_atmosphere_transport()
{

cout << "Atmospheric transport coeff: " << (atmosphere.get_column_mass() / atmosphere_column_mass_earth) * (atmosphere.get_cp() / cp_atmosphere_earth) << endl;
}

double Planet::solar_elevation_angle_sin (double lat, double lon)
{

double sin_Psi = sin(lat) * sin(solar_declination) - cos(lat) * cos(solar_declination) * cos(2.0 * pi * day_fraction - lon + day_correction_angle_star);

if (elevation_correction_angle_star != 0.0)
	{
	double cos_Psi = sqrt(1.0 - pow(sin_Psi,2.0));

	sin_Psi = sin_Psi * cos(elevation_correction_angle_star) + cos_Psi * sin(elevation_correction_angle_star);
	}


if (sin_Psi < 0) {return 0.0;} else {return sin_Psi;}
}

double Planet::companion_elevation_angle_sin (double lat, double lon)
{
double sin_Psi = sin(lat) * sin(companion_declination) - cos(lat) * cos(companion_declination) * cos(2.0 * pi * longday_fraction - lon);

if (sin_Psi < 0) {return 0.0;} else {return sin_Psi;}
}


double Planet::binary_elevation_angle_sin (double lat, double lon)
{
double sin_Psi = sin(lat) * sin(solar_declination) - cos(lat) * cos(solar_declination) * cos(2.0 * pi * day_fraction - lon + day_correction_angle_binary);

if (elevation_correction_angle_binary != 0.0)
	{
	double cos_Psi = sqrt(1.0 - pow(sin_Psi,2.0));

	sin_Psi = sin_Psi * cos(elevation_correction_angle_binary) + cos_Psi * sin(elevation_correction_angle_binary);
	}

if (sin_Psi < 0) {return 0.0;} else {return sin_Psi;}
}

double Planet::albedo_water (double sinphi)
{
if (sinphi < 0.0) {return 0.0;} 
else {return 0.02 + 0.98 * exp(-sinphi/0.156377);}
}


void Planet::create_surface_elements (int res_lat, int res_lon)
{
double lat, lon;
double ridge_lat_min_N, ridge_lat_max_N, ridge_lat_min_S, ridge_lat_max_S, ridge_migration_deg, ridge_suppression;

double size_lat = 180.0 / res_lat;
double size_lon = 360.0 / res_lon;

ridge_lat_min_N = weather.get_ridge_deg_low(1); 
ridge_lat_max_N = weather.get_ridge_deg_high(1);
ridge_lat_min_S = weather.get_ridge_deg_low(-1); 
ridge_lat_max_S = weather.get_ridge_deg_high(-1);
ridge_migration_deg = ridge_migration_factor * solar_declination * rad_to_deg;
ridge_suppression = weather.get_ridge_suppression();

// dynamical memory allocation for surface element array

surface_elements = new Surface *[res_lat];

for (int i=0; i< res_lat; i++)
	{
	surface_elements[i] = new Surface [res_lon];
	}


DE_array = new double *[res_lat];

for (int i=0; i< res_lat; i++)
	{
	DE_array[i] = new double [res_lon];
	}

double thermal_depth = diurnal_depth * sqrt(rotation_period/86400.0);
if ((max_diurnal_depth > 0.0) && (thermal_depth > max_diurnal_depth)) {thermal_depth = max_diurnal_depth;}

double initial_temperature = T_rad_av_periapsis;
if (T_init > 0.0) {initial_temperature = T_init;}


resolution_lat = res_lat;
resolution_lon = res_lon;

for (int i=0; i< res_lat; i++)
	{
	for (int j=0; j< res_lon; j++)
		{
		lat = -90.0 + 0.5 * size_lat + i * size_lat;
		lon = j * size_lon;
		
		Surface surface_element(lat, lon, size_lat, size_lon, radius);
		surface_elements[i][j] = surface_element;
		surface_elements[i][j].set_albedo(albedo);
		surface_elements[i][j].set_thermal_properties (CV, thermal_depth, initial_temperature);
		surface_elements[i][j].set_atmosphere (atmosphere.get_infrared_blocking());
			
		if (weather_defined)
			{
			surface_elements[i][j].set_weather(true);

			if ((surface_elements[i][j].get_lat() * rad_to_deg > (ridge_lat_min_N + ridge_migration_deg)) && (surface_elements[i][j].get_lat() * rad_to_deg < (ridge_lat_max_N + ridge_migration_deg)))
				{surface_elements[i][j].set_convection(ridge_suppression);}

			if ((surface_elements[i][j].get_lat() * rad_to_deg > (ridge_lat_min_S + ridge_migration_deg)) && (surface_elements[i][j].get_lat() * rad_to_deg < (ridge_lat_max_S +ridge_migration_deg)))
				{surface_elements[i][j].set_convection(ridge_suppression);}
			}
		}	

	}


}


double Planet::get_element_temperature(int i, int j)
{
return surface_elements[i][j].get_temperature();
}

double Planet::get_element_latitude(int i, int j)
{
return surface_elements[i][j].get_lat();
}

double Planet::get_element_longitude(int i, int j)
{
return surface_elements[i][j].get_lon();
}

double Planet::get_element_solar_angle(int i, int j, int body_index)
{

//cout << designation << " " << i << " " << j << " " << body_index << endl;

return surface_elements[i][j].get_solar_elevation_angle(body_index);
}

double Planet::get_element_absorbed_energy (int i, int j)
{
return surface_elements[i][j].get_absorbed_energy();
}

double Planet::get_element_absorbed_flux (int i, int j)
{
return surface_elements[i][j].get_absorbed_energy_flux();
}

double Planet::get_element_emitted_energy (int i, int j)
{
return surface_elements[i][j].get_emitted_energy();
}

double Planet::get_element_emitted_IR_flux(int i, int j)
{
return surface_elements[i][j].get_emitted_energy_flux();
}

double Planet::get_element_reflected_flux(int i, int j)
{
return surface_elements[i][j].get_reflected_energy_flux();
}

double Planet::get_element_transported_flux(int i, int j)
{
return surface_elements[i][j].get_transported_energy_flux();
}

double Planet::get_element_cloud_cover(int i, int j)
{
return surface_elements[i][j].get_cloudcover();
}

double Planet::get_element_low_cloud_cover(int i, int j)
{
return surface_elements[i][j].get_low_cloudcover();
}

double Planet::get_element_mid_cloud_cover(int i, int j)
{
return surface_elements[i][j].get_mid_cloudcover();
}

double Planet::get_element_high_cloud_cover(int i, int j)
{
return surface_elements[i][j].get_high_cloudcover();
}

double Planet::get_element_ice_thickness(int i, int j)
{
return surface_elements[i][j].get_ice_thickness();
}

double Planet::get_element_snow_thickness(int i, int j)
{
return surface_elements[i][j].get_snow_thickness();
}

double Planet::get_element_precipitation(int i, int j)
{
return surface_elements[i][j].get_precipitation();
}

double Planet::get_element_convective_energy(int i, int j)
{
return surface_elements[i][j].get_convective_energy();
}

void Planet::set_distance_to_star (double set_dist, double L)
{
distance_to_star = set_dist;
solar_constant_current[0] = L / (4.0 * pi * pow(distance_to_star, 2.0));
}

void Planet::set_distance_to_companion (double set_dist, double L)
{
distance_to_companion = set_dist;
solar_constant_current[1] = L / (4.0 * pi * pow(distance_to_companion, 2.0));
}

void Planet::set_distance_to_binary (double set_dist, double L)
{
distance_to_binary = set_dist;
solar_constant_current[1] = L / (4.0 * pi * pow(distance_to_binary, 2.0));

}


void Planet::set_day_correction_angle_star(double set_angle)
{
day_correction_angle_star = set_angle;
}

void Planet::set_elevation_correction_angle_star(double set_angle)
{
elevation_correction_angle_star = set_angle;
}

void Planet::set_day_correction_angle_binary(double set_angle)
{
day_correction_angle_binary = set_angle;
}

void Planet::set_elevation_correction_angle_binary(double set_angle)
{
elevation_correction_angle_binary = set_angle;
}


double Planet::search_ce_maximum(int i, int j)
{
double ce_max, ce_test;
double direction_deg;

ce_max = surface_elements[i][j].get_convective_energy();

if (i>0)
	{
	ce_test = surface_elements[i-1][j].get_convective_energy();
	if (ce_test > ce_max) {ce_max = ce_test; direction_deg = 180.0;}
	if (j>0)
		{
		ce_test = surface_elements[i-1][j-1].get_convective_energy();
		if (ce_test > ce_max) {ce_max = ce_test; direction_deg = 225.0;}
		}
	if (j<resolution_lon-1)
		{
		ce_test = surface_elements[i-1][j+1].get_convective_energy();
		if (ce_test > ce_max) {ce_max = ce_test; direction_deg = 135.0;}
		}
	}

if (j>0)
	{
	ce_test = surface_elements[i][j-1].get_convective_energy();
	if (ce_test > ce_max) {ce_max = ce_test; direction_deg = 270.0;}
	}
if (j<resolution_lon-1)
	{
	ce_test = surface_elements[i][j+1].get_convective_energy();
	if (ce_test > ce_max) {ce_max = ce_test; direction_deg = 90.0;}
	}

if (i<resolution_lat-1)
	{
	ce_test = surface_elements[i+1][j].get_convective_energy();
	if (ce_test > ce_max) {ce_max = ce_test; direction_deg = 0.0;}
	if (j>0)
		{
		ce_test = surface_elements[i+1][j-1].get_convective_energy();
		if (ce_test > ce_max) {ce_max = ce_test; direction_deg = 335.0;}
		}
	if (j<resolution_lon-1)
		{
		ce_test = surface_elements[i+1][j+1].get_convective_energy();
		if (ce_test > ce_max) {ce_max = ce_test; direction_deg = 45.0;}
		}
	}

if (ce_max == surface_elements[i][j].get_convective_energy()) {return -1.0;}
else {return direction_deg * deg_to_rad;}

}


void Planet::set_surface_irradiation ()
{
double lat, lon;
double radiative_flux,  T_av;
double synoptic_cloudcover, midlevel_cloudcover, highlevel_cloudcover, precipitation;
double rn_cloud, rn_precipitation;
double ridge_lat_min_N, ridge_lat_max_N, ridge_lat_min_S, ridge_lat_max_S, ridge_migration_deg, ridge_suppression;
double average_smoke_level, convective_energy, storm_energy, grad_dir;
double res_lat_rad, res_lon_rad;


ridge_lat_min_N = weather.get_ridge_deg_low(1); 
ridge_lat_max_N = weather.get_ridge_deg_high(1);
ridge_lat_min_S = weather.get_ridge_deg_low(-1); 
ridge_lat_max_S = weather.get_ridge_deg_high(-1);
ridge_migration_deg = ridge_migration_factor * solar_declination * rad_to_deg;
ridge_suppression = weather.get_ridge_suppression();

res_lat_rad = pi/resolution_lat;
res_lon_rad = (2.0 * pi)/resolution_lon;

if (weather_defined)
	{
	T_av = get_average_temperature();
	weather.set_atmosphere_temperature(T_av);

	if (compute_smoke_transfer)
		{
		average_smoke_level = get_average_smoke();
		//cout << "Average smoke level is now: " << average_smoke_level << endl;
		}
	}

//cout << "Ridge: " << ridge_migration_deg << endl;

for (int i=0; i< resolution_lat; i++)
	{
	for (int j=0; j < resolution_lon; j++)
		{
		
		//cout << designation << " " << surface_elements[i][j].get_temperature() << endl;
		//cout << surface_elements[i][j].get_area() << endl;

		lat = surface_elements[i][j].get_lat();
		lon = surface_elements[i][j].get_lon();

		if (weather_defined)
			{
			synoptic_cloudcover = weather.get_low_cloudcover(lat, lon, 0.5* pi/resolution_lat, 0.5 * 2.0 * pi/resolution_lon);
			midlevel_cloudcover = weather.get_mid_cloudcover(lat, lon);
			highlevel_cloudcover = weather.get_high_cloudcover(lat, lon);
			precipitation = weather.get_precipitation(lat, lon, 0.5* pi/resolution_lat, 0.5 * 2.0 * pi/resolution_lon);
			rn_cloud = weather.get_convection_rn_cloud(lat, lon);
			rn_precipitation = weather.get_convection_rn_precipitation(lat, lon);
			
			surface_elements[i][j].set_synoptic_cloudcover(synoptic_cloudcover); 
			surface_elements[i][j].set_midlevel_cloudcover(midlevel_cloudcover); 
			surface_elements[i][j].set_highlevel_cloudcover(highlevel_cloudcover); 
			surface_elements[i][j].set_precipitation(precipitation);
			surface_elements[i][j].set_weather_rn(rn_cloud, rn_precipitation);
			
			if (compute_smoke_transfer)
				{
				surface_elements[i][j].set_highlevel_smoke(average_smoke_level * smoke_stratosphere_transfer_coefficient);
				}

			if (compute_storms)
				{

				convective_energy = surface_elements[i][j].get_convective_energy();

				weather.check_storm_generation(lat, lon,res_lat_rad, res_lon_rad, convective_energy);
				storm_energy = weather.get_storm_energy_usage(lat - 0.5 * res_lat_rad, lat + 0.5 * res_lat_rad, lon - 0.5 * res_lon_rad, lon + 0.5 * res_lon_rad);
	

				
				if (storm_energy > 0.0)
					{

					surface_elements[i][j].set_storm_cloudcover(weather.get_storm_cloudcover(lat, lon, 0.5* res_lat_rad, 0.5 * res_lon_rad));
					//surface_elements[i][j].set_storm_cloudcover(1.0);

					grad_dir = search_ce_maximum(i,j);
					if (storm_energy > convective_energy) 
						{
						weather.set_storm_energy_usage(lat - 0.5 * res_lat_rad, lat + 0.5 * res_lat_rad, lon - 0.5 * res_lon_rad, lon + 0.5 * res_lon_rad, grad_dir, 0);
						}
					else
						{
						weather.set_storm_energy_usage(lat - 0.5 * res_lat_rad, lat + 0.5 * res_lat_rad, lon - 0.5 * res_lon_rad, lon + 0.5 * res_lon_rad, grad_dir, 1);
						}
					surface_elements[i][j].use_convective_energy(storm_energy * 1.227e+12 / surface_elements[i][j].get_area());

					}
				else
					{
					surface_elements[i][j].set_storm_cloudcover(0.0);
					}

				}

			}


		double elevation_angle_sin = solar_elevation_angle_sin(lat, lon);

		//if ((i==8) && (j==10)) {cout << elevation_angle_sin << endl;}
		
		surface_elements[i][j].set_elevation_angle(elevation_angle_sin, 0);
		radiative_flux = elevation_angle_sin * solar_constant_current[0];
		if (n_stars ==2) 
			{
			if (companion_defined)
				{
				double companion_angle_sin = companion_elevation_angle_sin(lat, lon);
				surface_elements[i][j].set_elevation_angle(companion_angle_sin, 1);
				radiative_flux += companion_angle_sin * solar_constant_current[1];
				}
			if (binary_defined)	
				{
				double binary_angle_sin = binary_elevation_angle_sin(lat, lon);
				surface_elements[i][j].set_elevation_angle(binary_angle_sin, 2);
				radiative_flux += binary_angle_sin * solar_constant_current[1];
				}
			}

		radiative_flux *= (1.0 - atmosphere.get_albedo());
		
		surface_elements[i][j].set_radiative_flux(radiative_flux);
		surface_elements[i][j].set_internal_flux(internal_heat_flux);

		if (surface_elements[i][j].get_albedo_dynamical() == 1)
			{
			surface_elements[i][j].set_albedo(albedo_water(elevation_angle_sin)); 
			}

		if (weather_defined)
			{
			if ((surface_elements[i][j].get_lat() * rad_to_deg > (ridge_lat_min_N+ ridge_migration_deg)) && (surface_elements[i][j].get_lat() * rad_to_deg < (ridge_lat_max_N + ridge_migration_deg)))
				{surface_elements[i][j].set_convection(ridge_suppression);}
			else if ((surface_elements[i][j].get_lat() *rad_to_deg > (ridge_lat_min_S + ridge_migration_deg)) && (surface_elements[i][j].get_lat() * rad_to_deg < (ridge_lat_max_S + ridge_migration_deg)))
				{surface_elements[i][j].set_convection(ridge_suppression);}
			else
				{surface_elements[i][j].set_convection(1.0);}
			}


		}
	}

}



void Planet::add_indirect_surface_irradiation ()
{
int i,j;

double lat, lon;
double radiative_flux;


for (i=0; i< resolution_lat; i++)
	{
	for (j=0; j < resolution_lon; j++)
		{
		lat = surface_elements[i][j].get_lat();
		lon = surface_elements[i][j].get_lon();
		double elevation_angle_sin = solar_elevation_angle_sin(lat, lon);
		surface_elements[i][j].set_elevation_angle(elevation_angle_sin, 0);
		radiative_flux = surface_elements[i][j].get_radiative_flux() + elevation_angle_sin * planetary_constant_current;
		
		//cout << "Before: " << surface_elements[i][j].get_radiative_flux() << " after: " << radiative_flux << endl;

		radiative_flux *= (1.0 - atmosphere.get_albedo());

		surface_elements[i][j].set_radiative_flux(radiative_flux);

		/*if (surface_elements[i][i].get_albedo_dynamical() == 1)
			{
			surface_elements[i][j].set_albedo(albedo_water(elevation_angle_sin)); 
			}*/
		}
	}
}

void Planet::set_sun_angle()
{
int i,j;

double lat, lon, elevation_angle_sin;

for (i=0; i< resolution_lat; i++)
	{
	for (j=0; j < resolution_lon; j++)
		{
		lat = surface_elements[i][j].get_lat();
		lon = surface_elements[i][j].get_lon();
		elevation_angle_sin = solar_elevation_angle_sin(lat, lon);
		surface_elements[i][j].set_elevation_angle(elevation_angle_sin, 0);
		}
	}

}



void Planet::assign_materials_map (string materials_file, Material materials[])
{
int i,j;
int index;
double albedo, C_V, V_diurnal, fire_factor, local_convection_factor;

ifstream matfile (materials_file.c_str());
  if (matfile.is_open())
  {

	for (i=resolution_lat-1; i>=0; i--)
		{
		for (j=0; j < resolution_lon; j++)
			{
      			matfile >> index;
			albedo = materials[index].albedo;
			C_V = materials[index].C_V;
			V_diurnal = materials[index].depth_diurnal_norm * sqrt(rotation_period/86400.0);
			fire_factor = materials[index].fire_factor;
			local_convection_factor = materials[index].convection_factor;
			surface_elements[i][j].set_thermal_properties(C_V, V_diurnal, T_rad_av_periapsis);		
			surface_elements[i][j].set_fire_factor(fire_factor);	
			surface_elements[i][j].set_local_convection(local_convection_factor);	

			if (albedo > 0.0) 
				{surface_elements[i][j].set_albedo(materials[index].albedo);}
			else
				{
				surface_elements[i][j].set_liquid(true);	
				surface_elements[i][j].set_ice_buildup(hydrosphere_ice_factor);
				surface_elements[i][j].set_albedo_dynamical((int) -albedo);
				}
			
			}

		}
    matfile.close();
  }

   else cout << "Unable to open file " << materials_file << " !" << endl;


//compute_average_albedo();
//list_albedo_dynamical();
}

void Planet::compute_average_albedo()
{

// to get dynamical albedo, set surface irradiation first

set_surface_irradiation();

double albedo_sum = 0.0;
double area_sum = 0.0;

for (int i=0; i < resolution_lat; i++)
	{
	for (int j=0; j< resolution_lon; j++)
		{
		if (surface_elements[i][j].get_solar_elevation_angle(0) == 0.0) {continue;}
		albedo_sum+= surface_elements[i][j].get_ground_albedo() * surface_elements[i][j].get_area();
		area_sum  += surface_elements[i][j].get_area();
		}
	}

//albedo = albedo_sum / (resolution_lon * resolution_lat);
albedo = albedo_sum/area_sum;

cout << "Average albedo: " << albedo << endl;
}


double Planet::get_average_albedo()
{
set_surface_irradiation();

double albedo_sum = 0.0;
double area_sum = 0.0;


for (int i=0; i < resolution_lat; i++)
	{
	for (int j=0; j< resolution_lon; j++)
		{
		//cout << surface_elements[i][j].get_solar_elevation_angle(0) << endl;
		if (surface_elements[i][j].get_solar_elevation_angle(0) == 0.0) {continue;}

		albedo_sum+= surface_elements[i][j].get_ground_albedo() * surface_elements[i][j].get_area();
		area_sum  += surface_elements[i][j].get_area();
		}
	}
return albedo_sum/area_sum;
}

void Planet::compute_average_temperature()
{

double T_sum = 0.0;
double area_sum = 0.0;

for (int i=0; i < resolution_lat; i++)
	{
	for (int j=0; j< resolution_lon; j++)
		{
		T_sum+= surface_elements[i][j].get_temperature() * surface_elements[i][j].get_area();
		area_sum  += surface_elements[i][j].get_area();
		}
	}

//double T = T_sum / (resolution_lon * resolution_lat);
double T = T_sum/ area_sum;
temperature = T;

cout << "Current average temperature: " << T << endl;
}

double Planet::get_average_temperature()
{

double T_sum = 0.0;
double area_sum = 0.0;

for (int i=0; i < resolution_lat; i++)
	{
	for (int j=0; j< resolution_lon; j++)
		{
		T_sum+= surface_elements[i][j].get_temperature() * surface_elements[i][j].get_area();
		area_sum  += surface_elements[i][j].get_area();
		}
	}

temperature = T_sum/area_sum;

return T_sum/ area_sum;
}


void Planet::compute_average_cloudcover()
{

double c_sum = 0.0;
double area_sum = 0.0;

for (int i=0; i < resolution_lat; i++)
	{
	for (int j=0; j< resolution_lon; j++)
		{
		c_sum+= surface_elements[i][j].get_cloudcover() * surface_elements[i][j].get_area();
		area_sum  += surface_elements[i][j].get_area();
		}
	}


cout << "Current average cloudcover: " << c_sum / area_sum << endl;
}

double Planet::get_average_cloudcover()
{

double c_sum = 0.0;
double area_sum = 0.0;

for (int i=0; i < resolution_lat; i++)
	{
	for (int j=0; j< resolution_lon; j++)
		{
		c_sum+= surface_elements[i][j].get_cloudcover() * surface_elements[i][j].get_area();
		area_sum  += surface_elements[i][j].get_area();
		}
	}


return c_sum / area_sum;
}

double Planet::get_average_smoke()
{

double s_sum = 0.0;
double area_sum = 0.0;

for (int i=0; i < resolution_lat; i++)
	{
	for (int j=0; j< resolution_lon; j++)
		{
		s_sum+= surface_elements[i][j].get_smoke_level() * surface_elements[i][j].get_area();
		area_sum  += surface_elements[i][j].get_area();
		}
	}


return s_sum / area_sum;
}


void Planet::compute_thermal_flux ()
{
int day_counter = 0;
int night_counter = 0;

dayside_flux = 0.0;
nightside_flux = 0.0;

for (int i=0; i < resolution_lat; i++)
	{
	for (int j=0; j< resolution_lon; j++)
		{
		if (surface_elements[i][j].get_solar_elevation_angle(0) > 0.0)
			{
			dayside_flux += surface_elements[i][j].get_emitted_energy_flux();
			day_counter ++;
			}
		else
			{
			nightside_flux += surface_elements[i][j].get_emitted_energy_flux();
			night_counter++;
			}
		}
	}

dayside_flux = dayside_flux / day_counter;
nightside_flux = nightside_flux / night_counter;

//cout << designation << " Day: " << dayside_flux << endl;
//cout << designation << " Night: " << nightside_flux << endl;

}

void Planet::compute_reflected_flux ()
{
int counter = 0;


reflected_flux = 0.0;


for (int i=0; i < resolution_lat; i++)
	{
	for (int j=0; j< resolution_lon; j++)
		{
		if (surface_elements[i][j].get_solar_elevation_angle(0) > 0.0)
			{
			reflected_flux += surface_elements[i][j].get_reflected_energy_flux();
			counter ++;
			}
		}
	}

reflected_flux /=counter;

}


void Planet::evolve_temperature (double time)
{
double day_fraction_set;



// timestep is fine-grained if requested, but never falls below 1/50 of a day

int n_evolution_steps = 10;

double timestep = time/n_evolution_steps;

//cout << timestep << endl;

if (timestep > rotation_period/50.0) 
	{
	timestep = rotation_period/50.0;
	n_evolution_steps = (int) time/timestep;
	}





double daytime ;


for (int i=0; i< n_evolution_steps; i++)
	{
	daytime =  day_fraction * rotation_period;
	set_surface_irradiation();
	for (int j=0; j< resolution_lat; j++)
		{
		for (int k=0; k < resolution_lon; k++)
			{
			surface_elements[j][k].evolve_temperature(timestep);
			}
		}
	day_fraction_set = (daytime + timestep)/rotation_period;
	set_day_fraction(day_fraction_set);
	atmosphere_transport (timestep);
	}

//compute_average_temperature();
compute_thermal_flux();
compute_reflected_flux();
}


void Planet::evolve_temperature_onestep (double timestep)
{


for (int j=0; j< resolution_lat; j++)
	{
	for (int k=0; k < resolution_lon; k++)
		{
		surface_elements[j][k].evolve_temperature(timestep);
		}
	}
	
if (weather_defined) {weather.evolve(timestep);}
//cout << designation << " " << surface_elements[0][0].get_temperature() << endl;
atmosphere_transport (timestep);
compute_thermal_flux();
compute_reflected_flux();
}


void Planet::atmosphere_transport (double timestep)
{
bool liquid_flag;

int i,j;

double DE_left, DE_up;
double transport_coeff;

double atmosphere_transport_coeff = 0.7 * 0.66 * 4e8 * atmosphere.get_transport_coeff();
double hydrosphere_transport_coeff = 0.7 * 0.33 * 4e8 * hydrosphere_transport_coefficient;



for (i=0; i< resolution_lat; i++)
	{
	for (j=0; j< resolution_lon; j++)
		{
		DE_array[i][j] = 0.0;
		}
	}




for (i=1; i< resolution_lat; i++)
	{
	for (j=0; j< resolution_lon; j++)
		{

		//cout << designation << " " << i << " " << j << " "<< surface_elements[i][j].get_temperature() << endl;
		transport_coeff = atmosphere_transport_coeff;
		liquid_flag = false;

		if (j>0)
			{
			DE_left = surface_elements[i][j].get_temperature() - surface_elements[i][j-1].get_temperature(); 
			
			if ((surface_elements[i][j].get_liquid() == true) && (surface_elements[i][j-1].get_liquid() == true))
				{
				liquid_flag = true;
				}
			}
		else
			{
			DE_left = surface_elements[i][j].get_temperature() - surface_elements[i][resolution_lon-1].get_temperature(); 

			if ((surface_elements[i][j].get_liquid() == true) && (surface_elements[i][resolution_lon-1].get_liquid() == true))
				{
				liquid_flag = true;
				}

			}


		if (liquid_flag == true)
			{
			transport_coeff += hydrosphere_transport_coeff;
			}
	
		DE_up = surface_elements[i][j].get_temperature() - surface_elements[i-1][j].get_temperature();
		
		DE_array[i][j] -= DE_left * surface_elements[i][j].transport_factor_lon * transport_coeff * timestep;
		DE_array[i][j] -= DE_up * surface_elements[i][j].transport_factor_lat * transport_coeff * timestep;



		if (j>0)
			{DE_array[i][j-1] += DE_left *  surface_elements[i][j].transport_factor_lon * transport_coeff * timestep;}
		else 
			{DE_array[i][resolution_lon-1] += DE_left *  surface_elements[i][j].transport_factor_lon * transport_coeff * timestep;}
	
		
		DE_array[i-1][j] += DE_up *  surface_elements[i][j].transport_factor_lat * transport_coeff * timestep;

		
		}
	}

double DE_sum = 0.0;

for (i=0; i< resolution_lat; i++)
	{
	for (j=0; j< resolution_lon; j++)
		{
		if (DE_array[i][j] > 0.5 * surface_elements[i][j].get_total_energy()) {cout << "Transport: Warning, unstable fluxes, decrease timestep!" << endl;}
		surface_elements[i][j].set_total_energy (surface_elements[i][j].get_total_energy() + DE_array[i][j]);
		surface_elements[i][j].set_transport_energy (DE_array[i][j]);
		DE_sum += DE_array[i][j];
		}
	}


}


double Planet::get_atmosphere_uv_heating(double altitude)
{
return interpolate(altitude, atmosphere_heating_profile, atmosphere_heating_profile_size);
}

double Planet::get_atmosphere_o3_profile(double altitude)
{
return interpolate(altitude, atmosphere_o3_profile, atmosphere_heating_profile_size);
}

double Planet::get_atmosphere_uv_heating_T(double altitude)
{
double T_base, T_rad, T_at;

T_base = T_rad_av - atmosphere.get_lapse_rate() * 0.001 * altitude;

T_rad = 1.35e3 * pow(interpolate(altitude, atmosphere_heating_profile, atmosphere_heating_profile_size), 0.25);


if (T_rad > T_base)
	{
	T_at = T_rad;
	}
else
	{
	T_at = T_base;
	}

return T_at;
}


void Planet::compute_atmosphere_uv_heating(double lambda, double I_lambda, int resolution)
{
double* result;

result = atmosphere.compute_chapman_heating_profile(resolution, lambda * 1e9, I_lambda);

for (int i=0; i< resolution; i++)
	{
	atmosphere_heating_profile[i][1] += result[resolution - i - 1];
	}
}

void Planet::compute_atmosphere_o3_profile(double lambda, double I_lambda, int resolution)
{
double* result;

result = atmosphere.compute_chapman_o3_profile(resolution, lambda * 1e9, I_lambda);

for (int i=0; i< resolution; i++)
	{
	atmosphere_o3_profile[i][1] += result[resolution - i - 1];
	}
}

void Planet::init_atmosphere_uv_heating(int resolution)
{
double alt_res;
atmosphere_heating_profile = new double *[resolution];
atmosphere_o3_profile = new double *[resolution];
atmosphere.init_ozone_profile(resolution);

atmosphere_heating_profile_size = resolution;

for (int i=0; i< resolution; i++)
	{
	atmosphere_heating_profile[i] = new double [2];
	atmosphere_o3_profile[i] = new double [2];
	}


alt_res = 20.0 * atmosphere.get_scale_height() / resolution;

for (int i=0; i< resolution; i++)
	{
	atmosphere_heating_profile[i][0] = i* alt_res;
	atmosphere_heating_profile[i][1] = 0.0;
	atmosphere_o3_profile[i][0] = i* alt_res;
	atmosphere_o3_profile[i][1] = 0.0;
	}

}


void Planet::dump_atmosphere_heating_profile()
{
for (int i=0; i< atmosphere_heating_profile_size; i++)
	{
	cout << atmosphere_heating_profile[i][0] << " " << atmosphere_heating_profile[i][1] << endl;
	}
}

void Planet::list_properties ()
{

cout << endl;
cout << designation << endl;
cout << "-----------" << endl;
cout << "Mass [m_earth]:        " << mass_earth << endl;
cout << "Radius [R_earth]:      " << radius_earth << endl;
cout << "Mean density [g/cm^3]: " << density/1000.0 << endl;
cout << "Surface gravity [g]:   " << surface_gravity /9.81 << endl;
}

void Planet::list_orbit ()
{
if (lagrange_orbit)
	{
	cout << "Orbiting at Lagrange point " << lagrange_point << "." << endl;
	}
else
	{	
	cout << "Orbit around star:" << endl; 
	
	cout << "Semimajor axis [Mkm]:  " << semimajor/1e9 << endl;
	cout << "Eccentricity           " << eccentricity  << endl;
	cout << "Periapsis [Mkm]:       " << periapsis/1e9 << endl;
	cout << "Apoapsis [Mkm]:        " << apoapsis/1e9 << endl;
	cout << "Period [days]:         " << period/86400.0 << endl;
	}
}

void Planet::list_orbit_longyear ()
{
cout << "Orbit around companion:" << endl; 
cout << "Semimajor axis [Mkm]:  " << semimajor_longyear/1e9 << endl;
cout << "Eccentricity           " << eccentricity_longyear  << endl;
cout << "Periapsis [Mkm]:       " << periapsis_longyear/1e9 << endl;
cout << "Apoapsis [Mkm]:        " << apoapsis_longyear/1e9 << endl;
cout << "Period [days]:         " << period_longyear/86400.0 << endl;
if (inclination_longyear != 0.0)
	{
	cout << "Inclination [deg]:     " << inclination_longyear * rad_to_deg << endl;
	}
}

void Planet::list_orbit_binary ()
{
cout << " " << endl;
cout << "Central binary star system orbit: " << endl;
cout << "Semimajor axis [Mkm]:  " << semimajor_binary/1e9 << endl;
cout << "Eccentricity           " << eccentricity_binary  << endl;
cout << "Periapsis [Mkm]:       " << periapsis_binary/1e9 << endl;
cout << "Apoapsis [Mkm]:        " << apoapsis_binary/1e9 << endl;
cout << "Period [days]:         " << period_binary/86400.0 << endl;
if (inclination_binary != 0.0)
	{
	cout << "Inclination [deg]:     " << inclination_binary * rad_to_deg << endl;
	}
cout << " " << endl;
}

void Planet::list_rotation ()
{

double converter = 1.0;
string unit = "[s]";

if ((rotation_period > 900000.0) || (sidereal_rotation_period > 900000.0))
	{
	converter = 1.0/86400.0;
	unit = "[d]";
	}

cout << "Rotational state: " << endl;
cout << "Rotation period " << unit <<":   " << rotation_period * converter<< endl;
cout << "Sid. rot. period "<< unit <<":  " << sidereal_rotation_period * converter << endl;
cout << "Axis tilt [deg]        " << inclination * rad_to_deg  << endl;

if (companion_defined)
	{
cout << "Tilt to comp. [deg]:   " << inclination_axis_companion * rad_to_deg  << endl;
//cout << "Dec. offs. comp.[deg]: " << companion_declination_offset * rad_to_deg  << endl;
	}
}


void Planet::list_thermal_properties ()
{
double I_max = 0.0;
double I_min = 0.0;



for (int i = 0; i < n_stars; i++)
	{
	I_max +=solar_constant_max[i];
	I_min +=solar_constant_min[i];
	}

cout << endl;
cout << "Thermal properties ("<< designation <<")" << endl;
cout << "------------------" << endl;

cout << "Max. irrad. [W/m^2]:   " << I_max << endl;
cout << "Min. irrad. [W/m^2]:   " << I_min << endl;
cout << "Albedo:                " << albedo << endl;
cout << "T_rad periapsis [K]:   " << T_rad_av_periapsis << endl;
cout << "T_rad apoapsis [K]:    " << T_rad_av_apoapsis << endl;
}

void Planet::list_storms(double time)
{
weather.list_storms(time);
}


void Planet::list_surface_elements (int index)
{


for (int i=0; i< resolution_lat; i++)
	{
	for (int j=0; j<resolution_lon; j++)
		{
		if (index == 1) {surface_elements[i][j].list_coords();}
		else if (index == 2) 
			{surface_elements[i][j].list_instantaneous_temperature();}
		else if (index == 3)
			{cout << surface_elements[i][j].get_temperature();}
		else if (index == 4)
			{cout << surface_elements[i][j].get_albedo();}
		}
	cout << endl;
	}
}







void Planet::plot_surface_data_mtv (int index, string filename)
{

int xsize, ysize;
double x_min, x_max, y_min, y_max;

xsize = resolution_lat;
ysize = resolution_lon;

x_min = 0.0;
x_max = 360.0;

y_min = -90.0;
y_max = 90.0;




ofstream resfile (filename.c_str());
  if (resfile.is_open())
  {
  resfile << "$DATA=CONTOUR" << endl;
  resfile << "%contstyle=2" << endl;
  resfile << "%nx=" << ysize << endl;
  resfile << "%ny=" << xsize << endl;
  resfile << "%xmax=" << x_max << endl;
  resfile << "%xmin=" << x_min << endl;
  resfile << "%ymax=" << y_max << endl;
  resfile << "%ymin=" << y_min << endl;
  resfile << "%nsteps=20" << endl;

  for (int i = 0; i< xsize; i=i+1)
	{
	for(int j=0; j<ysize; j++)
		{
		if (index == 1) {resfile << surface_elements[i][j].get_albedo() << endl;}
		else if (index == 2) {resfile << surface_elements[i][j].get_temperature() << endl;}
		else if (index == 3) {resfile << surface_elements[i][j].get_total_irradiation() << endl;}
		}
	}


    resfile.close();

  }
 else cout << "Unable to open file " << "surface_plot.mtv" << " !" << endl;

}





void Planet::plot_surface_data_gnu (int index, string filename)
{

int xsize, ysize;
double x, y;

xsize = resolution_lat;
ysize = resolution_lon;



ofstream resfile (filename.c_str());
  if (resfile.is_open())
  {
  

for(int j=0; j<ysize; j++)
	{
	for (int i = 0; i< xsize; i=i+1)
		{
		x = -90.0 + i * 180.0/(xsize -1);
		y =  j * 360.0/(ysize -1);

		resfile << y << " " << x << " ";

		//cout << i << " " << j << " " << surface_elements[i][j].get_albedo() << endl;	

		if (index == 1) {resfile << surface_elements[i][j].get_albedo() << endl;}
		else if (index == 2) {resfile << surface_elements[i][j].get_temperature() << endl;}
		else if (index == 3) {resfile << surface_elements[i][j].get_total_irradiation() << endl;}
		else if (index == 4) {resfile << surface_elements[i][j].get_emitted_energy_flux() << endl;}
		else if (index == 5) {resfile << surface_elements[i][j].get_reflected_energy_flux() << endl;}
		else if (index == 6) {resfile << surface_elements[i][j].get_transported_energy_flux() << endl;}
		else if (index == 8) {resfile << surface_elements[i][j].get_transported_energy() << endl;}
		else if (index == 9) {resfile << surface_elements[i][j].get_cloudcover() << endl;}
		else if (index == 10) {resfile << surface_elements[i][j].get_absorbed_energy_flux() << endl;}
		else if (index == 11) {resfile << surface_elements[i][j].get_low_cloudcover() << endl;}
		else if (index == 12) {resfile << surface_elements[i][j].get_mid_cloudcover() << endl;}
		else if (index == 13) {resfile << surface_elements[i][j].get_ice_thickness() << endl;}
		else if (index == 14) {resfile << surface_elements[i][j].get_snow_thickness() << endl;}
		else if (index == 15) {resfile << surface_elements[i][j].get_precipitation() << endl;}
		else if (index == 16) {resfile << surface_elements[i][j].get_high_cloudcover() << endl;}
		else if (index == 17) {resfile << surface_elements[i][j].get_ground_albedo() << endl;}
		else if (index == 18) {resfile << surface_elements[i][j].get_total_albedo() << endl;}
		else if (index == 19) {resfile << surface_elements[i][j].get_storm_cloudcover() << endl;}
		else if (index == 20) {resfile << surface_elements[i][j].get_convective_energy() << endl;}
		}
	resfile << " " << endl;
	}


    resfile.close();
    cout << "Surface plot written to " << filename << "." << endl;

  }
 else cout << "Unable to open file " << "surface_plot.mtv" << " !" << endl;

}




void Planet::plot_surface_data_mtv (int index, string filename, double c_min, double c_max)
{

int xsize, ysize;
double x_min, x_max, y_min, y_max;

xsize = resolution_lat;
ysize = resolution_lon;

x_min = 0.0;
x_max = 360.0;

y_min = -90.0;
y_max = 90.0;





ofstream resfile (filename.c_str());
  if (resfile.is_open())
  {
  resfile << "$DATA=CONTOUR" << endl;
  resfile << "%contstyle=2" << endl;
  resfile << "%nx=" << ysize << endl;
  resfile << "%ny=" << xsize << endl;
  resfile << "%xmax=" << x_max << endl;
  resfile << "%xmin=" << x_min << endl;
  resfile << "%ymax=" << y_max << endl;
  resfile << "%ymin=" << y_min << endl;
  resfile << "%cmin=" << c_min << endl;
  resfile << "%cmax=" << c_max << endl;
  resfile << "%nsteps=20" << endl;

  for (int i = 0; i< xsize; i=i+1)
	{
	for(int j=0; j<ysize; j++)
		{
		if (index == 1) {resfile << surface_elements[i][j].get_albedo() << endl;}
		else if (index == 2) {resfile << surface_elements[i][j].get_temperature() << endl;}
		else if (index == 3) {resfile << surface_elements[i][j].get_total_irradiation() << endl;}
		}
	}


    resfile.close();

  }
 else cout << "Unable to open file " << "surface_plot.mtv" << " !" << endl;

}




void Planet::plot_diurnal_evolution (double t_min, double t_max, int i, int j)
{
int n_points = 10000;
double t;



evolve_temperature(t_min);

double t_step = (t_max - t_min) / (n_points - 1);

//cout << t_step << endl;

ofstream resfile ("temperature.dat");
  if (resfile.is_open())
  {
  
  for (int k = 0; k< n_points; k=k+1)

	{
	t = t_min + k * t_step;
	resfile << t << " " << surface_elements[i][j].get_temperature() << endl;
	//cout << surface_elements[i][j].get_lat() << " " << surface_elements[i][j].get_lon() << " " << surface_elements[i][j].get_albedo() << endl;
	evolve_temperature(t_step);
	}


    resfile.close();

  }
 else cout << "Unable to open file " << "temperature.dat" << " !" << endl;
}


void Planet::list_albedo_dynamical ()
{
for (int i=0; i< resolution_lat; i++)
	{
	for (int j=0; j< resolution_lon; j++)
		{
		cout << surface_elements[i][j].get_albedo_dynamical();
		}
	cout << endl;
	}


}


double Planet::get_T_at_depth(double depth)
{
if (geology_defined)
	{
	return geo.get_temperature(depth, 100.0);
	}
else return 0.0;
		
}

double Planet::get_p_at_depth(double depth)
{
if (geology_defined)
	{
	return geo.get_pressure(depth, 100.0);
	}
else return 0.0;
}


int Planet::get_num_storms()
{
if (weather_defined) {return weather.get_num_storms();}
else {return 0;}
}

double Planet::get_storm_lat(int index)
{
return weather.get_storm_lat(index);
}

double Planet::get_storm_lon(int index)
{
return weather.get_storm_lon(index);
}

int Planet::get_storm_category(int index)
{
return weather.get_storm_category(index);
}
