// orbital_solver.hxx -- routines to compute planar n-body orbital motion 
//
// Written by Thorsten Renk, started 2018
//
// Copyright (C) 2018  Thorsten Renk - thorsten@science-and-fiction.org 
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301


class AxisTransform {

	double axis_tilt;
	double dec_offset;

	double x[3];
	double y[3];
	double z[3];

	double axis_tilt_out;
	double dec_offset_out;

	public:

	void initialize (double, double);
	void tilt_orbital_plane(double, double);
	
	double get_eff_tilt();
	double get_eff_dec_offset();
	
};

class CoordinateObject {

	long double X[2];
	long double Y[2];
	long double Z[2];

	long double X_planar[2];
	long double Y_planar[2];
	long double Z_planar[2];

	public:

	void initialize (long double, long double, long double, long double, long double, long double);
	void tilt_by (long double, long double);

	long double get_x_planar(int);
	long double get_y_planar(int);

};



class OrbitalSystem {

	  std::string* designation;
	  
	  int n_bodies;
	  int collision_index_1;
	  int collision_index_2;

	  bool planar_orbit;
	  bool collision_flag;
	  bool GR_corrections;
	  bool init_at_apoapsis;
	  bool retrograde;
	  bool moon_retrograde;
	  bool do_collisions;

	  long double Dt;
	  long double* x;
	  long double* y;
	  long double* z;
	  long double* vx;
	  long double* vy;
	  long double* vz;
	  long double* M;
	  long double* R;
	  long double* L;
	  int* orbit_counter;
	  int* tie_index;
	  bool* active_flag;
	  long double apsis_angle;
	  long double moon_apsis_angle;
	  long double ang_01;
	  long double ang_02;
	  long double ang_03;
	  long double ang_12;
	  long double ang_13;
	  long double ang_23;
	  long double lan_01;
	  long double lan_02;
	  long double lan_03;
	  long double lan_12;
	  long double lan_13;
	  long double lan_23;

	  void rotate_X(int, double);
	  void rotate_Y(int, double);
	  void rotate_Z(int, double);

	  void tilt_plane(int, double, double);

	  void move_to_CoG(int, int);
	  void adjust_momentum_balance();
	  void adjust_momentum_balance(int, int);

	  long double get_plane_tilt_angle(int, int);
	  long double get_lan(int, int);

	  long double V(long double, long double);
	  long double V_N(long double, long double);
	  long double get_force(long double, long double, int , int );
	  long double closest_distance();
	  long double GR_correction(int , int, long double, long double, long double);
	  long double norm(long double, long double);
	  

	  void collision_detection();
	  void process_collision();
	  void process_ties();

	public:
	  OrbitalSystem();
	  void initialize (Star, Planet);
	  void initialize (Star, Star, Planet, int);
	  void initialize (Star, Star, Planet, Planet, int);
	  void initialize (Star, Planet, Planet);
	  int add_body (Body, int);
	  void evolve_timestep();
	  void evolve_for(double);
	  void set_timestep(long double step) {Dt = step;}
	  void list_state();
	  void set_GR(bool state) {GR_corrections = state;}
	  void set_collisions(bool state) {do_collisions = state;}
	  void set_retrograde(bool state) {retrograde = state;}
	  void set_moon_retrograde(bool state) {moon_retrograde = state;}
	  void set_init_apoapsis (bool state) {init_at_apoapsis = state;}
	  void set_apsis_angle (double angle) {apsis_angle = (long double) angle;}
	  void set_moon_apsis_angle (double angle) {moon_apsis_angle = (long double) angle;}
	  void save(std::string);
	  void resume(std::string);
	  double get_x(int index) {return  x[index];}
	  double get_y(int index) {return  y[index];}
	  double get_z(int index) {return  z[index];}
	  double get_vx(int index) {return  vx[index];}
	  double get_vy(int index) {return  vy[index];}
	  double get_vz(int index) {return  vz[index];}
	  double get_true_anomaly(int, int);
	  double get_true_anomaly_planar(int, int);
	  double get_true_anomaly_CoG(int);
	  double get_year_fraction (int, int);
	  double get_year_fraction_CoG (int);
	  double get_distance(int, int);
	  double get_rel_velocity(int, int);
	  double get_semimajor(int, int);
	  double get_eclipse_factor(int, int, int);
	  double get_phase_area(int, int, int);
	  double get_day_correction_angle(int, int);
	  double get_elevation_correction_angle(int, int);
	  double get_timestep() {return (double) Dt;}
	  void set_x(int index, double set_x) {x[index] = (long double) set_x;} 
	  void set_y(int index, double set_y) {y[index] = (long double) set_y;} 
	  void set_z(int index, double set_z) {z[index] = (long double) set_z;} 
	  void set_vx(int index, double set_vx) {vx[index] = (long double) set_vx;} 
	  void set_vy(int index, double set_vy) {vy[index] = (long double) set_vy;} 
	  void set_vz(int index, double set_vz) {vz[index] = (long double) set_vz;}
};



