// orbital_solver.cxx -- routines to compute planar n-body orbital motion 
//
// Written by Thorsten Renk, started 2018
//
// Copyright (C) 2018  Thorsten Renk - thorsten@science-and-fiction.org 
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301


#include <iostream>
#include <fstream>
#include <stdlib.h>
#include <string>
#include <math.h>

#include "planet.hxx"
#include "star.hxx"
#include "body.hxx"
#include "orbital_solver.hxx"
#include "constants.hxx"

using namespace std;


void AxisTransform::initialize(double tilt_in, double offset_in)
{
double st, ct, so, co;
double x_tmp, y_tmp, z_tmp;

axis_tilt = tilt_in;
dec_offset = offset_in;
axis_tilt_out = axis_tilt;
dec_offset_out = dec_offset;

st = sin(axis_tilt);
ct = cos(axis_tilt);
so = sin(dec_offset);
co = cos(dec_offset);

x[0] = 1.0; y[0] = 0.0; z[0] = 0.0;
x[1] = 0.0; y[1] = 1.0; z[1] = 0.0;
x[2] = 0.0; y[2] = 0.0; z[2] = 1.0;

// tilt along y-axis

for (int i=0; i<3; i++)
	{
	x_tmp = st * z[i] + ct * x[i];
	z_tmp = ct * z[i] - st * x[i];

	x[i] = x_tmp;
	z[i] = z_tmp;
	}

// external rotation along fixed z-axis

for (int i=0; i<3; i++)
	{
	x_tmp = so * y[i] + co * x[i];
	y_tmp = co * y[i] - so * x[i];

	x[i] = x_tmp;
	y[i] = y_tmp;
	}

}


void AxisTransform::tilt_orbital_plane(double inc, double lan)
{
double si, ci, sl, cl;
double x_tmp, y_tmp, z_tmp;

si = sin(inc);
ci = cos(inc);
sl = sin(-lan);
cl = cos(-lan);

// external rotation back to standard tilting along y-axis

for (int i=0; i<3; i++)
	{
	x_tmp = sl * y[i] + cl * x[i];
	y_tmp = cl * y[i] - sl * x[i];

	x[i] = x_tmp;
	y[i] = y_tmp;
	}

// tilt along y-axis

for (int i=0; i<3; i++)
	{
	x_tmp = si * z[i] + ci * x[i];
	z_tmp = ci * z[i] - si * x[i];

	x[i] = x_tmp;
	z[i] = z_tmp;
	}

// external rotation  to correct lan

sl = sin(lan);
cl = cos(lan);

for (int i=0; i<3; i++)
	{
	x_tmp = sl * y[i] + cl * x[i];
	y_tmp = cl * y[i] - sl * x[i];

	x[i] = x_tmp;
	y[i] = y_tmp;
	}

axis_tilt_out = acos(z[2]);

//cout << "Axis: " << x[0] << " " << y[0] << endl;

dec_offset_out = -atan2(y[2], x[2]);

}

double AxisTransform::get_eff_tilt()
{
return axis_tilt_out;
}

double AxisTransform::get_eff_dec_offset()
{
return dec_offset_out;
}


void CoordinateObject::initialize(long double x0, long double y0, long double z0, long double x1, long double y1, long double z1)
{
X[0] = x0; Y[0] = y0; Z[0] = z0;
X[1] = x1; Y[1] = y1; Z[1] = z1;
}


void CoordinateObject::tilt_by(long double inc, long double lan)
{
long double si, ci, sl, cl;
long double x_tmp, y_tmp, z_tmp;




si = sin(-inc);
ci = cos(-inc);
sl = sin(-lan);
cl = cos(-lan);



for (int i=0; i<2; i++)
	{
	X_planar[i] = X[i];
	Y_planar[i] = Y[i];
	Z_planar[i] = Z[i];
	}

// external rotation back to standard tilting along x-axis


for (int i=0; i<2; i++)
	{
	x_tmp = sl * Y[i] + cl * X[i];
	y_tmp = cl * Y[i] - sl * X[i];

	X_planar[i] = x_tmp;
	Y_planar[i] = y_tmp;
	Z_planar[i] = Z[i];
	}


// tilt along x-axis

for (int i=0; i<2; i++)
	{
	y_tmp = si * Z_planar[i] + ci * Y_planar[i];
	z_tmp = ci * Z_planar[i] - si * Y_planar[i];

	Y_planar[i] = y_tmp;
	Z_planar[i] = z_tmp;
	}

// external rotation  to correct lan


sl = sin(lan);
cl = cos(lan);

for (int i=0; i<2; i++)
	{
	x_tmp = sl * Y_planar[i] + cl * X_planar[i];
	y_tmp = cl * Y_planar[i] - sl * X_planar[i];

	X_planar[i] = x_tmp;
	Y_planar[i] = y_tmp;
	}



}

long double CoordinateObject::get_x_planar(int index)
{
return X_planar[index];
}

long double CoordinateObject::get_y_planar(int index)
{
return Y_planar[index];
}



OrbitalSystem::OrbitalSystem()
{
int num_slots = 25;

planar_orbit = true;
collision_flag = false;
do_collisions = false;
collision_index_1 = 0;
collision_index_2 = 0;

ang_01 = 0.0;
ang_02 = 0.0;
ang_03 = 0.0;
ang_12 = 0.0;
ang_13 = 0.0;
ang_23 = 0.0;
lan_01 = 0.0;
lan_02 = 0.0;
lan_03 = 0.0;
lan_12 = 0.0;
lan_13 = 0.0;
lan_23 = 0.0;

x = new long double[num_slots];
y = new long double[num_slots];
z = new long double[num_slots];

vx = new long double[num_slots];
vy = new long double[num_slots];
vz = new long double[num_slots];

M = new long double[num_slots];
R = new long double[num_slots];
L = new long double[num_slots];

orbit_counter = new int[num_slots];
tie_index = new int[num_slots];
active_flag = new bool[num_slots];

designation = new string[num_slots];

for (int i=0; i<num_slots;i++)
	{
	active_flag[i] = true;
	}
}

void OrbitalSystem::rotate_X(int id, double ang)
{
long double sa, ca;
long double tmp_y, tmp_z;

sa = sin(ang);
ca = cos(ang);

tmp_y = ca * y[id] + sa * z[id];
tmp_z = -sa * y[id] + ca * z[id];

y[id] = tmp_y;
z[id] = tmp_z; 

tmp_y = ca * vy[id] + sa * vz[id];
tmp_z = -sa * vy[id] + ca * vz[id];

vy[id] = tmp_y;
vz[id] = tmp_z;
}

void OrbitalSystem::rotate_Y(int id, double ang)
{
long double sa, ca;
long double tmp_x, tmp_z;

sa = sin(ang);
ca = cos(ang);

tmp_x = ca * x[id] + sa * z[id];
tmp_z = -sa * x[id] + ca * z[id];

x[id] = tmp_x;
z[id] = tmp_z; 

tmp_x = ca * vx[id] + sa * vz[id];
tmp_z = -sa * vx[id] + ca * vz[id];

vx[id] = tmp_x;
vz[id] = tmp_z; 

}

void OrbitalSystem::rotate_Z(int id, double ang)
{
long double sa, ca;
long double tmp_x, tmp_y;

sa = sin(ang);
ca = cos(ang);

tmp_x = sa * y[id] + ca * x[id];
tmp_y = ca * y[id] - sa * x[id];
x[id] = tmp_x;
y[id] = tmp_y;

tmp_x = sa * vy[id] + ca * vx[id];
tmp_y = ca * vy[id] - sa * vx[id];

vx[id] = tmp_x;
vy[id] = tmp_y;
}

void OrbitalSystem::tilt_plane(int id, double inc, double lan)
{

rotate_Z(id, -lan);
rotate_X(id, inc);
rotate_Z(id, lan);
}

void OrbitalSystem::move_to_CoG(int id1, int id2)
{
long double M_T, x_T, y_T, z_T;
long double vx_T, vy_T, vz_T;

M_T = M[id1] + M[id2];

x_T = (M[id1] * x[id1] + M[id2] * x[id2])/M_T;
y_T = (M[id1] * y[id1] + M[id2] * y[id2])/M_T;
z_T = (M[id1] * z[id1] + M[id2] * z[id2])/M_T;

vx_T = (M[id1] * vx[id1] + M[id2] * vx[id2])/M_T;
vy_T = (M[id1] * vy[id1] + M[id2] * vy[id2])/M_T;
vz_T = (M[id1] * vz[id1] + M[id2] * vz[id2])/M_T;

x[id1] -= x_T;
y[id1] -= y_T;
z[id1] -= z_T;

x[id2] -= x_T;
y[id2] -= y_T;
z[id2] -= z_T;

vx[id1] -= vx_T;
vy[id1] -= vy_T;
vz[id1] -= vz_T;

vx[id2] -= vx_T;
vy[id2] -= vy_T;
vz[id2] -= vz_T;

}


void OrbitalSystem::adjust_momentum_balance()
{
long double M_T, px, py, pz;
long double vx_T, vy_T, vz_T;
int i;

M_T=0.0; px=0.0; py= 0.0; pz = 0.0;

for (i=0; i<n_bodies; i++)
	{
	M_T += M[i];
	px += M[i] * vx[i];
	py += M[i] * vy[i];
	pz += M[i] * vz[i];
	}

vx_T = px/M_T;
vy_T = py/M_T;
vz_T = pz/M_T;


for (i=0; i<n_bodies; i++)
	{
	vx[i] -= vx_T;
	vy[i] -= vy_T;
	vz[i] -= vz_T;
	}


}

void OrbitalSystem::adjust_momentum_balance(int id1, int id2)
{
long double M_T, vx_T, vy_T, vz_T;

M_T = M[id1] + M[id2];

vx_T = (M[id1] * vx[id1] + M[id2] * vx[id2])/M_T;
vy_T = (M[id1] * vy[id1] + M[id2] * vy[id2])/M_T;
vz_T = (M[id1] * vz[id1] + M[id2] * vz[id2])/M_T;


vx[id1] -= vx_T;
vy[id1] -= vy_T;
vz[id1] -= vz_T;

vx[id2] -= vx_T;
vy[id2] -= vy_T;
vz[id2] -= vz_T;
}


long double OrbitalSystem::get_plane_tilt_angle(int id1, int id2)
{
if (id1 == 0)
	{
	if (id2 ==1) {return ang_01;}
	else if (id2 == 2) {return ang_02;}
	else if (id2 == 3) {return ang_03;}
	}
else if (id1 == 1)
	{
	if (id2 ==0) {return ang_01;}
	else if (id2 == 2) {return ang_12;}
	else if (id2 == 3) {return ang_13;}
	}
else if (id1 == 2)
	{
	if (id2 ==0) {return ang_02;}
	else if (id2 == 1) {return ang_12;}
	else if (id2 == 3) {return ang_23;}
	}
else if (id1 == 3)
	{
	if (id2 ==0) {return ang_03;}
	else if (id2 == 1) {return ang_13;}
	else if (id2 == 2) {return ang_23;}
	}

return 0.0;
}

long double OrbitalSystem::get_lan(int id1, int id2)
{
if (id1 == 0)
	{
	if (id2 ==1) {return lan_01;}
	else if (id2 == 2) {return lan_02;}
	else if (id2 == 3) {return lan_03;}
	}
else if (id1 == 1)
	{
	if (id2 ==0) {return lan_01;}
	else if (id2 == 2) {return lan_12;}
	else if (id2 == 3) {return lan_13;}
	}
else if (id1 == 2)
	{
	if (id2 ==0) {return lan_02;}
	else if (id2 == 1) {return lan_12;}
	else if (id2 == 3) {return lan_23;}
	}
else if (id1 == 3)
	{
	if (id2 ==0) {return lan_03;}
	else if (id2 == 1) {return lan_13;}
	else if (id2 == 2) {return lan_23;}
	}

return 0.0;
}

void OrbitalSystem::initialize (Star central_sun, Planet planet)
{

int i;

long double M_T, px, py;
long double vx_T, vy_T;
long double e_Tx, e_Ty, r, R_S, v_orbit ;

n_bodies = 2;

// Star
M[0] = central_sun.get_mass();
R[0] = central_sun.get_radius();
x[0] = 0.0;
y[0] = 0.0;
z[0] = 0.0;
vx[0] = 0.0;
vy[0] = 0.0;
vz[0] = 0.0;
orbit_counter[0] = 0;
designation[0] = central_sun.get_designation();


//Planet

M[1] = planet.get_mass();
R[1] = planet.get_radius();
x[1] = 0.0;
y[1] = planet.get_periapsis();
z[1] = 0.0;
vx[1] = planet.get_speed_periapsis();
vy[1] = 0.0; 
vz[1] = 0.0;
designation[1] = planet.get_designation();
orbit_counter[1] = 0;

if (init_at_apoapsis == true)
	{
	cout << "Initializing at apoapsis" << endl;
	y[1] = planet.get_apoapsis();
	vx[1] = planet.get_speed_apoapsis();
	}
else
	{
	cout << "Initializing at periapsis" << endl;
	y[1] = planet.get_periapsis();
	vx[1] = planet.get_speed_periapsis();
	}


// now adjust momentum balance

M_T=0.0; px=0.0; py= 0.0;

for (i=0; i<n_bodies; i++)
	{
	M_T += M[i];
	px += M[i] * vx[i];
	py += M[i] * vy[i];
	}

vx_T = px/M_T;
vy_T = py/M_T;


for (i=0; i<n_bodies; i++)
	{
	vx[i] -= vx_T;
	vy[i] -= vy_T;
	}



for (i=0; i<n_bodies; i++)
	{
	vx_T += vx[i] * M[i];
	vy_T += vy[i] * M[i]; 
	}

//cout << vx_T << " " << vy_T << endl;

// compute the initial angular momenta in GR normalization

L[0] = 0.0;

for (i=1; i< n_bodies; i++)
	{
	r = sqrt(pow(x[i] - x[0],2.0) + pow(y[i] - y[0],2.0)); 
	e_Tx = (y[i] - y[0])/r;
	e_Ty = (x[0] - x[i])/r;
	R_S = 2.0 * G * M[0] /pow(c_light,2.0);
	v_orbit = sqrt(G * M[0]/r);
	L[i] =  abs((e_Tx * vx[i] + e_Ty * vy[i])/v_orbit * r/R_S);
	}



cout << "2-body orbital solver initialized!" << endl;

//cout << vz[0] << " " << vz[1] << endl;
}


void OrbitalSystem::initialize (Star central_sun, Star secondary_star, Planet planet, int sys_type)
{

int i;

long double e_Tx, e_Ty, r, R_S, v_orbit ;
long double plane_tilt, lan;

n_bodies = 3;


// stable 3-body systems can be set up in three ways:

// 1: the planet circles the first star, which in turn orbits the companion in a 'long year'
// 2: the planet circles the barycenter of both stars orbiting each other in the center
// 3: the planet is on the L4 or L5 Lagrange point of the 3-body system

// in the first and third case, planetary orbital parameters refer to the first star
// and long year parameters are declared separately, in the second case orbital parameters refer to
// the barycenter
// in all cases we init with everything aligned at periapsis

double vx_boost = 0.0;
double y_offset = 0.0;
double tmp_y, tmp_vx;
double R_tmp, sign, ecc_tmp, v_tmp, vR_tmp, vR_factor;



if (sys_type == 1)
	{
	if (init_at_apoapsis == true)
		{	
		cout << "Initializing at apoapsis" << endl;
		vx_boost = planet.get_speed_apoapsis_longyear();
		y_offset = planet.get_apoapsis_longyear();
		}
	else
		{
		cout << "Initializing at periapsis" << endl;
		vx_boost = planet.get_speed_periapsis_longyear();
		y_offset = planet.get_periapsis_longyear();
		}
	



	// Star
	M[0] = central_sun.get_mass();
	R[0] = central_sun.get_radius();
	x[0] = 0.0;
	y[0] = y_offset;
	z[0] = 0.0;
	vx[0] = vx_boost;
	vy[0] = 0.0;
	vz[0] = 0.0;
	designation[0] = central_sun.get_designation();
	orbit_counter[0] = 0;



	// Planet
	M[1] = planet.get_mass();
	R[1] = planet.get_radius();
	x[1] = 0.0;
	z[1] = 0.0;
	vy[1] = 0.0;
	vz[1] = 0.0; 
	designation[1] = planet.get_designation();
	orbit_counter[1] = 0;

	if (init_at_apoapsis == true)
		{

		tmp_y = planet.get_apoapsis();
		x[1] = sin(apsis_angle) * tmp_y;
		y[1] = cos(apsis_angle) * tmp_y;

		tmp_vx = planet.get_speed_apoapsis();

		if (retrograde) {tmp_vx = - tmp_vx;}

		vx[1] = cos(apsis_angle) * tmp_vx;
		vy[1] = - sin(apsis_angle) * tmp_vx; 

		plane_tilt =  planet.get_inc_longyear();
		if (plane_tilt != 0.0)
			{
			lan = planet.get_lan_longyear();

			tilt_plane(1, -plane_tilt, -lan);

			ang_01 = plane_tilt;
			lan_01 = lan;
			planar_orbit = false;
			}

		y[1] +=y_offset;
		vx[1] += vx_boost;
		}
	else
		{

		tmp_y = planet.get_periapsis();
		x[1] = sin(apsis_angle) * tmp_y;
		y[1] = cos(apsis_angle) * tmp_y;

		tmp_vx = planet.get_speed_periapsis();

		if (retrograde) {tmp_vx = - tmp_vx;}

		vx[1] = cos(apsis_angle) * tmp_vx;
		vy[1] = - sin(apsis_angle) * tmp_vx; 

		plane_tilt =  planet.get_inc_longyear();

		if (plane_tilt != 0.0)
			{

			lan = planet.get_lan_longyear();

			tilt_plane(1, -plane_tilt, -lan);

			ang_01 = plane_tilt;
			lan_01 = lan;
			planar_orbit = false;
			}

		y[1] += y_offset;
		vx[1] += vx_boost;
		}


	// Companion

	M[2] = secondary_star.get_mass();
	R[2] = secondary_star.get_radius();
	x[2] = 0.0;
	y[2] = 0.0;
	z[2] = 0.0;
	vx[2] = 0.0;
	vy[2] = 0.0;
	vz[2] = 0.0;
	designation[2] = secondary_star.get_designation();
	orbit_counter[2] = 0;
	


	}
else if (sys_type == 2)
	{

	//cout << "SYS Type 2 " << endl;
	

	// Star
	M[0] = central_sun.get_mass();
	R[0] = central_sun.get_radius();
	x[0] = 0.0;
	y[0] = 0.0;
	z[0] = 0.0;
	vx[0] = 0.0;
	vy[0] = 0.0;
	vz[0] = 0.0;
	designation[0] = central_sun.get_designation();
	orbit_counter[0] = 0;

	// Binary

	M[2] = secondary_star.get_mass();
	R[2] = secondary_star.get_radius();
	x[2] = 0.0;
	y[2] = 0.0;
	z[2] = 0.0;
	vx[2] = 0.0;
	vy[2] = 0.0;
	vz[2] = 0.0;
	designation[2] = secondary_star.get_designation();
	orbit_counter[2] = 0;

	if (init_at_apoapsis == true)
		{
		y[2] = planet.get_apoapsis_binary();
		vx[2] = planet.get_speed_apoapsis_binary();

		}
	else
		{
		y[2] = planet.get_periapsis_binary();
		vx[2] = planet.get_speed_periapsis_binary();

		}

	// now adjust momentum balance between binaries


	move_to_CoG(0,2);


	// now tilt binaries

	plane_tilt =  planet.get_inc_binary();

	if (plane_tilt != 0.0)
		{

		lan =  planet.get_lan_binary();

		tilt_plane(0, plane_tilt, lan);
		tilt_plane(2, plane_tilt, lan);


		ang_02 = plane_tilt;
		planar_orbit = false;
		}

	// Planet
	M[1] = planet.get_mass();
	R[1] = planet.get_radius();
	x[1] = 0.0;
	z[1] = 0.0;
	vy[1] = 0.0;
	designation[1] = planet.get_designation();
	vz[1] = 0.0; 
	orbit_counter[1] = 0;

	if (init_at_apoapsis == true)
		{

		tmp_y = planet.get_apoapsis();
		x[1] = sin(apsis_angle) * tmp_y;
		y[1] = cos(apsis_angle) * tmp_y;

		tmp_vx = planet.get_speed_apoapsis();

		if (retrograde) {tmp_vx = - tmp_vx;}

		vx[1] = cos(apsis_angle) * tmp_vx;
		vy[1] = - sin(apsis_angle) * tmp_vx; 

		}
	else
		{

		tmp_y = planet.get_periapsis();
		x[1] = sin(apsis_angle) * tmp_y;
		y[1] = cos(apsis_angle) * tmp_y;

		tmp_vx = planet.get_speed_periapsis();

		if (retrograde) {tmp_vx = - tmp_vx;}

		vx[1] = cos(apsis_angle) * tmp_vx;
		vy[1] = - sin(apsis_angle) * tmp_vx; 


		}




	}
else if (sys_type == 3)
	{
	cout << "Initializing on Lagrange point " << planet.get_lagrange_point() << endl;
	if (init_at_apoapsis == true)
		{	
		cout << "Initializing at apoapsis" << endl;
		vx_boost = planet.get_speed_apoapsis_longyear();
		y_offset = planet.get_apoapsis_longyear();
		}
	else
		{
		cout << "Initializing at periapsis" << endl;
		vx_boost = planet.get_speed_periapsis_longyear();
		y_offset = planet.get_periapsis_longyear();
		}
	



	// Star
	M[0] = central_sun.get_mass();
	R[0] = central_sun.get_radius();
	x[0] = 0.0;
	y[0] = y_offset;
	z[0] = 0.0;
	vx[0] = vx_boost;
	vy[0] = 0.0;
	vz[0] = 0.0;
	designation[0] = central_sun.get_designation();
	orbit_counter[0] = 0;



	// Planet
	M[1] = planet.get_mass();
	R[1] = planet.get_radius();
	z[1] = 0.0;
	vz[1] = 0.0; 
	designation[1] = planet.get_designation();
	orbit_counter[1] = 0;

	if (init_at_apoapsis == true) {sign = -1.0;} else {sign = 1.0;}

	double phi = 0.0;
	if (planet.get_lagrange_point() == "L4") 
		{
		phi = 1.04719753333333;
		vR_factor = 0.81 * sign;
		}
	else 
		{
		phi = -1.04719753333333;
		vR_factor = -0.81  * sign;
		}

	phi += planet.get_Ldphi();


	ecc_tmp = planet.get_eccentricity();
	R_tmp = (planet.get_semimajor() * ( 1.0 - ecc_tmp * ecc_tmp)) / (1.0 + sign *  ecc_tmp * cos(phi));

	R_tmp += planet.get_LdR();

	v_tmp = vx[0] * y[0]/R_tmp * (1.0 + 0.2 * ecc_tmp);
	vR_tmp = vR_factor * ecc_tmp * v_tmp;

	z[1] += planet.get_Ldz();


	x[1] = R_tmp * sin(phi);
	y[1] = R_tmp * cos(phi);

	vx[1] = vR_tmp * sin(phi) + v_tmp * cos(phi);
	vy[1] = vR_tmp * cos(phi) -v_tmp * sin(phi);

	// Companion

	M[2] = secondary_star.get_mass();
	R[2] = secondary_star.get_radius();
	x[2] = 0.0;
	y[2] = 0.0;
	z[2] = 0.0;
	vx[2] = 0.0;
	vy[2] = 0.0;
	vz[2] = 0.0;
	designation[2] = secondary_star.get_designation();
	orbit_counter[2] = 0;
	


	}


adjust_momentum_balance();




// compute the initial angular momenta in GR normalization

L[0] = 0.0;

for (i=1; i< n_bodies; i++)
	{
	r = sqrt(pow(x[i] - x[0],2.0) + pow(y[i] - y[0],2.0)); 
	e_Tx = (y[i] - y[0])/r;
	e_Ty = (x[0] - x[i])/r;
	R_S = 2.0 * G * M[0] /pow(c_light,2.0);
	v_orbit = sqrt(G * M[0]/r);
	L[i] =  abs((e_Tx * vx[i] + e_Ty * vy[i])/v_orbit * r/R_S);
	}



cout << "3-body orbital solver for two stars and planet initialized!" << endl;;



}




void OrbitalSystem::initialize (Star central_sun, Planet planet, Planet moon)
{

int i;

long double e_Tx, e_Ty, r, R_S, v_orbit ;

n_bodies = 3;

// planet encircles the star, moon orbital parameters refer to planet as center or orbits on Lagrange point

double vx_boost = 0.0;
double y_offset = 0.0;
double tmp_y,  tmp_vx;
double plane_tilt, lan;
double R_tmp, sign, ecc_tmp, v_tmp, vR_tmp, vR_factor;

if (init_at_apoapsis == true)
	{	
	cout << "Initializing at apoapsis" << endl;
	vx_boost = planet.get_speed_apoapsis();
	y_offset = planet.get_apoapsis();
	}
else
	{
	cout << "Initializing at periapsis" << endl;
	vx_boost = planet.get_speed_periapsis();
	y_offset = planet.get_periapsis();
	}

if (moon.has_lagrange_orbit())
	{
	cout << "Initializing moon on Lagrange point " << moon.get_lagrange_point() << endl;

	// Planet
	M[1] = planet.get_mass();
	R[1] = planet.get_radius();
	x[1] = 0.0;
	y[1] = y_offset;
	z[1] = 0.0;
	vx[1] = vx_boost;
	vy[1] = 0.0;
	vz[1] = 0.0;
	designation[1] = planet.get_designation();
	orbit_counter[1] = 0;

	// Moon
	M[2] = moon.get_mass();
	R[2] = moon.get_radius();
	z[2] = 0.0;
	vz[2] = 0.0; 
	designation[2] = moon.get_designation();
	orbit_counter[2] = 0;

	if (init_at_apoapsis == true) {sign = -1.0;} else {sign = 1.0;}

	double phi = 0.0;
	if (moon.get_lagrange_point() == "L4") 
		{
		phi = 1.04719753333333;
		vR_factor = 0.81 * sign;
		}
	else 
		{
		phi = -1.04719753333333;
		vR_factor = -0.81  * sign;
		}

	phi += moon.get_Ldphi();


	ecc_tmp = planet.get_eccentricity();
	R_tmp = (planet.get_semimajor() * ( 1.0 - ecc_tmp * ecc_tmp)) / (1.0 + sign *  ecc_tmp * cos(phi));

	R_tmp += moon.get_LdR();

	v_tmp = vx[1] * y[1]/R_tmp * (1.0 + 0.2 * ecc_tmp);
	vR_tmp = vR_factor * ecc_tmp * v_tmp;

	z[2] += moon.get_Ldz();

	x[2] = R_tmp * sin(phi);
	y[2] = R_tmp * cos(phi);

	vx[2] = vR_tmp * sin(phi) + v_tmp * cos(phi);
	vy[2] = vR_tmp * cos(phi) -v_tmp * sin(phi);

	// Star

	M[0] = central_sun.get_mass();
	R[0] = central_sun.get_radius();
	x[0] = 0.0;
	y[0] = 0.0;
	z[0] = 0.0;
	vx[0] = 0.0;
	vy[0] = 0.0;
	vz[0] = 0.0;
	designation[0] = central_sun.get_designation();
	orbit_counter[0] = 0;

	}
else
	{

	// Planet
	M[1] = planet.get_mass();
	R[1] = planet.get_radius();
	x[1] = 0.0;
	y[1] = y_offset;
	z[1] = 0.0;
	vx[1] = vx_boost;
	vy[1] = 0.0;
	vz[1] = 0.0;
	designation[1] = planet.get_designation();
	orbit_counter[1] = 0;

	// Moon
	M[2] = moon.get_mass();
	R[2] = moon.get_radius();
	x[2] = 0.0;
	z[2] = 0.0;
	vy[2] = 0.0;
	vz[2] = 0.0; 
	designation[2] = moon.get_designation();
	orbit_counter[2] = 0;

	if (init_at_apoapsis == true)
		{

		tmp_y = moon.get_apoapsis();
		x[2] = sin(moon_apsis_angle) * tmp_y;
		y[2] = cos(moon_apsis_angle) * tmp_y;

		tmp_vx = moon.get_speed_apoapsis();

		if (moon_retrograde) {tmp_vx = - tmp_vx;}
		vx[2] = cos(moon_apsis_angle) * tmp_vx;
		vy[2] = - sin(moon_apsis_angle) * tmp_vx; 

		plane_tilt =  moon.get_inc_planet();

		if (plane_tilt != 0.0)
			{

			lan = moon.get_lan_planet();

			tilt_plane(2, -plane_tilt, lan);

			ang_12 = plane_tilt;
			lan_12 = lan;
			planar_orbit = false;
			}

		y[2]+=y_offset;
		vx[2] += vx_boost;
		}
	else
		{

		tmp_y = moon.get_periapsis();
		x[2] = sin(moon_apsis_angle) * tmp_y;
		y[2] = cos(moon_apsis_angle) * tmp_y;

		tmp_vx = moon.get_speed_periapsis();

		if (moon_retrograde) {tmp_vx = - tmp_vx;}

		vx[2] = cos(moon_apsis_angle) * tmp_vx;
		vy[2] = - sin(moon_apsis_angle) * tmp_vx; 

		plane_tilt =  moon.get_inc_planet();

		if (plane_tilt != 0.0)
			{

			lan = moon.get_lan_planet();

			tilt_plane(2, -plane_tilt, lan);

			ang_12 = plane_tilt;
			lan_12 = lan;
			planar_orbit = false;
			}

		y[2]+=y_offset;
		vx[2] += vx_boost;
		}


	// Star

	M[0] = central_sun.get_mass();
	R[0] = central_sun.get_radius();
	x[0] = 0.0;
	y[0] = 0.0;
	z[0] = 0.0;
	vx[0] = 0.0;
	vy[0] = 0.0;
	vz[0] = 0.0;
	designation[0] = central_sun.get_designation();
	orbit_counter[0] = 0;

	}

// now adjust momentum balance

adjust_momentum_balance();



// compute the initial angular momenta in GR normalization

L[0] = 0.0;

for (i=1; i< n_bodies; i++)
	{
	r = sqrt(pow(x[i] - x[0],2.0) + pow(y[i] - y[0],2.0)); 
	e_Tx = (y[i] - y[0])/r;
	e_Ty = (x[0] - x[i])/r;
	R_S = 2.0 * G * M[0] /pow(c_light,2.0);
	v_orbit = sqrt(G * M[0]/r);
	L[i] =  abs((e_Tx * vx[i] + e_Ty * vy[i])/v_orbit * r/R_S);
	}



cout << "3-body orbital solver for star, planet and moon initialized!" << endl;;

	
}




void OrbitalSystem::initialize (Star central_sun, Star secondary_star, Planet planet, Planet moon, int sys_type)
{

int i;

long double e_Tx, e_Ty, r, R_S, v_orbit ;
long double plane_tilt, lan;

n_bodies = 4;


// stable 4-body systems can be set up in two ways:

// 1: the planet circles the first star, which in turn orbits the companion in a 'long year'
// 2: the planet circles the barycenter of both stars orbiting each other in the center
// in all cases the moon follows the planet
// in the first case, planetary orbital parameters refer to the first star
// and long year parameters are declared separately, in the second case orbital parameters refer to
// the barycenter
// in all cases we init with everything aligned at periapsis

double vx_boost = 0.0;
double y_offset = 0.0;
double tmp_y, tmp_vx;

if (sys_type == 1)
	{
	if (init_at_apoapsis == true)
		{	
		cout << "Initializing at apoapsis" << endl;
		vx_boost = planet.get_speed_apoapsis_longyear();
		y_offset = planet.get_apoapsis_longyear();
		}
	else
		{
		cout << "Initializing at periapsis" << endl;
		vx_boost = planet.get_speed_periapsis_longyear();
		y_offset = planet.get_periapsis_longyear();
		}
	


	// Star
	M[0] = central_sun.get_mass();
	R[0] = central_sun.get_radius();
	x[0] = 0.0;
	y[0] = 0.0;
	z[0] = 0.0;
	vx[0] = 0.0;
	vy[0] = 0.0;
	vz[0] = 0.0;
	designation[0] = central_sun.get_designation();
	orbit_counter[0] = 0;



	// Planet
	M[1] = planet.get_mass();
	R[1] = planet.get_radius();
	x[1] = 0.0;
	y[1] = 0.0;
	z[1] = 0.0;
	vx[1] = 0.0;
	vy[1] = 0.0;
	vz[1] = 0.0; 
	designation[1] = planet.get_designation();
	orbit_counter[1] = 0;

	// Moon

	M[3] = moon.get_mass();
	R[3] = moon.get_radius();
	x[3] = 0.0;
	z[3] = 0.0;
	vy[3] = 0.0;
	vz[3] = 0.0; 
	designation[3] = moon.get_designation();
	orbit_counter[3] = 0;


	if (init_at_apoapsis == true)
		{
		y[3] = moon.get_apoapsis();
		vx[3] = moon.get_speed_apoapsis();

		rotate_Z(3, moon_apsis_angle);

		plane_tilt =  moon.get_inc_planet();

		if (plane_tilt != 0.0)
			{

			lan = moon.get_lan_planet();

			tilt_plane(1, -plane_tilt, lan);
			tilt_plane(3, -plane_tilt, lan);

			ang_13 = plane_tilt;
			lan_13 = lan;
			planar_orbit = false;
			}
		

		move_to_CoG(1,3);

		tmp_y = planet.get_apoapsis();

		y[1] += tmp_y;
		y[3] += tmp_y;

		tmp_vx = planet.get_speed_apoapsis();
		if (retrograde) {tmp_vx = - tmp_vx;}

		vx[1] += tmp_vx;
		vx[3] += tmp_vx;

		rotate_Z(1, apsis_angle);
		rotate_Z(3, apsis_angle); 	
		
		}
	else
		{
		
		y[3] = moon.get_periapsis();
		vx[3] = moon.get_speed_periapsis();

		//cout << y[3] <<  " " << vx[3] << endl;

		rotate_Z(3, moon_apsis_angle);

		plane_tilt =  moon.get_inc_planet();

		if (plane_tilt != 0.0)
			{

			lan = moon.get_lan_planet();

			tilt_plane(1, -plane_tilt, lan);
			tilt_plane(3, -plane_tilt, lan);

			ang_13 = plane_tilt;
			lan_13 = lan;
			planar_orbit = false;
			}
		

		move_to_CoG(1,3);

		tmp_y = planet.get_periapsis();

		y[1] += tmp_y;
		y[3] += tmp_y;

		tmp_vx = planet.get_speed_periapsis();
		if (retrograde) {tmp_vx = - tmp_vx;}

		vx[1] += tmp_vx;
		vx[3] += tmp_vx;

		//cout << y[3] <<  " " << vx[3] << endl;

		rotate_Z(1, apsis_angle);
		rotate_Z(3, apsis_angle); 
		}

	plane_tilt = planet.get_inc_longyear();

	// tilt all else
	if (plane_tilt !=0.0)
		{
		lan = planet.get_lan_longyear();

		tilt_plane(0, plane_tilt, -lan);
		tilt_plane(1, plane_tilt, -lan);
		tilt_plane(3, plane_tilt, -lan);

		
		}

	y[0] += y_offset;
	y[1] += y_offset;
	y[3] += y_offset;

	vx[0] += vx_boost;
	vx[1] += vx_boost;
	vx[3] += vx_boost;


	if (plane_tilt !=0.0)
		{
		lan = planet.get_lan_longyear();

		tilt_plane(0, -plane_tilt, -lan);
		tilt_plane(1, -plane_tilt, -lan);
		tilt_plane(3, -plane_tilt, -lan);

		ang_02 = plane_tilt;
		ang_12 = plane_tilt;
		ang_23 = plane_tilt;
			
		lan_01 = lan;
		lan_12 = lan;
		lan_23 = lan;
		planar_orbit = false;
		
		}

		//cout << y[3] <<  " " << vx[3] << endl;


	// Companion

	M[2] = secondary_star.get_mass();
	R[2] = secondary_star.get_radius();
	x[2] = 0.0;
	y[2] = 0.0;
	z[2] = 0.0;
	vx[2] = 0.0;
	vy[2] = 0.0;
	vz[2] = 0.0;
	designation[2] = secondary_star.get_designation();
	orbit_counter[2] = 0;

	}
else if (sys_type == 2)
	{

	//cout << "SYS Type 2 " << endl;
	

	// Star
	M[0] = central_sun.get_mass();
	R[0] = central_sun.get_radius();
	x[0] = 0.0;
	y[0] = 0.0;
	z[0] = 0.0;
	vx[0] = 0.0;
	vy[0] = 0.0;
	vz[0] = 0.0;
	designation[0] = central_sun.get_designation();
	orbit_counter[0] = 0;

	// Binary

	M[2] = secondary_star.get_mass();
	R[2] = secondary_star.get_radius();
	x[2] = 0.0;
	y[2] = 0.0;
	z[2] = 0.0;
	vx[2] = 0.0;
	vy[2] = 0.0;
	vz[2] = 0.0;
	designation[2] = secondary_star.get_designation();
	orbit_counter[2] = 0;

	if (init_at_apoapsis == true)
		{
		y[2] = planet.get_apoapsis_binary();
		vx[2] = planet.get_speed_apoapsis_binary();

		}
	else
		{
		y[2] = planet.get_periapsis_binary();
		vx[2] = planet.get_speed_periapsis_binary();

		}

	// now adjust momentum balance between binaries

	move_to_CoG(0, 2);

	plane_tilt =  planet.get_inc_binary();

	if (plane_tilt != 0.0)
		{

		lan =  planet.get_lan_binary();

		tilt_plane(0, plane_tilt, lan);
		tilt_plane(2, plane_tilt, lan);


		ang_02 = plane_tilt;
		planar_orbit = false;
		}


	// Planet
	M[1] = planet.get_mass();
	R[1] = planet.get_radius();
	x[1] = 0.0;
	y[1] = 0.0;
	z[1] = 0.0;
	vy[1] = 0.0;
	vx[1] = 0.0;
	vz[1] = 0.0; 
	designation[1] = planet.get_designation();
	orbit_counter[1] = 0;

	// Moon
	M[3] = moon.get_mass();
	R[3] = moon.get_radius();
	x[3] = 0.0;
	z[3] = 0.0;
	vy[3] = 0.0;
	vz[3] = 0.0;
	designation[3] = moon.get_designation();
	orbit_counter[3] = 0;

	if (init_at_apoapsis == true)
		{

		y[3] = moon.get_apoapsis();
		vx[3] = moon.get_speed_apoapsis();

		rotate_Z(3, moon_apsis_angle);

		plane_tilt =  moon.get_inc_planet();

		if (plane_tilt != 0.0)
			{

			lan = moon.get_lan_planet();

			tilt_plane(1, -plane_tilt, lan);
			tilt_plane(3, -plane_tilt, lan);

			ang_13 = plane_tilt;
			lan_13 = lan;
			planar_orbit = false;
			}
		

		move_to_CoG(1,3);

		tmp_y = planet.get_apoapsis();

		y[1] += tmp_y;
		y[3] += tmp_y;

		tmp_vx = planet.get_speed_apoapsis();
		if (retrograde) {tmp_vx = - tmp_vx;}

		vx[1] += tmp_vx;
		vx[3] += tmp_vx;

		rotate_Z(1, apsis_angle);
		rotate_Z(3, apsis_angle); 


		
		}
	else
		{


		y[3] = moon.get_periapsis();
		vx[3] = moon.get_speed_periapsis();

		rotate_Z(3, moon_apsis_angle);

		plane_tilt =  moon.get_inc_planet();

		if (plane_tilt != 0.0)
			{

			lan = moon.get_lan_planet();

			tilt_plane(1, -plane_tilt, lan);
			tilt_plane(3, -plane_tilt, lan);

			ang_13 = plane_tilt;
			lan_13 = lan;
			planar_orbit = false;
			}
		

		move_to_CoG(1,3);

		tmp_y = planet.get_periapsis();

		y[1] += tmp_y;
		y[3] += tmp_y;

		tmp_vx = planet.get_speed_periapsis();
		if (retrograde) {tmp_vx = - tmp_vx;}

		vx[1] += tmp_vx;
		vx[3] += tmp_vx;

		rotate_Z(1, apsis_angle);
		rotate_Z(3, apsis_angle); 


		}




	}
 



adjust_momentum_balance();





// compute the initial angular momenta in GR normalization

L[0] = 0.0;

for (i=1; i< n_bodies; i++)
	{
	r = sqrt(pow(x[i] - x[0],2.0) + pow(y[i] - y[0],2.0)); 
	e_Tx = (y[i] - y[0])/r;
	e_Ty = (x[0] - x[i])/r;
	R_S = 2.0 * G * M[0] /pow(c_light,2.0);
	v_orbit = sqrt(G * M[0]/r);
	L[i] =  abs((e_Tx * vx[i] + e_Ty * vy[i])/v_orbit * r/R_S);
	}



cout << "4-body orbital solver for two stars, planet and moon initialized!" << endl;;

}


int OrbitalSystem::add_body(Body body, int attach_index)
{
//cout << "Attach index " << attach_index << endl;

int index = n_bodies;
double plane_tilt, lan;

M[index] = body.get_mass();
R[index] = body.get_radius();
x[index] = 0.0;
z[index] = 0.0;
vy[index] = 0.0;
vz[index] = 0.0;
designation[index] = body.get_designation();

orbit_counter[index] = 0;

if (body.get_hyperbolic_trajectory())
	{
	y[index] = body.get_distance();
	vx[index] = body.get_initial_velocity() * cos(body.get_flight_path_angle());
	vy[index] = -body.get_initial_velocity() * sin(body.get_flight_path_angle());
	rotate_Z(index, body.get_apsis_angle());

	plane_tilt =  body.get_inclination();

	if (plane_tilt != 0.0)
		{
		lan = body.get_lan();
		tilt_plane(index, -plane_tilt, lan);
		planar_orbit = false;
		}

	x[index]+= x[attach_index];
	y[index]+= y[attach_index];
	z[index]+= z[attach_index];

	vx[index]+= vx[attach_index];
	vy[index]+= vy[attach_index];
	vz[index]+= vz[attach_index];
	}
else
	{

	if (init_at_apoapsis == true)
		{
		y[index] =  body.get_apoapsis();
		vx[index] =  body.get_speed_apoapsis();
	
		rotate_Z(index, body.get_apsis_angle());

		plane_tilt =  body.get_inclination();

		if (plane_tilt != 0.0)
			{
			lan = body.get_lan();
			tilt_plane(index, -plane_tilt, lan);
			planar_orbit = false;
			}

		x[index]+= x[attach_index];
		y[index]+= y[attach_index];
		z[index]+= z[attach_index];

		vx[index]+= vx[attach_index];
		vy[index]+= vy[attach_index];
		vz[index]+= vz[attach_index];

		}
	else
		{
		y[index] =  body.get_periapsis();
		vx[index] =  body.get_speed_periapsis();
	
		rotate_Z(index, body.get_apsis_angle());


		plane_tilt =  body.get_inclination();

		if (plane_tilt != 0.0)
			{
			lan = body.get_lan();
			tilt_plane(index, -plane_tilt, lan);
			planar_orbit = false;
			}

		x[index]+= x[attach_index];
		y[index]+= y[attach_index];
		z[index]+= z[attach_index];

		vx[index]+= vx[attach_index];
		vy[index]+= vy[attach_index];
		vz[index]+= vz[attach_index];
		}
	}

adjust_momentum_balance();

n_bodies ++;
if (body.get_hyperbolic_trajectory())
	{
	cout << "Adding body " << body.get_designation() << " passing around " << body.get_central_body() << " to solver." << endl;
	}
else
	{
	cout << "Adding body " << body.get_designation() << " orbiting around " << body.get_central_body() << " to solver." << endl;
	}

return n_bodies -1;
}


long double OrbitalSystem::get_force(long double X, long double Y, int i, int j)
{
long double r;

r = sqrt(pow(X,2.0) + pow(Y,2.0));
					
return  G * M[i] * M[j]/pow(r,2.0);

}

long double OrbitalSystem::closest_distance()
{
long double dist_min, dist;

int i, j;

dist_min = 1e20;

for (i=0; i<n_bodies; i++)
	{
	for (j=0; j< n_bodies; j++)
		{
		if (i==j) {continue;}
		dist = sqrt(pow(x[i]-x[j],2.0) + pow(y[i]-y[j],2.0) + pow(z[i]- z[j], 2.0));
		if (dist < dist_min) 
			{dist_min = dist;}
		}
	
	}

return dist_min;


}

long double OrbitalSystem::V(long double L, long double r)
{
long double Mass = 1.0;

return 0.5*(((1.0-(2.0 * Mass)/r)*(1+pow(L,2.0)/pow(r,2.0))) -1.0);
}

long double OrbitalSystem::V_N(long double L, long double r)
{
long double Mass = 1.0;

return - Mass/r  + pow(L, 2.0)/(2.0 * pow(r,2.0));
}


long double OrbitalSystem::GR_correction(int i, int j, long double r, long double X, long double Y)
{

if (GR_corrections == false)
	{
	return 0.0;
	}

long double dV_N, dV;
long double e_Tx, e_Ty;
long double r_s;
long double R_S, v;//, vR;
long double a_N, a_C;
long double correction;

e_Tx = Y/r;
e_Ty = -X/r;


// the Newtonian Forces 

v = e_Tx * vx[i] + e_Ty * vy[i];
//vR = X/r * vx[i] + Y/r * vy[i];

// centrifugal force
a_C = pow(v,2.0)/r;

// gravity
a_N = -G * M[j]/pow(r,2.0);


// the force acting in the Schwarzschild metric is still radial, but given by the ratio of the effective potential differentials

R_S = 2.0 * G * M[j] /pow(c_light,2.0);

// go to Schwarzschild radius coords
r_s = r/R_S;


dV = V(L[i], r_s) - V(L[i], r+ 0.05);
dV_N = V_N(L[i], r_s) - V_N(L[i], r + 0.05);

correction = ((a_C + a_N) * (dV/dV_N -1.0));

return correction;
}


void OrbitalSystem::collision_detection()
{
long double dist;


for (int i=0; i<n_bodies; i++)
	{
	for (int j=0; j< n_bodies; j++)
		{
		if (i==j) {continue;}
		if ((active_flag[i] == false) || (active_flag[j] == false)) {continue;}
		dist = sqrt(pow(x[i]-x[j],2.0) + pow(y[i]-y[j],2.0) + pow(z[i]- z[j], 2.0));
		if (dist < (R[i]  + R[j])) 
			{collision_flag = true; collision_index_1 = i; collision_index_2 = j;}
		//else if (dist < 5.0 * (R[i]  + R[j]))
		//	{cout << "Near miss!" << endl;}
		
		}
	
	}
}


void OrbitalSystem::process_collision()
{


collision_flag = false;



if (M[collision_index_1] > M[collision_index_2])
	{
	M[collision_index_1] += M[collision_index_2];
	active_flag[collision_index_2] = false;
	tie_index[collision_index_2] = collision_index_1;

	cout << "Impact of " << designation[collision_index_2] << " on " << designation[collision_index_1] << endl;
	}
else
	{
	M[collision_index_2] += M[collision_index_1];
	active_flag[collision_index_1] = false;
	tie_index[collision_index_1] = collision_index_2;

	cout << "Impact of " << designation[collision_index_1] << " on " << designation[collision_index_2] << endl;
	}
}

void OrbitalSystem::process_ties()
{
for (int i=0; i<n_bodies;i++)
	{
	if (active_flag[i] == false)
		{
		x[i] = x[tie_index[i]];
		y[i] = y[tie_index[i]];
		z[i] = z[tie_index[i]];
		vx[i] = x[tie_index[i]];
		vy[i] = y[tie_index[i]];
		vz[i] = z[tie_index[i]];
		}
	}
}


void OrbitalSystem::evolve_timestep()
{
long double X,Y, Z;

long double x_old;//, y_old;


long double r;
long double a, ax, ay, az;



int i, j;


for (i=0; i< n_bodies; i++)
	{
	ax = 0.0; ay = 0.0; az = 0.0;
	for (j=0; j< n_bodies; j++)
		{

		if (i==j) {continue;}
		if ((active_flag[i] == false) || (active_flag[j] == false)) {continue;}

		X = x[i] - x[j];
		Y = y[i] - y[j];
		Z = z[i] - z[j];
		r = sqrt(pow(X,2.0) + pow(Y,2.0) + pow(Z, 2.0));	
		a = -G * M[j]/pow(r,2.0) + GR_correction(i,j,r,X,Y);
		ax +=  a* X/r;
		ay +=  a* Y/r;
		az +=  a* Z/r;
		}

	// move body i;

	if (active_flag[i] == false) {continue;}
	
	x_old = x[i];

	vx[i] += ax * Dt;
	vy[i] += ay * Dt;
	vz[i] += az * Dt;
		
	x[i] += vx[i] * Dt;
	y[i] += vy[i] * Dt;
	z[i] += vz[i] * Dt;
	
	if ((x[i] > 0.0) && (x_old <0.0) && (y[i] >0))
		{
		orbit_counter[i]++;

		}
	

	}

}


void OrbitalSystem::evolve_for(double duration)
{
double elapsed = 0;


while (elapsed < duration)
	{
	evolve_timestep();
	elapsed = elapsed + Dt;
	if (do_collisions)
		{
		collision_detection();
		if (collision_flag) 
			{
			process_collision();
			}
		process_ties();
		}
	}


}

void OrbitalSystem::list_state()
{
int i;

cout << "Orbital " << n_bodies << " body state:" << endl;

for (i=0; i< n_bodies; i++)
	{
	cout << "Body " << i+1 <<": x: " << x[i]/1e9 << "  y: " << y[i]/1e9 << endl;
	cout << "      "  <<" vx: " << vx[i]/1000 << " vy: " << vy[i]/1000 << endl;
	}
}


long double OrbitalSystem::norm(long double x, long double y)
{
return sqrt(x*x + y*y);
}

double OrbitalSystem::get_true_anomaly(int id1, int id2)
{

CoordinateObject coords;
double tilt_angle, lan;

tilt_angle = get_plane_tilt_angle(id1, id2);
lan = get_lan(id1, id2);

coords.initialize(x[id1],y[id1],z[id1], x[id2], y[id2], z[id2]);
coords.tilt_by(-tilt_angle, -lan);

long double X = coords.get_x_planar(1) - coords.get_x_planar(0);
long double Y = coords.get_y_planar(1) - coords.get_y_planar(0);

//long double X = x[id2] - x[id1];
//long double Y = y[id2] - y[id1];

double res = atan2(X, Y);

if (res < 0.0) {res = (2.0 * pi) + res;}

return res;
}

double OrbitalSystem::get_true_anomaly_CoG(int id)
{

double res = atan2(x[id], y[id]);

if (res < 0.0) {res = (2.0 * pi) + res;}

return res;
}

double OrbitalSystem::get_true_anomaly_planar(int id1, int id2)
{

long double X = x[id2] - x[id1];
long double Y = y[id2] - y[id1];

double res = atan2(X, Y);

if (res < 0.0) {res = (2.0 * pi) + res;}

return res;
}


double OrbitalSystem::get_year_fraction(int id1, int id2) // id1 is orbited body, id2 orbiting body
{

if (planar_orbit) 
	{return get_true_anomaly_planar(id1, id2) / (2.0 * pi);}

return get_true_anomaly(id1, id2) / (2.0 * pi);
}

double OrbitalSystem::get_distance(int id1, int id2)
{
long double X = x[id2] - x[id1];
long double Y = y[id2] - y[id1];
long double Z = z[id2] - z[id1];

return sqrt(pow(X, 2.0) + pow(Y, 2.0) + pow(Z, 2.0));
}

double OrbitalSystem::get_rel_velocity(int id1, int id2)
{
long double VX = vx[id2] - vx[id1];
long double VY = vy[id2] - vy[id1];
long double VZ = vz[id2] - vz[id1];

return sqrt(pow(VX, 2.0) + pow(VY, 2.0) + pow(VZ, 2.0));
}

double OrbitalSystem::get_semimajor(int id1,int id2)
{
double R, V, mu;

R = get_distance(id1, id2);
V = get_rel_velocity(id1, id2);

mu = G * (M[id1] + M[id2]);

return - mu/ (2.0 * (pow(V,2.0)/2.0 - mu/R  ));

}


double OrbitalSystem::get_year_fraction_CoG(int id) // id is orbiting body, calculation is done around CoG
{
return get_true_anomaly_CoG(id) / (2.0 * pi);
}



double OrbitalSystem::get_day_correction_angle (int id1, int id2) //id2 is outer planet around binary, id1 one of the binary stars
{
long double X = x[id2] - x[id1];
long double Y = y[id2] - y[id1];

//cout << "xid1: " << x[id1] << " yid1: " << y[id1] << endl;


long double norm = sqrt(pow(X, 2.0) + pow(Y, 2.0));

X = X/ norm;
Y = Y/ norm;

//cout << "X: " << X << "Y: " << Y << endl;

long double X0 = x[id2];
long double Y0 = y[id2];

norm = sqrt(pow(X0, 2.0) + pow(Y0, 2.0));

X0 = X0/ norm;
Y0 = Y0/ norm;

//cout << "X0: " << X0 << " Y0: " << Y0 << endl;

long double arg = X * X0 + Y * Y0;
if (arg > 1.0) {arg = 1.0;}
if (arg < -1.0) {arg = -1.0;}



long double ang = acos(arg);



if (vx[id2] * x[id1] + vy[id2] * y[id1] > 0.0) {ang = - ang;} 

//cout << "arg: " << arg << "ang: " << ang << endl;
//cout << "vdotx: " << vx[id2] * x[id1] + vy[id2] * y[id1] << endl;


return ang;
}


double OrbitalSystem::get_elevation_correction_angle(int id1, int id2) //id2 is outer planet around binary, id1 one of the binary stars
{
// we can simply make use of the fact here that the planet must effectively in the z=0 plane

long double Z = z[id1] - z[id2];
long double R = get_distance(id1, id2);


return asin(Z/R);

}


double OrbitalSystem::get_eclipse_factor (int id1, int id2, int id3) // id3 is the planet, id1 radiating body and id2 eclipsing body
{
long double X_to_star = x[id3] - x[id1];
long double Y_to_star = y[id3] - y[id1];
long double Z_to_star = z[id3] - z[id1];

long double dist_to_star = sqrt(pow(X_to_star, 2.0) + pow(Y_to_star, 2.0) + pow(Z_to_star, 2.0));


X_to_star = X_to_star/ dist_to_star;
Y_to_star = Y_to_star/ dist_to_star;
Z_to_star = Z_to_star/ dist_to_star;


long double X_to_ec = x[id3] - x[id2];
long double Y_to_ec = y[id3] - y[id2];
long double Z_to_ec = z[id3] - z[id2];

long double dist_to_ec = sqrt(pow(X_to_ec, 2.0) + pow(Y_to_ec, 2.0) + pow(Z_to_ec, 2.0));



X_to_ec = X_to_ec/ dist_to_ec;
Y_to_ec = Y_to_ec/ dist_to_ec;
Z_to_ec = Z_to_ec/ dist_to_ec;

long double R_star = R[id1]/dist_to_star;
long double R_ec = R[id2] /dist_to_ec;

long double R_critical = R_star + R_ec;
long double Delta = acos(X_to_star * X_to_ec + Y_to_star * Y_to_ec + Z_to_star * Z_to_ec);



if ((Delta < R_critical) && (dist_to_star > dist_to_ec)) 
	{
	long double obs;


	long double mag= (R_star + R_ec - Delta) / (2.0 * R_star);
	if (mag > 1.0) {mag = 1.0;}

	long double R_large;
	long double R_small;

	if (R_star > R_ec)
		{R_large = R_star; R_small = R_ec;}
	else
		{R_large = R_ec; R_small = R_star;}

	if (Delta < R_large - R_small)
		{
		obs = pow(R_ec/R_star,2.0);
		if (obs > 1.0) {obs = 1.0;}
		}
	else
		{


		long double Y = (pow(R_star,2.0) - pow(R_ec, 2.0) + pow(Delta,2.0)) / (2.0 * Delta);
		long double cos_a12 = Y/ R_star;
		long double cos_a22 = (Delta - Y) / R_ec;

		long double A = 0.5 * pow(R_star,2.0) * (2.0 * acos(cos_a12) - sin (2.0 * acos(cos_a12)));
		A+= 0.5 * pow(R_ec,2.0) * (2.0 * acos(cos_a22) - sin (2.0 * acos(cos_a22)));

		obs = A/(pi * pow(R_star,2.0));
		}

	
	return obs;
	}
else
	{return 0.0;}
}

double OrbitalSystem::get_phase_area(int id1, int id2, int id3) // id3 is the planet, id1 radiating body and id2 the illuminated planet
{
double phase_angle;


phase_angle = 2.0 * pi * (get_year_fraction(id2, id3) - get_year_fraction(id1, id2));

return 1.0 - (0.5 * (cos(phase_angle) + 1.0));

}

void OrbitalSystem::save(std::string filename)
{

ofstream file (filename.c_str());
 
if (file.is_open())
  	{
  	file << n_bodies << endl;

	for (int i=0; i< n_bodies; i++)
		{
		file << M[i] << " " << R[i] << endl;
		file << x[i] << " " << y[i] << " " << vx[i] << " " << vy[i] << " " << endl;
		}

	//file << "end" << endl;

  	}
else
  	{
	cout << "Unable to open file " << filename << " !" << endl;
	}



file.close();
}


void OrbitalSystem::resume(std::string filename)
{

std::string element;
ifstream file (filename.c_str());

file >> element;

int n_bodies = atoi (element.c_str());

for (int i=0; i< n_bodies; i++)
	{
	file >> element;
	M[i] = atof (element.c_str());
	file >> element;
	R[i] = atof (element.c_str());
	file >> element;
	x[i] = atof (element.c_str());
	file >> element;
	y[i] = atof (element.c_str());
	file >> element;
	vx[i] = atof (element.c_str());
	file >> element;
	vy[i] = atof (element.c_str());
	}

file.close();
}
