// geology.hxx -- routines to compute geological properties of a planet
//
// Written by Thorsten Renk, started 2018
//
// Copyright (C) 2018  Thorsten Renk - thorsten@science-and-fiction.org 
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301

#ifndef geology_H
#define geology_H

class Geology {

	bool tidal_defined;
	bool solid_convection_flag;
	bool liquid_convection_flag;
	bool is_differentiated;

	bool rho_core_override;
	bool rho_inner_mantle_override;
	bool rho_outer_mantle_override;

	bool profile_verbose;

	int num_layers;


	double R_outer_mantle;
	double R_inner_mantle;
	double R_core;
	double R_total;

	double rho_outer_mantle;
	double rho_inner_mantle;
	double rho_core;
	double rho_mean;

	double M_outer_mantle;
	double M_inner_mantle;
	double M_core;
	double M_total;

	double V_outer_mantle;
	double V_inner_mantle;
	double V_core;
	double V_total;

	double material_1_fraction;

	double rho_outer_mantle_ov;
	double rho_inner_mantle_ov;
	double rho_core_ov;
	double T_surface_ov;

	double surface_gravity;
	double T_surface;

	double radiogenic_heat_factor;
	double radiogenic_heat_total;
	double primordial_heat_total;
	double tidal_heat_total;
	double radiogenic_heat_flux;
	double primordial_heat_flux;
	double tidal_heat_flux;
	double total_heat_flux;
	double love_number;
	double system_age;
	double isotope_abundancy;
	double tidal_dissipation_factor;
	double R_rad_high;
	double R_rad_low;

	double crust_thickness;
	double volcanic_activity_index;
	double max_elevation_estimate;
	double crust_volcanic_transport_fraction;
	double liquidness_factor;

	double depth_profile_resolution;
	
	std::string material_outer_mantle;
	std::string material_inner_mantle;
	std::string material_core;
	std::string crust_material;

	std::string component_1;
	std::string component_2;

	std::string heat_dissipation_mode;
	std::string volcanism_level;

	int n_phasedata_melt_mantle_rock;
	double phasedata_melt_mantle_rock[20][2];

	int n_phasedata_melt_Fe;
	double phasedata_melt_Fe[20][2];

	int n_phasedata_melt_ice;
	double phasedata_melt_ice[20][2];

	int n_phasedata_melt_C;
	double phasedata_melt_C[20][2];

	int n_phasedata_diamond_C;
	double phasedata_diamond_C[20][2];

	int n_data_thermal_conductivity_Fe;
	double data_thermal_conductivity_Fe[20][2];

	double density_by_material(std::string);
	double love_by_material(std::string);

	double density_by_phase(double, double, std::string);
	double get_lambda(double, double, double);
	double lambda_lookup(double, double, double, std::string);

	double compute_primordial_cooling_factor();
	
	bool check_melting(double, double, std::string);
	bool check_diamond(double, double);

	void compute_radiogenic_heat();
	void compute_primordial_heat();
	void compute_profile();
	void compute_crust_volcanic_transport();
	void compute_max_elevation();


	public:

	// constructur
	Geology();

	void initialize (double, double, double, double, double, std::string, std::string, std::string);
	void initialize (double, double, double, double, std::string, std::string);
	void initialize (double, double, double, double, std::string);
	void initialize_undifferentiated (double, double, double, double, std::string, std::string);
	void set_tidal(double, double, double, double);
	void finish_init();
	void set_isotope_abundancy(double);
	void set_parameter(std::string, double);
	void override_density(double, std::string);

	double get_temperature(double, double);
	double get_pressure(double, double);


	void list();
	void show_profile();

};

#endif
