// geology.cxx -- routines to compute geological properties of a planet
//
// Written by Thorsten Renk, started 2018
//
// Copyright (C) 2018  Thorsten Renk - thorsten@science-and-fiction.org 
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301


#include <iostream>
#include <iomanip>
#include <stdlib.h>
#include <math.h>

#include "geology.hxx"
#include "constants.hxx"
#include "utilities.hxx"

using namespace std;


Geology::Geology()
{
R_outer_mantle = 0.0;
rho_outer_mantle = 0.0;
love_number = 0.0;
material_outer_mantle = "";
tidal_heat_total = 0.0;
tidal_heat_flux = 0.0;
isotope_abundancy = 1.0;
depth_profile_resolution = 10000.0;
material_1_fraction = 0.0;
crust_volcanic_transport_fraction = 0.0;
tidal_dissipation_factor = 1.0;
T_surface_ov = -1.0;

n_phasedata_melt_mantle_rock = read_array("data/mantle_rock_phases_melting.dat", phasedata_melt_mantle_rock, 1.0);
n_phasedata_melt_Fe = read_array("data/fe_phases_melting.dat", phasedata_melt_Fe, 1.0);
n_phasedata_melt_ice = read_array("data/ice_phases_melting.dat", phasedata_melt_ice, 1.0);
n_phasedata_melt_C = read_array("data/c_phases_melting.dat", phasedata_melt_C, 1.0);
n_phasedata_diamond_C = read_array("data/c_phases_diamond.dat", phasedata_diamond_C, 1.0);

n_data_thermal_conductivity_Fe = read_array("data/fe_thermal_conductivity.dat", data_thermal_conductivity_Fe, 1.0);

tidal_defined = false;
is_differentiated = false;

rho_core_override = false;
rho_inner_mantle_override = false;
rho_outer_mantle_override = false;

profile_verbose = false;
}


void Geology::override_density(double rho, string layer)
{

if (layer == "core")
	{
	rho_core_ov = rho;
	rho_core_override = true;
	}
else if (layer == "inner_mantle")
	{
	rho_inner_mantle_ov = rho;
	rho_inner_mantle_override = true;
	}
else if (layer == "outer_mantle")
	{
	rho_outer_mantle_ov = rho;
	rho_outer_mantle_override = true;
	}
	
}

double Geology::density_by_material(string material)
{
double rho = 0.0;
double factor = 1.0;

if (num_layers == 1) {factor = 0.5;}

if (material == "nickel_iron") {rho = 7.0 + 0.056 * factor * pow(surface_gravity,2.0);}
else if (material == "silicate_rock") {rho = 2.7 + 0.018 * factor * pow(surface_gravity,2.0);}
else if (material == "carbon") {rho = 2.2 + 0.015 * factor * pow(surface_gravity,2.0);}
else if (material == "sulfur_iron") {rho = 6.0 + 0.056 * factor * pow(surface_gravity,2.0);}
else if (material == "water_ice") {rho = 0.9;}
else if (material == "carborundum") {rho = 3.21 + 0.018 * factor * pow(surface_gravity,2.0);}

return rho * 1e3;
}

double Geology::density_by_phase(double T, double p, string material)
{
double T_melt;

if (material == "water_ice")
	{
	T_melt = interpolate(p, phasedata_melt_ice, n_phasedata_melt_ice);
	if (T > T_melt) {return 1.0;}
	
	if (p > 3e6) {return 0.93;} // ice XI
 	else if (p > 6.5e5) {return 2.4;} // ice X
	else if (p > 2.e4) {return 1.4;} // ice VII
	else if (p > 5e3) {return 1.3;} // ice VI
	else {return 0.9;}

	}


return 0.0;

}

double Geology::love_by_material(string material)
{

if (love_number > 0.0) {return love_number;}

double k = 0.0;


if (material == "silicate_rock") {k = 0.016;}
else if (material == "water_ice") {k = 0.0035;}
else if (material == "carbon") {k = 0.016;} // guess


return k * tidal_dissipation_factor;
}


bool Geology::check_melting(double T, double p, string material)
{
bool flag;
double T_melt;

flag = false;

liquidness_factor = 1.0;

if (material == "silicate_rock")
	{
	T_melt = interpolate(p, phasedata_melt_mantle_rock, n_phasedata_melt_mantle_rock);
	if (T > T_melt) {flag = true;}
	if (T > 1.2 * T_melt) {liquidness_factor = 1.0 + (T - 1.2 * T_melt) * 1.0;}
	}
else if ((material == "nickel_iron") || (material == "sulfur_iron"))
	{
	T_melt = interpolate(p, phasedata_melt_Fe, n_phasedata_melt_Fe);
	//cout << T << " " << T_melt << endl;
	if (T > T_melt) {flag = true;}
	}
else if (material == "carbon")
	{
	T_melt = interpolate(p, phasedata_melt_C, n_phasedata_melt_C);
	if (T > T_melt) {flag = true;}
	}
else if (material == "water_ice")
	{
	T_melt = interpolate(p, phasedata_melt_ice, n_phasedata_melt_ice);
	if (T > T_melt) {flag = true;}
	}
else if (material == "carborundum")
	{
	T_melt = 3100.0 - 40. * (p/10000.);
	if (T > T_melt) {flag = true;}
	}

return flag;

}


bool Geology::check_diamond(double T, double p)
{
double p_diamond;
bool flag;

flag = false;

if ((check_melting (T,p, "carbon") == false) && (T > 4400.))
	{return true;}

p_diamond = interpolate(T, phasedata_diamond_C, n_phasedata_diamond_C);

if (p > p_diamond) {flag = true;}


return flag;
}

void Geology::compute_radiogenic_heat()
{
double decay_th232, decay_u238, decay_u235, decay_k40;


decay_th232 = exp(-system_age/14.0);
decay_u238 = exp(-system_age/4.47);
decay_k40 = exp(-system_age/1.25);
decay_u235 = exp(-system_age/0.704); 

radiogenic_heat_factor = 4.5e-12 * decay_th232 + 7.96e-12 * decay_u238 + 39.5e-12 * decay_k40 + 74.6e-12 * decay_u235;

}

void Geology::compute_primordial_heat()
{
double cooldown_time, initial_heat;

cooldown_time = 6.5 * R_total/R_earth * compute_primordial_cooling_factor();
initial_heat = 33.97e12 * pow(M_total/M_earth,2.0);

primordial_heat_total = initial_heat * exp(-system_age/cooldown_time);


}

void Geology::compute_crust_volcanic_transport()
{
string material;

if (is_differentiated == false) {crust_volcanic_transport_fraction = 0.0; return;}

if (num_layers == 1) {material = material_core;}
else if (num_layers == 2) {material = material_inner_mantle; }
else if (num_layers == 3) {material = material_outer_mantle;}

if (material == "silicate_rock")
	{
	if (total_heat_flux > 0.24)
		{
		crust_volcanic_transport_fraction = (total_heat_flux - 0.24)/total_heat_flux;
		volcanic_activity_index = total_heat_flux / 0.24;
		}
	else
		{
		crust_volcanic_transport_fraction = 0.0;
		volcanic_activity_index = 1.0;
		}
	}
else
	{
	crust_volcanic_transport_fraction = 0.0;
	volcanic_activity_index = 1.0;
	}
}

double Geology::compute_primordial_cooling_factor()
{
double f_core;
double cfc, cfm;

// assume primordial heat vanishes proportional to average conductivity
// lambda = 50 for iron, 200 for graphite (assuming random orientation of flakes)

return 1.0;

if (num_layers == 1)
	{
	if ((material_core == "nickel_iron") || (material_core == "sulfur_iron")) {return 0.05;}
	else if (material_core == "carbon") {return 0.015;}
	}
else if (num_layers == 2)
	{
	if (R_core > 0.5 * R_total)
		{
		f_core  = (R_core - 0.5 * R_total) / (0.5 * R_total);
		cfc = 1.0; cfm = 1.0;
		if ((material_core == "nickel_iron") || (material_core == "sulfur_iron")) {cfc = 0.05;}
		else if (material_core == "carbon") {cfc =  0.015;}

		if ((material_inner_mantle == "nickel_iron") || (material_inner_mantle == "sulfur_iron")) {cfm = 0.05;}
		else if (material_inner_mantle == "carbon") {cfm =  0.015;}
	
		return f_core * cfc + (1.0 - f_core) * cfm;
		}
	else
		{
		if ((material_inner_mantle == "nickel_iron") || (material_inner_mantle == "sulfur_iron")) {return 0.05;}
		else if (material_inner_mantle == "carbon") {return   0.015;}
		}
	}

return 1.0;
}


void Geology::initialize (double M, double R, double age, double T, string material_1, string material_2)
{
double core_vol_fraction;

M_total = M;
R_total = R;
system_age = age;
T_surface = T;

if (T_surface_ov > 0.0) {T_surface = T_surface_ov;}

num_layers = 2;
is_differentiated = true;

surface_gravity = G * M / pow(R_total, 2.0);
V_total = 4./3. * pi * pow(R_total, 3.0);

rho_mean = M_total/ V_total;


material_core = material_1;
material_inner_mantle = material_2;

rho_core = density_by_material(material_core);
rho_inner_mantle = density_by_material(material_inner_mantle);

if (rho_core_override) {rho_core = rho_core_ov;}
if (rho_inner_mantle_override) {rho_inner_mantle = rho_inner_mantle_ov;}

if (rho_inner_mantle > rho_core) {cout << "Geology solver finds more dense outer material , exiting..." << endl; exit(0);}

core_vol_fraction = (rho_mean - rho_inner_mantle) / (rho_core - rho_inner_mantle);

if ((core_vol_fraction < 0.0) || (core_vol_fraction > 1.0)) {cout << "Geology solver has no density solution, exiting..." << endl; exit(0);}

V_core = core_vol_fraction * V_total;
V_inner_mantle = (1.0 - core_vol_fraction) * V_total;

R_core = pow(V_core * 3./(4. * pi), 0.333333);
R_inner_mantle = R_total - R_core;

M_inner_mantle = V_inner_mantle * rho_inner_mantle;
M_core = V_core * rho_core;


compute_radiogenic_heat();
radiogenic_heat_total = 0.0;

if ((material_core == "silicate_rock")|| (material_core == "carbon")) {radiogenic_heat_total = M_core * radiogenic_heat_factor;}
if ((material_inner_mantle == "silicate_rock")||(material_inner_mantle == "carbon")) {radiogenic_heat_total = M_inner_mantle * radiogenic_heat_factor;}

radiogenic_heat_total *=isotope_abundancy;

compute_primordial_heat();

radiogenic_heat_flux = radiogenic_heat_total/ (4.0 * pi * pow(R_total,2.0));
primordial_heat_flux = primordial_heat_total/ (4.0 * pi * pow(R_total,2.0));
total_heat_flux = radiogenic_heat_flux + primordial_heat_flux;

}



void Geology::initialize_undifferentiated (double M, double R, double age, double T, string material_1, string material_2)
{

double rho_1, rho_2;

M_total = M;
R_total = R;
system_age = age;
T_surface = T;

if (T_surface_ov > 0.0) {T_surface = T_surface_ov;}

component_1 = material_1;
component_2 = material_2;

num_layers = 1;
is_differentiated = false;

surface_gravity = G * M / pow(R_total, 2.0);
V_total = 4./3. * pi * pow(R_total, 3.0);

rho_mean = M_total/ V_total;
material_core = "mixture";
rho_core = rho_mean;

rho_1 = density_by_material(material_1);
rho_2 = density_by_material(material_2);

material_1_fraction = (rho_mean - rho_2) / (rho_1 - rho_2);


if ((material_1_fraction < 0.0) || (material_1_fraction > 1.0)) {cout << "Geology solver has no density solution, exiting..." << endl; exit(0);}

V_core =  V_total;
R_core = R_total;
M_core = M_total;


compute_radiogenic_heat();
radiogenic_heat_total = 0.0;

if ((material_1 == "silicate_rock")|| (material_1 == "carbon")) {radiogenic_heat_total = M_core * material_1_fraction * radiogenic_heat_factor;}
if ((material_2 == "silicate_rock")|| (material_2 == "carbon")) {radiogenic_heat_total = M_core * (1.0 - material_1_fraction) * radiogenic_heat_factor;}

radiogenic_heat_total *=isotope_abundancy;

compute_primordial_heat();

radiogenic_heat_flux = radiogenic_heat_total/ (4.0 * pi * pow(R_total,2.0));
primordial_heat_flux = primordial_heat_total/ (4.0 * pi * pow(R_total,2.0));
total_heat_flux = radiogenic_heat_flux + primordial_heat_flux;


}





void Geology::initialize (double M, double R, double age, double R_c, double T, string material_1, string material_2, string material_3)
{
double core_vol_fraction;
double inner_mantle_vol_fraction;
double V_remain, M_remain, rho_remain;

M_total = M;
R_total = R;
system_age = age;
T_surface = T;

if (T_surface_ov > 0.0) {T_surface = T_surface_ov;}

num_layers = 3;
is_differentiated = true;

surface_gravity = G * M / pow(R_total, 2.0);
V_total = 4./3. * pi * pow(R_total, 3.0);

rho_mean = M_total/ V_total;


material_core = material_1;
material_inner_mantle = material_2;
material_outer_mantle = material_3;

rho_core = density_by_material(material_core);
rho_inner_mantle = density_by_material(material_inner_mantle);
rho_outer_mantle = density_by_material(material_outer_mantle);

if (rho_core_override) {rho_core = rho_core_ov;}
if (rho_inner_mantle_override) {rho_inner_mantle = rho_inner_mantle_ov;}
if (rho_outer_mantle_override) {rho_outer_mantle = rho_outer_mantle_ov;}

if ((rho_inner_mantle > rho_core) || (rho_outer_mantle > rho_core) || (rho_outer_mantle > rho_inner_mantle)) {cout << "Geology solver finds more dense outer material , exiting..." << endl; exit(0);}

R_core = R_c;
core_vol_fraction = pow(R_c/R_total, 3.0);
V_core = core_vol_fraction * V_total;

V_remain = V_total - V_core;
M_remain = M_total - V_core * rho_core;
rho_remain = M_remain/V_remain;

inner_mantle_vol_fraction = (rho_remain - rho_outer_mantle) / (rho_inner_mantle - rho_outer_mantle);

if ((inner_mantle_vol_fraction < 0.0) || (inner_mantle_vol_fraction > 1.0)) {cout << "Geology solver has no density solution, exiting..." << endl; exit(0);}

V_inner_mantle = inner_mantle_vol_fraction * V_remain;
V_outer_mantle = (1.0 - inner_mantle_vol_fraction) * V_remain;

R_inner_mantle = pow((V_core + V_inner_mantle) * 3./(4. * pi), 0.333333) - R_core;

R_outer_mantle = R_total - R_inner_mantle - R_core;

M_inner_mantle = V_inner_mantle * rho_inner_mantle;
M_core = V_core * rho_core;
M_outer_mantle = V_outer_mantle * rho_outer_mantle;


compute_radiogenic_heat();
radiogenic_heat_total = 0.0;

if ((material_core == "silicate_rock")|| (material_core == "carbon")) {radiogenic_heat_total += M_core * radiogenic_heat_factor;}
if ((material_inner_mantle == "silicate_rock")||(material_inner_mantle == "carbon")) {radiogenic_heat_total += M_inner_mantle * radiogenic_heat_factor;}
if ((material_outer_mantle == "silicate_rock")||(material_outer_mantle == "carbon")) {radiogenic_heat_total += M_outer_mantle * radiogenic_heat_factor;}

radiogenic_heat_total *=isotope_abundancy;

compute_primordial_heat();

radiogenic_heat_flux = radiogenic_heat_total/ (4.0 * pi * pow(R_total,2.0));
primordial_heat_flux = primordial_heat_total/ (4.0 * pi * pow(R_total,2.0));
total_heat_flux = radiogenic_heat_flux + primordial_heat_flux;


}



void Geology::initialize (double M, double R, double age, double T, string material)
{
double rho_ref;

M_total = M;
R_total = R;
system_age = age;
T_surface = T;

if (T_surface_ov > 0.0) {T_surface = T_surface_ov;}

num_layers = 1;
is_differentiated = true;

surface_gravity = G * M / pow(R_total, 2.0);
V_total = 4./3. * pi * pow(R_total, 3.0);

rho_mean = M_total/ V_total;
material_core = material;

rho_ref = density_by_material(material);

if ((rho_mean < 0.7 * rho_ref) || (rho_mean > 1.3 * rho_ref))
	{
	cout << "Geology solver has no density solution with this material, exiting..." << endl; exit(0);
	}


rho_core = rho_ref;


V_core = V_total;
V_inner_mantle = 0.0;

R_core = R_total;
R_inner_mantle =0.0;

M_inner_mantle = 0.0;
M_core = M_total;


compute_radiogenic_heat();
radiogenic_heat_total = 0.0;

if ((material_core == "silicate_rock")||(material_core == "carbon")) {radiogenic_heat_total = M_core * radiogenic_heat_factor;}
radiogenic_heat_total *=isotope_abundancy;

compute_primordial_heat();

radiogenic_heat_flux = radiogenic_heat_total/ (4.0 * pi * pow(R_total,2.0));
primordial_heat_flux = primordial_heat_total/ (4.0 * pi * pow(R_total,2.0));
total_heat_flux = radiogenic_heat_flux + primordial_heat_flux;


}

void Geology::finish_init()
{
compute_crust_volcanic_transport();

compute_profile();
}



void Geology::set_tidal(double M_H, double period, double e, double a)
{
double l2, n;

if (is_differentiated)
	{
	if (num_layers == 1) {l2 = love_by_material(material_core);}
	else if (num_layers == 2) {l2 = love_by_material(material_inner_mantle);}
	else if (num_layers == 3) {l2 = love_by_material(material_outer_mantle);}
	}
else
	{
	l2 = material_1_fraction * love_by_material(component_1) + (1.0 - material_1_fraction) * love_by_material(component_2);
	}

n = 2.0 * pi/ period;

tidal_heat_total = l2 * 21./2. * G * pow(M_H,2.0) * pow(R_total, 5.0) * n * pow(e,2.0) / pow(a, 6.0);
tidal_heat_flux = tidal_heat_total/ (4.0 * pi * pow(R_total,2.0));

total_heat_flux = radiogenic_heat_flux + primordial_heat_flux + tidal_heat_flux;

tidal_defined = true;
}

void Geology::set_isotope_abundancy(double iso)
{
isotope_abundancy = iso;
}

void Geology::set_parameter(string parameter, double value)
{
if (parameter == "isotope_abundancy") 
	{isotope_abundancy = value;}
else if (parameter == "depth_resolution")
	{depth_profile_resolution = value;}
else if (parameter == "tidal_dissipation")
	{tidal_dissipation_factor = value;}
else if (parameter == "surface_temperature")
	{T_surface_ov = value;}

}


double Geology::get_pressure(double depth, double res)
{

double R_current, M_above, M_below, M_column, F_current, rho;


R_current = R_total;
M_above = 0.0;
M_column = 0.0;
F_current = 0.0;


while (R_current > R_total - depth)
	{
	R_current -= res;
	if (R_current > R_core + R_inner_mantle) 
		{rho = rho_outer_mantle;}
	else if (R_current > R_core)
		{rho = rho_inner_mantle;}
	else 
		{rho = rho_core; }
	M_above += 4.0 * pi * pow(R_current,2.0) * res * rho;	
	M_column+= res * rho;
	M_below = M_total - M_above;

	F_current += res * rho  * G* M_below/pow(R_current,2.0) / 9.81;

	}



F_current /= 10000.0; // convert to bar

return F_current;
}



double Geology::lambda_lookup(double R_current, double T, double p, string material)
{
bool melting_flag;
double lambda;

melting_flag = check_melting (T, p, material);

if (material == "silicate_rock")
	{
	lambda = 2.5;

	if ((melting_flag == true) && (solid_convection_flag == false))
		{solid_convection_flag = true; 	crust_volcanic_transport_fraction = 0.0;}

	if (solid_convection_flag)
		{lambda = 150.0 * liquidness_factor;}

	// transition zone to core

	
	if ((R_current < R_core + 900000.0) && (R_current > R_core) && solid_convection_flag)
		{
		lambda = 150.0 * liquidness_factor *  (R_current - R_core)/900000.0 + 2.5;
		}

	}
else if ((material == "nickel_iron")||(material == "sulfur_iron"))
	{
	lambda = interpolate(T, data_thermal_conductivity_Fe, n_data_thermal_conductivity_Fe);
	if (T > 1400.0)
			{lambda = (T - 1200.) * 0.06;} // empirical continuation of data to Earth center fit

	if (R_current < R_core) 
		{
		if (melting_flag)
			{lambda = 20.0 + 45.5 * (R_core - R_current)/R_core; } // empirical after Earth temperature
		}

	}
else if (material == "carbon")
	{

	if (check_diamond(T,p))
		{lambda = 2200.0;}	
	else
		{lambda = 200.0;}

	}
else if (material == "carborundum")
	{
	lambda = 320.;
	}
else if (material == "water_ice")
	{


	if ((melting_flag == true) && (liquid_convection_flag == false)) 
		{liquid_convection_flag = true; }

	if ((liquid_convection_flag) && (p <= 5.0e3))
		{lambda = 100000.0; } 
	else
		{
		if (p > 3e6) {lambda =  12.0; liquid_convection_flag = false;} // ice XI
	 	else if (p > 6.5e5) {lambda = 10.0; liquid_convection_flag = false;} // ice X
		else if (p > 2.e4) {lambda = 7.0; liquid_convection_flag = false;} // ice VII
		else if (p > 5e3) {lambda = 2.0; liquid_convection_flag = false;} // ice VI
		else {lambda = 2.2; }
		}
	
	}


return lambda;

}


void Geology::compute_max_elevation()
{
double isostasis_limit, erosion_limit;

if (crust_material == "silicate_rock")
	{isostasis_limit = crust_thickness * 0.33333;}
else if (crust_material == "water_ice")	
	{isostasis_limit = crust_thickness * 0.05;}
else 
	{isostasis_limit = crust_thickness;}

isostasis_limit *= 1.0+ 0.35 * (volcanic_activity_index - 1.0);

erosion_limit = 10000. * 9.81/surface_gravity;

max_elevation_estimate = erosion_limit;
if (isostasis_limit < erosion_limit) {max_elevation_estimate = isostasis_limit;}

}


double Geology::get_lambda(double R_current, double T, double p)
{
string material;


if (is_differentiated == false)
	{
	return material_1_fraction * lambda_lookup(R_current, T, p, component_1) + (1.0-material_1_fraction) * lambda_lookup(R_current, T, p, component_2);
	}

if (R_current > R_core + R_inner_mantle) {material = material_outer_mantle;}
else if (R_current > R_core) {material = material_inner_mantle;}
else {material = material_core;}

return lambda_lookup(R_current, T, p, material);

}




double Geology::get_temperature(double depth, double res)
{
double R_current, M_above, M_below, M_column, p, lambda, rho;
double T, T_gradient;
double local_heat_flux;
string material;


R_current = R_total;
M_above = 0.0;
M_column = 0.0;
p = 0.0;

solid_convection_flag = false;
liquid_convection_flag = false;

compute_crust_volcanic_transport();

T = T_surface;
lambda = get_lambda(R_current, T, p);

if (solid_convection_flag == true) {crust_volcanic_transport_fraction = 0.0;}

T_gradient = (1.0 - crust_volcanic_transport_fraction) * total_heat_flux/lambda;




while (R_current > R_total - depth)
	{
	R_current -= res;
	T += T_gradient * res;
	


	lambda = get_lambda(R_current, T, p);

	local_heat_flux = primordial_heat_flux * pow(R_current/R_total,0.25);
	local_heat_flux += tidal_heat_flux * pow(R_current/R_total, 2.0);
	if ((R_current < R_rad_high) && (R_current > R_rad_low))
		{
		local_heat_flux += radiogenic_heat_flux * pow((R_current - R_rad_low)/(R_rad_high - R_rad_low), 0.5);
		}
	else if (R_current > R_rad_high)
		{
		local_heat_flux += radiogenic_heat_flux;
		}

	if (R_current > R_core + R_inner_mantle)
		{
		rho = rho_outer_mantle;
		}
	else if (R_current > R_core) 
		{
		rho = rho_inner_mantle;
		}
	else 
		{
		rho = rho_core;
		}
	T_gradient = (1.0 - crust_volcanic_transport_fraction) * local_heat_flux / lambda;

		
	M_above += 4.0 * pi * pow(R_current,2.0) * res * rho;	
	M_column+= res * rho;
	M_below = M_total - M_above;
	p += res * rho  * G* M_below/pow(R_current,2.0) / 9.81 / 10000.0;
	

	}

return T;

}


void Geology::compute_profile()
{
double R_current, M_above, M_below, M_column, p, lambda, rho;
double T, T_gradient;
double local_heat_flux;
double res;
string material;
int counter, count_max;


R_current = R_total;
M_above = 0.0;
M_column = 0.0;
p = 0.0;
counter = 0;
res = 100.0;

count_max = static_cast<int> (depth_profile_resolution/ res);


solid_convection_flag = false;
liquid_convection_flag = false;

crust_thickness = R_total;

if (num_layers == 1)
	{
	R_rad_high = R_total; R_rad_low = 0.0;
	crust_material = material_core;
	}
else if (num_layers == 2)
	{
	crust_material = material_inner_mantle;
	if ((material_inner_mantle == "silicate_rock") || (material_inner_mantle == "carbon"))
		{
		R_rad_high = R_total; R_rad_low = R_core;
		}
	else if ((material_core == "silicate_rock") || (material_core == "carbon"))
		{
		R_rad_high = R_core; R_rad_low = 0.0;
		}
	else
		{
		R_rad_high = R_total; R_rad_low = 0.0;
		}
	}
else if (num_layers == 3)
	{
	crust_material = material_outer_mantle;
	if ((material_outer_mantle == "silicate_rock") || (material_outer_mantle == "carbon"))
		{
		R_rad_high = R_total; R_rad_low = R_inner_mantle + R_core;
		}
	else if ((material_inner_mantle == "silicate_rock") || (material_inner_mantle == "carbon"))
		{
		R_rad_high = R_inner_mantle + R_core; R_rad_low = R_core;
		}
	else if ((material_core == "silicate_rock") || (material_core == "carbon"))
		{
		R_rad_high = R_core; R_rad_low = 0.0;
		}
	else
		{
		R_rad_high = R_total; R_rad_low = 0.0;
		}

	}


T = T_surface;
lambda = get_lambda(R_current, T, p);
if (solid_convection_flag == true) {crust_volcanic_transport_fraction = 0.0; }
T_gradient = (1.0 - crust_volcanic_transport_fraction) * total_heat_flux/lambda;

if (profile_verbose)
	{
	cout << endl;
	cout << "Geological depth profile" << endl;
	cout << "========================" << endl;
	cout << endl;
	cout << left << "Depth" << " " << setw(15) << "material";
	cout << left << setw(8) << "phase" << setw(5) << " T[K]" << "    p[kbar]"; 
	cout << endl;
	}


while (R_current > 0.01 * R_total)
	{
	R_current -= res;
	T += T_gradient * res;
	
	lambda = get_lambda(R_current, T, p);

	if (R_total - R_current < crust_thickness)
		{
		if ((liquid_convection_flag) || (solid_convection_flag))
			{
			crust_thickness = R_total - R_current;
			}
		}

	local_heat_flux = primordial_heat_flux * pow(R_current/R_total,0.25);
	local_heat_flux += tidal_heat_flux * pow(R_current/R_total, 2.0);
	
	if ((R_current < R_rad_high) && (R_current > R_rad_low))
		{local_heat_flux += radiogenic_heat_flux * pow((R_current - R_rad_low)/(R_rad_high - R_rad_low), 0.5);}
	else if (R_current > R_rad_high)
		{local_heat_flux += radiogenic_heat_flux;}

	if (R_current > R_core + R_inner_mantle)
		{rho = rho_outer_mantle;}
	else if (R_current > R_core) 
		{rho = rho_inner_mantle;}
	else 
		{rho = rho_core;}

	T_gradient = (1.0 - crust_volcanic_transport_fraction) * local_heat_flux / lambda;

		
	M_above += 4.0 * pi * pow(R_current,2.0) * res * rho;	
	M_column+= res * rho;
	M_below = M_total - M_above;
	p += res * rho  * G* M_below/pow(R_current,2.0) / 9.81 / 10000.0;
	

	if ((counter == count_max) && (profile_verbose))
		{
		string material;
		string phase;		

		if (R_current > R_core + R_inner_mantle) {material = material_outer_mantle;}
		else if (R_current > R_core) {material = material_inner_mantle;}
		else {material = material_core;}

		if ((material == "water_ice") && (liquid_convection_flag == true))
			{phase = "liquid";}
		else if ((material == "nickel_iron") && (check_melting(T, p, material)))
			{phase = "liquid";}
		else if ((material == "sulfur_iron") && (check_melting(T, p, material)))
			{phase = "liquid";}
		else if	((material == "silicate_rock") && (solid_convection_flag == true))
			{phase = "liquid";}
		else
			{phase = "solid";}	

		if (material == "water_ice")
			{
			if (p > 3e6) {material = "ice_XI";} // ice XI
		 	else if (p > 6.5e5) {material = "ice_X";} // ice X
			else if (p > 2.e4) {material = "ice_VII";} // ice VII
			else if (p > 5e3) {material = "ice_VI";} // ice VI
			else {material = "ice_I";}
			}
		if (material == "carbon")
			{
			if (check_diamond(T, p)) {material = "diamond";}
			}
	
		if ((phase == "liquid") && (material == "ice_I")) {material = "water";}

		cout << right << setprecision(4) <<setw(5) << (R_total - R_current) / 1000.;
		cout << left << " " << setw(14) <<  material << " " << setw(8) << phase << " " ;
		cout << setprecision(5) << setw(6) << T <<  "  " << setprecision(4) << setw(5) << p/1000 << " "  << endl;
		counter = 0;
		}

	counter ++;
	}

compute_max_elevation();

if (crust_material == "silicate_rock")
	{
	if ((crust_thickness > 14000.0) && (crust_thickness < 40000.0)) {heat_dissipation_mode = "plate tectonics";}
	else if (crust_thickness < 14000.0) {heat_dissipation_mode = "volcanism";}
	else {heat_dissipation_mode = "conduction";}
	}
else if (crust_material == "water_ice")
	{
	if ((crust_thickness > 20000.0) && (crust_thickness < 60000.0)) {heat_dissipation_mode = "plate tectonics";}
	else if (crust_thickness < 20000.0) {heat_dissipation_mode = "cryovolcanism";}
	else {heat_dissipation_mode = "conduction";}
	}
else 
	{heat_dissipation_mode = "conduction";}
}

void Geology::show_profile()
{
profile_verbose = true;
compute_crust_volcanic_transport();
compute_profile();
profile_verbose = false;
}



void Geology::list()
{
cout << endl;
cout << "Geology" << endl;
cout << "-----------" << endl;
if (num_layers > 1)
	{
	cout << "Core material:         " << material_core << endl;
	cout << "Core radius [km]:      " << R_core/1e3 << endl;
	cout << "Core mass [m_earth]:   " << M_core/M_earth << endl;
	cout << "Core mass fraction:    " << M_core/M_total << endl;
	cout << "Core vol. fraction:    " << V_core/V_total << endl;
	cout << "Core density [g/cm^3]: " << rho_core/1e3 << endl;
	cout << "Mantle material:       " << material_inner_mantle << endl;
	cout << "Mantle radius [km]:    " << R_inner_mantle/1e3 << endl;
	cout << "Mantle mass [m_earth]: " << M_inner_mantle/M_earth << endl;
	cout << "Mantle mass fraction:  " << M_inner_mantle/M_total << endl;
	cout << "Mantle vol. frac.:     " << V_inner_mantle/V_total << endl;
	cout << "Mantle dens. [g/cm^3]: " << rho_inner_mantle/1e3 << endl;
	}
if (num_layers == 3)
	{
	cout << "Outer material:        " << material_outer_mantle << endl;
	cout << "Outer radius [km]:     " << R_outer_mantle/1e3 << endl;
	cout << "Outer mass [m_earth]:  " << M_outer_mantle/M_earth << endl;
	cout << "Outer mass fraction:   " << M_outer_mantle/M_total << endl;
	cout << "Outer vol. frac.:      " << V_outer_mantle/V_total << endl;
	cout << "Outer dens. [g/cm^3]:  " << rho_outer_mantle/1e3 << endl;
	}
if (num_layers == 1)
	{
	cout << "Material:              " << material_core << endl;
	if (is_differentiated == false)
	{
	cout << "Component 1:           " << component_1 << endl;	
	cout << "Component 2:           " << component_2 << endl;
	cout << "Mixing fraction:       " << material_1_fraction << endl;	
	}
	cout << "Radius [km]:           " << R_core/1e3 << endl;
	cout << "Mass [m_earth]:        " << M_core/M_earth << endl;
	cout << "Density [g/cm^3]:      " << rho_core/1e3 << endl;

	}
cout << "Crust thickness [km]:  " << crust_thickness / 1e3 << endl;
cout << "Radiogenic heat [TW]:  " << radiogenic_heat_total / 1e12 << endl;
cout << "Radiog. flux [W/m^2]:  " << radiogenic_heat_total/ (4.0 * pi * pow(R_total,2.0)) << endl;
cout << "Primordial heat [TW]:  " << primordial_heat_total / 1e12 << endl;
cout << "Prim. flux [W/m^2]:    " << primordial_heat_total/ (4.0 * pi * pow(R_total,2.0)) << endl;
if (tidal_defined)
	{
	cout << "Tidal heat [TW]:       " << tidal_heat_total / 1e12 << endl;
	cout << "Tidal flux [W/m^2]:    " << tidal_heat_total/ (4.0 * pi * pow(R_total,2.0)) << endl;
	}
cout << "Total flux [W/m^2]:    " << (primordial_heat_total + radiogenic_heat_total + tidal_heat_total)/ (4.0 * pi * pow(R_total,2.0)) << endl;
cout << "Heat dissipation mode: " << heat_dissipation_mode << endl;
cout << "Max. elev. est. [km]:  " << max_elevation_estimate / 1e3 << endl;
cout << endl;

//cout << get_temperature(0.99 * R_total, 100.0, 105.0) << endl;
}
