// gases.cxx -- a collection of gas properties
//
// Written by Thorsten Renk, started 2018
//
// Copyright (C) 2018  Thorsten Renk - thorsten@science-and-fiction.org 
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301

#include <iostream>
#include <fstream>
#include <string>
#include <stdlib.h>
#include <math.h>

using namespace std;

#include "constants.hxx"
#include "gases.hxx"
#include "utilities.hxx"

Gases::Gases ()
{
exclusive_component = "";

n_linedata_CO2 = read_array("data/co2.dat", line_array_CO2, 5.0);
n_linedata_H2O = read_array("data/h2o.dat", line_array_H2O, 6e20);
n_linedata_CH4 = read_array("data/ch4.dat", line_array_CH4, 6e20);
n_linedata_O3 = read_array("data/o3.dat", line_array_O3, 5e23);

n_linedata_O3_uv = read_array("data/o3_uv.dat", line_array_O3_uv, 1e-4);
n_linedata_O2_uv = read_array("data/o2_uv.dat", line_array_O2_uv, 1e-4);
n_linedata_N2_uv = read_array("data/n2_uv.dat", line_array_N2_uv, 1e-4);
n_linedata_CO2_uv = read_array("data/co2_uv.dat", line_array_CO2_uv, 1e-4);
n_linedata_SO2_uv = read_array("data/so2_uv.dat", line_array_SO2_uv, 1e-4);
n_linedata_H2_uv = read_array("data/h2_uv.dat", line_array_H2_uv, 1e-4);
n_linedata_NH3_uv = read_array("data/nh3_uv.dat", line_array_NH3_uv, 1e-4);

n_phasedata_sat_N2 = read_array("data/n2_phases_saturation.dat", phasedata_sat_N2, 1.0);
n_phasedata_melt_N2 = read_array("data/n2_phases_melting.dat", phasedata_melt_N2, 1.0);

n_phasedata_sat_O2 = read_array("data/o2_phases_saturation.dat", phasedata_sat_O2, 1.0);
n_phasedata_melt_O2 = read_array("data/o2_phases_melting.dat", phasedata_melt_O2, 1.0);

n_phasedata_sat_CO2 = read_array("data/co2_phases_saturation.dat", phasedata_sat_CO2, 1.0);
n_phasedata_melt_CO2 = read_array("data/co2_phases_melting.dat", phasedata_melt_CO2, 1.0);

n_phasedata_sat_CH4 = read_array("data/ch4_phases_saturation.dat", phasedata_sat_CH4, 1.0);
n_phasedata_melt_CH4 = read_array("data/ch4_phases_melting.dat", phasedata_melt_CH4, 1.0);

n_phasedata_sat_Ar = read_array("data/ar_phases_saturation.dat", phasedata_sat_Ar, 1.0);
n_phasedata_melt_Ar = read_array("data/ar_phases_melting.dat", phasedata_melt_Ar, 1.0);

n_phasedata_sat_H2O = read_array("data/h2o_phases_saturation.dat", phasedata_sat_H2O, 1.0);
n_phasedata_melt_H2O = read_array("data/h2o_phases_melting.dat", phasedata_melt_H2O, 1.0);

n_phasedata_sat_NH3 = read_array("data/nh3_phases_saturation.dat", phasedata_sat_NH3, 1.0);
n_phasedata_melt_NH3 = read_array("data/nh3_phases_melting.dat", phasedata_melt_NH3, 1.0);

n_phasedata_sat_SO2 = read_array("data/so2_phases_saturation.dat", phasedata_sat_SO2, 1.0);
n_phasedata_melt_SO2 = read_array("data/so2_phases_melting.dat", phasedata_melt_SO2, 1.0);

n_phasedata_sat_Ne = read_array("data/ne_phases_saturation.dat", phasedata_sat_Ne, 1.0);
n_phasedata_melt_Ne = read_array("data/ne_phases_melting.dat", phasedata_melt_Ne, 1.0);

n_phasedata_sat_H2 = read_array("data/h2_phases_saturation.dat", phasedata_sat_H2, 1.0);
n_phasedata_melt_H2 = read_array("data/h2_phases_melting.dat", phasedata_melt_H2, 1.0);

n_phasedata_sat_H2S = read_array("data/h2s_phases_saturation.dat", phasedata_sat_H2S, 1.0);
n_phasedata_melt_H2S = read_array("data/h2s_phases_melting.dat", phasedata_melt_H2S, 1.0);
}



double Gases::get_molecular_weight(string name)
{
if (name == "O2") {return 32.0;}
else if (name == "N2") {return 28.0;}
else if (name == "H2") {return 2.0;}
else if (name == "He") {return 4.0;}
else if (name == "CH4") {return 16.0;}
else if (name == "NH3") {return 18.0;}
else if (name == "Ne") {return 20.0;}
else if (name == "CO") {return 28.0;}
else if (name == "H2S") {return 34.0;}
else if (name == "Ar") {return 40.0;}
else if (name == "CO2") {return 44.0;}
else if (name == "Cl2") {return 70.0;}
else if (name == "H2O") {return 18.0;}
else if (name == "O3") {return 48.0;}
else if (name == "SO2") {return 64.0;}
else {return 0.0;}
}


double Gases::get_melting_point(string name)
{
if (name == "O2") {return 54.0;}
else if (name == "N2") {return 63.0;}
else if (name == "H2") {return 14.0;}
else if (name == "He") {return 0.0;}
else if (name == "CH4") {return 90.0;}
else if (name == "NH3") {return 195.0;}
else if (name == "Ne") {return 24.0;}
else if (name == "CO") {return 68.0;}
else if (name == "H2S") {return 191.0;}
else if (name == "Ar") {return 83.0;}
else if (name == "CO2") {return 194.0;}
else if (name == "Cl2") {return 171.0;}
else if (name == "H2O") {return 273.15;}
else if (name == "O3") {return 81.0;}
else if (name == "SO2") {return 201.0;}
else {return 0.0;}

}

double Gases::get_boiling_point(string name)
{
if (name == "O2") {return 90.0;}
else if (name == "N2") {return 77.0;}
else if (name == "H2") {return 20.0;}
else if (name == "He") {return 4.2;}
else if (name == "CH4") {return 111.0;}
else if (name == "NH3") {return 239.0;}
else if (name == "Ne") {return 27.0;}
else if (name == "CO") {return 81.0;}
else if (name == "H2S") {return 223.0;}
else if (name == "Ar") {return 87.0;}
else if (name == "CO2") {return 194.0;}
else if (name == "Cl2") {return 239.0;}
else if (name == "H2O") {return 373.13;}
else if (name == "O3") {return 161.0;}
else if (name == "SO2") {return 263.0;}
else {return 0.0;}

}

double Gases::get_c_p(string name)
{
if (name == "CO2") {return 0.84;}
else if (name == "O2") {return 0.92;}
else if (name == "H2") {return 14.3;}
else if (name == "He") {return 5.19;}
else if (name == "NH3") {return 4.7;}
else if (name == "N2") {return 1.04;}
else if (name == "CH4") {return 2.19;}
else if (name == "CO") {return 1.02;}
else if (name == "Ne") {return 1.03;}
else if (name == "Ar") {return 0.520;}
else if (name == "H2S") {return 1.015;}
else if (name == "Cl2") {return 0.48;}
else if (name == "H2O") {return 4.18;}
else if (name == "O3") {return 0.92;}
else if (name == "SO2") {return 1.3636;}
else {return 0.0;}
}

double Gases::get_R(string name)
{
if (name == "CO2") {return 188.9;}
else if (name == "O2") {return 259.8;}
else if (name == "H2") {return 4124.0;}
else if (name == "He") {return 2076.9;}
else if (name == "NH3") {return 488.23;}
else if (name == "N2") {return 296.8;}
else if (name == "CH4") {return 518.2;}
else if (name == "CO") {return 296.8;}
else if (name == "Ne") {return 411.9;}
else if (name == "Ar") {return 208.1;}
else if (name == "H2S") {return 243.96;}
else if (name == "Cl2") {return 117.26;}
else if (name == "H2O") {return 461.5;}
else if (name == "O3") {return 173.21;} 
else if (name == "SO2") {return 129.78;}
else {return 0.0;}
}

double Gases::lineshape(double lambda, string name)
{
if ((exclusive_component != "") && (name != exclusive_component)) {return 0.0;}

if (name == "CO2") { return interpolate(lambda, line_array_CO2, n_linedata_CO2); }
else if (name == "H2O") {return interpolate(lambda, line_array_H2O, n_linedata_H2O);}
else if (name == "CH4") {return interpolate(lambda, line_array_CH4, n_linedata_CH4);}
else if (name == "O3") {return interpolate(lambda, line_array_O3, n_linedata_O3);}
else {return 0.0;}
}

double Gases::lineshape_uv(double lambda, string name)
{
if ((exclusive_component != "") && (name != exclusive_component)) {return 0.0;}

if (name == "O3") {return interpolate(lambda, line_array_O3_uv, n_linedata_O3_uv);}
else if (name == "O2") {return interpolate(lambda, line_array_O2_uv, n_linedata_O2_uv);}
else if (name == "N2") {return interpolate(lambda, line_array_N2_uv, n_linedata_N2_uv);}
else if (name == "CO2") {return interpolate(lambda, line_array_CO2_uv, n_linedata_CO2_uv);}
else if (name == "SO2") {return interpolate(lambda, line_array_SO2_uv, n_linedata_SO2_uv);}
else if (name == "H2") {return interpolate(lambda, line_array_H2_uv, n_linedata_H2_uv);}
else if (name == "NH3") {return interpolate(lambda, line_array_NH3_uv, n_linedata_NH3_uv);}
else {return 0.0;}
}

int Gases::get_phase(double p, double T, string name)
{
int n_sat, n_melt;
double phasedata_sat[20][2];
double phasedata_melt[20][2];
double T_sat, T_melt;

if (name == "N2")
	{
	n_sat = n_phasedata_sat_N2; n_melt = n_phasedata_melt_N2;
	for (int i=0; i< n_sat; i++)
		{
		phasedata_sat[i][0] = phasedata_sat_N2[i][0];
		phasedata_sat[i][1] = phasedata_sat_N2[i][1];
		}
	for (int i=0; i< n_melt; i++)
		{
		phasedata_melt[i][0] = phasedata_melt_N2[i][0];
		phasedata_melt[i][1] = phasedata_melt_N2[i][1];
		}
	}
else if (name == "O2")
	{
	n_sat = n_phasedata_sat_O2; n_melt = n_phasedata_melt_O2;
	for (int i=0; i< n_sat; i++)
		{
		phasedata_sat[i][0] = phasedata_sat_O2[i][0];
		phasedata_sat[i][1] = phasedata_sat_O2[i][1];
		}
	for (int i=0; i< n_melt; i++)
		{
		phasedata_melt[i][0] = phasedata_melt_O2[i][0];
		phasedata_melt[i][1] = phasedata_melt_O2[i][1];
		}
	}
else if (name == "CO2")
	{
	n_sat = n_phasedata_sat_CO2; n_melt = n_phasedata_melt_CO2;
	for (int i=0; i< n_sat; i++)
		{
		phasedata_sat[i][0] = phasedata_sat_CO2[i][0];
		phasedata_sat[i][1] = phasedata_sat_CO2[i][1];
		}
	for (int i=0; i< n_melt; i++)
		{
		phasedata_melt[i][0] = phasedata_melt_CO2[i][0];
		phasedata_melt[i][1] = phasedata_melt_CO2[i][1];
		}
	}
else if (name == "CH4")
	{
	n_sat = n_phasedata_sat_CH4; n_melt = n_phasedata_melt_CH4;
	for (int i=0; i< n_sat; i++)
		{
		phasedata_sat[i][0] = phasedata_sat_CH4[i][0];
		phasedata_sat[i][1] = phasedata_sat_CH4[i][1];
		}
	for (int i=0; i< n_melt; i++)
		{
		phasedata_melt[i][0] = phasedata_melt_CH4[i][0];
		phasedata_melt[i][1] = phasedata_melt_CH4[i][1];
		}
	}
else if (name == "Ar")
	{
	n_sat = n_phasedata_sat_Ar; n_melt = n_phasedata_melt_Ar;
	for (int i=0; i< n_sat; i++)
		{
		phasedata_sat[i][0] = phasedata_sat_Ar[i][0];
		phasedata_sat[i][1] = phasedata_sat_Ar[i][1];
		}
	for (int i=0; i< n_melt; i++)
		{
		phasedata_melt[i][0] = phasedata_melt_Ar[i][0];
		phasedata_melt[i][1] = phasedata_melt_Ar[i][1];
		}
	}
else if (name == "H2O")
	{
	n_sat = n_phasedata_sat_H2O; n_melt = n_phasedata_melt_H2O;
	for (int i=0; i< n_sat; i++)
		{
		phasedata_sat[i][0] = phasedata_sat_H2O[i][0];
		phasedata_sat[i][1] = phasedata_sat_H2O[i][1];
		}
	for (int i=0; i< n_melt; i++)
		{
		phasedata_melt[i][0] = phasedata_melt_H2O[i][0];
		phasedata_melt[i][1] = phasedata_melt_H2O[i][1];
		}
	}
else if (name == "NH3")
	{
	n_sat = n_phasedata_sat_NH3; n_melt = n_phasedata_melt_NH3;
	for (int i=0; i< n_sat; i++)
		{
		phasedata_sat[i][0] = phasedata_sat_NH3[i][0];
		phasedata_sat[i][1] = phasedata_sat_NH3[i][1];
		}
	for (int i=0; i< n_melt; i++)
		{
		phasedata_melt[i][0] = phasedata_melt_NH3[i][0];
		phasedata_melt[i][1] = phasedata_melt_NH3[i][1];
		}
	}
else if (name == "SO2")
	{
	n_sat = n_phasedata_sat_SO2; n_melt = n_phasedata_melt_SO2;
	for (int i=0; i< n_sat; i++)
		{
		phasedata_sat[i][0] = phasedata_sat_SO2[i][0];
		phasedata_sat[i][1] = phasedata_sat_SO2[i][1];
		}
	for (int i=0; i< n_melt; i++)
		{
		phasedata_melt[i][0] = phasedata_melt_SO2[i][0];
		phasedata_melt[i][1] = phasedata_melt_SO2[i][1];
		}
	}
else if (name == "Ne")
	{
	n_sat = n_phasedata_sat_Ne; n_melt = n_phasedata_melt_Ne;
	for (int i=0; i< n_sat; i++)
		{
		phasedata_sat[i][0] = phasedata_sat_Ne[i][0];
		phasedata_sat[i][1] = phasedata_sat_Ne[i][1];
		}
	for (int i=0; i< n_melt; i++)
		{
		phasedata_melt[i][0] = phasedata_melt_Ne[i][0];
		phasedata_melt[i][1] = phasedata_melt_Ne[i][1];
		}
	}
else if (name == "H2")
	{
	n_sat = n_phasedata_sat_H2; n_melt = n_phasedata_melt_H2;
	for (int i=0; i< n_sat; i++)
		{
		phasedata_sat[i][0] = phasedata_sat_H2[i][0];
		phasedata_sat[i][1] = phasedata_sat_H2[i][1];
		}
	for (int i=0; i< n_melt; i++)
		{
		phasedata_melt[i][0] = phasedata_melt_H2[i][0];
		phasedata_melt[i][1] = phasedata_melt_H2[i][1];
		}
	}
else if (name == "H2S")
	{
	n_sat = n_phasedata_sat_H2S; n_melt = n_phasedata_melt_H2S;
	for (int i=0; i< n_sat; i++)
		{
		phasedata_sat[i][0] = phasedata_sat_H2S[i][0];
		phasedata_sat[i][1] = phasedata_sat_H2S[i][1];
		}
	for (int i=0; i< n_melt; i++)
		{
		phasedata_melt[i][0] = phasedata_melt_H2S[i][0];
		phasedata_melt[i][1] = phasedata_melt_H2S[i][1];
		}
	}
else if (name == "He")
	{
	return 2;
	}
else 	
	{
	//cout << "No phase information for " << name << " available." << endl;
	return -1;
	}

if ((p < phasedata_melt[0][0]) || (p > phasedata_melt[n_melt-1][0]))
	{
	//cout << "No phase information for " << name << " at this pressure." << endl;
	return -1;	
	}


if (p > phasedata_sat[n_sat-1][0]) // above critical point
	{
	T_melt = interpolate(p, phasedata_melt, n_melt);
	if (T > T_melt) {return 1;}
	else {return 0;}

	}
else if (p < phasedata_sat[0][0]) // below triple point
	{
	T_melt = interpolate(p, phasedata_melt, n_melt);
	if (T > T_melt) {return 2;}
	else {return 0;}
	}
else
	{
	T_sat = interpolate(p, phasedata_sat, n_sat);
	T_melt = interpolate(p, phasedata_melt, n_melt);
	//cout << T_sat << " " << T_melt << " " << T << endl;
	
	if (T < T_melt) {return 0;}
	else if ((T > T_melt) && (T < T_sat)) {return 1;}
	else {return 2;}
	}

return -1;
}

