// checks.cxx -- routines for various cross checks
//
// Written by Thorsten Renk, started 2023
//
// Copyright (C) 2023  Thorsten Renk - thorsten@science-and-fiction.org 
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301

#include <iostream>
#include <fstream>
#include <string>
#include <stdlib.h>
#include <math.h>

#include "checks.hxx"

using namespace std;

void check_roche (double R_minor, double M_major, double M_minor, double R_orb, string name_major, string name_minor)
{

double roche_radius, roche_ratio;


roche_radius = R_minor * pow(2.0 * M_major/M_minor , 0.33333);
roche_ratio = R_orb /roche_radius;

if (R_orb < 2.44 * roche_radius)
	{
	cout << name_minor << " orbits " << name_major << " inside the Roche radius for liquid objects." << endl;
	cout << "It might be pulled apart by tidal forces." << endl;
	}
if (R_orb < roche_radius)
	{
	cout << name_minor << " orbits " << name_major << " inside the Roche radius for solid objects." << endl;
	cout << "It is likely to be pulled apart by tidal forces." << endl;
	}
if (R_orb > 2.44 * roche_radius)
	{
	cout << name_minor << " orbits " << name_major << " at " << roche_ratio << " Roche radii." << endl;
	cout << "The situation is plausible." << endl;
	}

}


void check_hill(double semimajor, double M_major, double M_minor, double ecc, double R_test, string name_minor, string name_test)
{
double hill_radius, hill_ratio;

hill_radius = semimajor * (1.0 - ecc) * pow(M_minor/(3.0 * M_major), 0.33333);

hill_ratio = R_test/hill_radius;

if (R_test > hill_radius)
	{
	cout << "Orbit of " << name_test << " around " << name_minor << " is unstable at " << hill_ratio << " R_Hill." << endl;
	}
else if (R_test > 0.3 * hill_radius) 
	{
	cout << "Orbit of " << name_test << " around " << name_minor << " is possibly long-term unstable at " << hill_ratio << " R_Hill." << endl;
	}
else
	{
	cout << "Orbit of " << name_test << " around " << name_minor << " is stable at " << hill_ratio << " R_Hill." << endl;
	}

}


void check_locking (double semimajor, double R, double m_sat, double m_P, string name_major, string name_minor)
{
double tidal_locking_time_a;

tidal_locking_time_a = 6.0e10  *  (pow(semimajor, 6.0) * R * 3.0e10) / (m_sat * pow(m_P, 2.0));

//cout << tidal_locking_time_a << endl;

if (tidal_locking_time_a < 1.0e8)
	{
	cout << "Rotation of " << name_minor << " in orbit around " << name_major << " must be tidally locked." << endl;
	if (tidal_locking_time_a > 1.0e5)
		{cout << "Timescale for locking is " << tidal_locking_time_a/1e6 << " million years." << endl;}
	else
		{cout << "Timescale for locking is " << tidal_locking_time_a/1e3 << " thousand years." << endl;}
	}
else if (tidal_locking_time_a < 1.0e9)
	{
	cout << "Rotation of " << name_minor << " in orbit around " << name_major << " is possibly tidally locked." << endl;
	cout << "Timescale for locking is " << tidal_locking_time_a/1e9 << " billion years." << endl;
	}
else
	{
	cout << "Rotation of " << name_minor << " in orbit around " << name_major << " has no tidal locking constraints." << endl;
	}
}

