// body.cxx -- routines to assign and compute properties of otherwise undefined gravitating bodies
//
// Written by Thorsten Renk, started 2018
//
// Copyright (C) 2018  Thorsten Renk - thorsten@science-and-fiction.org 
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301


#include <iostream>
#include <stdlib.h>
#include <fstream>
#include <math.h>

#include "body.hxx"
#include "constants.hxx"

using namespace std;


void Body::assign_properties (double set_mass, double set_radius, double set_semimajor, double set_eccentricity, double set_apsis_angle, double set_inc, double set_lan)
{

mass_earth = set_mass;
mass = mass_earth * M_earth;
radius_earth = set_radius;
radius = radius_earth * R_earth; 
apsis_angle = set_apsis_angle;
inclination = set_inc;
lan = set_lan;

density = mass / (4.0/3.0 * pi * pow(radius, 3.0));

surface_gravity = G * mass / pow(radius, 2.0);

semimajor = set_semimajor;
eccentricity = set_eccentricity;

periapsis = semimajor * (1.0 - eccentricity);
apoapsis = semimajor * (1.0 + eccentricity);

hyperbolic_trajectory = false;
}


void Body::assign_properties (double set_mass, double set_radius, double set_semimajor, double set_distance, double set_eccentricity, double set_apsis_angle, double set_inc, double set_lan)
{

mass_earth = set_mass;
mass = mass_earth * M_earth;
radius_earth = set_radius;
radius = radius_earth * R_earth; 
apsis_angle = set_apsis_angle;
inclination = set_inc;
lan = set_lan;


density = mass / (4.0/3.0 * pi * pow(radius, 3.0));

surface_gravity = G * mass / pow(radius, 2.0);

semimajor = set_semimajor;
eccentricity = set_eccentricity;

distance = set_distance;

periapsis = -semimajor * (eccentricity - 1.0);

hyperbolic_trajectory = true;
}



double Body::rnd(double min, double max)
{
double average = 0.5 * (max + min);
double variance = max - average;

return average + variance * 2.0 * ( rand()/(RAND_MAX +1.0) - 0.5);
}


void Body::set_central_mass(double M)
{
central_mass = M;

if (hyperbolic_trajectory)
	{
	double l = semimajor * (pow(eccentricity,2.0) - 1.0);
	double costheta = (l/distance - 1.0) / eccentricity;
	double sintheta = sqrt(1.0 - pow(costheta,2.0));	

	//cout << "dist: " << distance << " semimajor: " << semimajor << endl;

	speed_initial = sqrt(G * (M + mass) * (2.0/distance - 1.0/semimajor));
	speed_hyperbolic_excess = sqrt(- (G * (M + mass))/semimajor);
	speed_esc = sqrt(pow(speed_initial,2.0) - pow(speed_hyperbolic_excess,2.0));
	flight_path_angle = -atan( (eccentricity * sintheta) / (1.0 + eccentricity * costheta)) ;
	}
else
	{
	speed_periapsis = sqrt(G * (M + mass) * (2.0/periapsis - 1.0/semimajor));
	speed_apoapsis = sqrt(G * (M + mass) * (2.0/apoapsis - 1.0/semimajor));

	period = 2.0 * pi * sqrt(pow(semimajor, 3.0) / (G * (M + mass)));
	}
}



void Body::set_random(string property, double min, double max)
{

if (property == "apsis_angle")
	{
	cout << "Random apsis angle!" << endl;
	apsis_angle = rnd(min, max);
	cout << "Angle is " << apsis_angle * rad_to_deg << endl;
	}
else if (property == "eccentricity")
	{
	eccentricity = rnd(min, max);
	}
else if (property == "semimajor")
	{
	semimajor = rnd(min, max);
	}
periapsis = semimajor * (1.0 - eccentricity);
apoapsis = semimajor * (1.0 + eccentricity);

}

void Body::set_random_seed(int seed)
{
srand(seed);
}

void Body::set_ref_index(int index)
{
ref_index = index;
}

void Body::set_name(string set_name)
{
designation = set_name;
}

void Body::set_central_body(string name)
{
central_body = name;
}

int Body::get_ref_index()
{
return ref_index;
}

double Body::get_mass()
{
return mass;
}

double Body::get_radius()
{
return radius;
}

double Body::get_apoapsis()
{
return apoapsis;
}

double Body::get_periapsis()
{
return periapsis;
}


double Body::get_speed_apoapsis()
{
return speed_apoapsis;
}

double Body::get_speed_periapsis()
{
return speed_periapsis;
}

double Body::get_apsis_angle()
{
return apsis_angle;
}

double Body::get_inclination()
{
return inclination;
}

double Body::get_lan()
{
return lan;
}

double Body::get_flight_path_angle()
{
return flight_path_angle;
}

double Body::get_initial_velocity()
{
return speed_initial;
}

double Body::get_distance()
{
return distance;
}

std::string Body::get_designation()
{
return designation;
}

std::string Body::get_central_body()
{
return central_body;
}

bool Body::get_hyperbolic_trajectory()
{
return hyperbolic_trajectory;
}

void Body::list_properties()
{
cout << endl;
cout << designation << endl;
cout << "-----------" << endl;
cout << "Mass [m_earth]:        " << mass_earth << endl;
cout << "Radius [R_earth]:      " << radius_earth << endl;
cout << "Mean density [g/cm^3]: " << mass/(pow(radius,3.0) * 4./3. * pi  )/1000.0 << endl;
cout << "Surface gravity [g]:   " << G * mass / pow(radius, 2.0) /9.81 << endl;
if (hyperbolic_trajectory)
	{
	cout << "Passes:                " << central_body << endl;
	}
else
	{
	cout << "Orbits:                " << central_body << endl;
	}
cout << "Semimajor axis [Mkm]:  " << semimajor/1e9 << endl;
cout << "Eccentricity           " << eccentricity  << endl;
cout << "Periapsis [Mkm]:       " << periapsis /1e9 << endl;
if (hyperbolic_trajectory)
	{
	cout << "Hyp. excess v [km/s]:  " << speed_hyperbolic_excess/1e3 << endl;
	cout << "Initial v [km/s]:      " << speed_initial/1e3 << endl;
	cout << "Escape v [km/s]:       " << speed_esc/1e3 << endl;
	cout << "Theta path [deg]:      " << flight_path_angle * rad_to_deg << endl;
	}
else
	{
	cout << "Apoapsis [Mkm]:        " << apoapsis /1e9 << endl;
	cout << "Period [days]:         " << period/86400.0 << endl;
	}
cout << "Inclination [deg]      " << inclination * rad_to_deg  << endl;
cout << endl;
}
