
// atmosphere.cxx -- routines to define and compute atmosphere properties
//
// Written by Thorsten Renk, started 2018
//
// Copyright (C) 2018  Thorsten Renk - thorsten@science-and-fiction.org 
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301

#include <iostream>
#include <string>
#include <stdlib.h>
#include <math.h>
#include <iomanip>

using namespace std;

#include "constants.hxx"
#include "atmosphere.hxx"

void Atmosphere::add_component(string component, double set_fraction)
{

components[n_components].name = component;
components[n_components].fraction = set_fraction / 100.0;
components[n_components].molecular_weight = gas_properties.get_molecular_weight(component);
components[n_components].c_p = gas_properties.get_c_p(component);
n_components++;
}


void Atmosphere::set_surface_pressure (double set_value) 
{
surface_pressure = set_value * p_surface_earth;
}



void Atmosphere::compute_mixture(void)
{
int i;
double fraction_sum = 0.0;
double molecular_weight_sum = 0.0;
double c_p_sum = 0.0;

for (i=0; i<n_components; i++)
	{
	fraction_sum += components[i].fraction;
	}


for (i=0; i<n_components; i++)
	{
	components[i].fraction/=fraction_sum;
	}

for (i=0; i<n_components; i++)
	{
	c_p_sum += components[i].fraction * components[i].c_p;
	molecular_weight_sum  += components[i].fraction * components[i].molecular_weight;
	}



for (i=0; i< n_components; i++)
	{
	components[i].mass_fraction = (components[i].fraction * components[i].molecular_weight)/molecular_weight_sum;		
	}

mean_molecular_weight = molecular_weight_sum;
mean_c_p = c_p_sum;
scale_height = k_B * mean_radiative_temperature / (mean_molecular_weight * amu * surface_gravity);
adiabatic_lapse_rate = surface_gravity/mean_c_p;
column_mass = surface_pressure / surface_gravity;
}

void Atmosphere::check_phases(void)
{
int phase;
string phase_name;

cout << endl;
cout << "Atmosphere gases phase check:" << endl;
cout << "At " << surface_pressure/100000.0 / 1.013 << " atm pressure and " <<   min_surface_temperature << " K apoapsis temperature: " << endl;

for (int i=0; i< n_components;i++)
	{
	phase = gas_properties.get_phase(surface_pressure / 100000.0,  mean_surface_temperature , components[i].name);
	
	if (phase == 0) {phase_name = "solid";}
	else if (phase == 1) {phase_name = "liquid";}
	else if (phase == 2) {phase_name = "gaseous";}
	else {phase_name = "not implemented";}	

	cout << components[i].name << " is " << phase_name;

	if ((components[i].fraction > 0.05)&& (phase != 2)) {cout << " (issue: major component is not gaseous)";}	

	cout << endl;

	}
cout << endl;

}


double Atmosphere::get_transmission (double lambda)
{
int i;

double optical_depth;
double depth_sum = 0.0;

for (i=0; i< n_components; i++)
	{
	optical_depth = gas_properties.lineshape(lambda, components[i].name) * components[i].fraction * column_mass / (components[i].molecular_weight);
	depth_sum += optical_depth;
	}


double re_emission_factor = (1.0 - 1.0/(depth_sum + 2.0));

if (lambda < 5e-6) {re_emission_factor = 1.0;}

return 1.0 - (re_emission_factor * (1.0 - exp(- depth_sum)));
}

double Atmosphere::get_transmission (double lambda, double alt_min)
{
int i;

double optical_depth, column_mass_above;
double depth_sum = 0.0;

column_mass_above = exp(-alt_min/scale_height) * column_mass;

for (i=0; i< n_components; i++)
	{
	optical_depth = gas_properties.lineshape(lambda, components[i].name) * components[i].fraction * column_mass_above / (components[i].molecular_weight);
	depth_sum += optical_depth;
	}


double re_emission_factor = (1.0 - 1.0/(depth_sum + 2.0));

if (lambda < 5e-6) {re_emission_factor = 1.0;}

return 1.0 - (re_emission_factor * (1.0 - exp(- depth_sum)));
}


double Atmosphere::get_uv_transmission (double lambda)
{
return compute_chapman_function(10000, lambda * 1.0e9);
}

double Atmosphere::get_uv_transmission (double lambda, double alt_min)
{
return compute_chapman_function(10000, lambda * 1.0e9, alt_min);
}



double Atmosphere::get_absorption_by_altitude(double lambda, double alt, double delta_alt)
{


int i;

double optical_depth_alt, optical_depth_delta_alt;
double depth_sum_alt = 0.0;
double depth_sum_delta_alt = 0.0;
double transmission_at_alt, transmission_at_delta_alt;

double column_mass_alt;
double column_mass_delta_alt;

column_mass_alt = column_mass * exp(-alt/scale_height);
column_mass_delta_alt = column_mass * exp(-(alt + delta_alt)/scale_height);

for (i=0; i< n_components; i++)
	{
	optical_depth_alt = gas_properties.lineshape(lambda, components[i].name) * components[i].fraction * column_mass_alt / (components[i].molecular_weight);
	depth_sum_alt += optical_depth_alt;


	optical_depth_delta_alt = gas_properties.lineshape(lambda, components[i].name) * components[i].fraction * column_mass_delta_alt / (components[i].molecular_weight);
	depth_sum_delta_alt += optical_depth_delta_alt;

	}

transmission_at_alt = 1.0 - exp(-depth_sum_alt);
transmission_at_delta_alt = 1.0 - exp(-depth_sum_delta_alt);

return transmission_at_alt - transmission_at_delta_alt;
}


void Atmosphere::init_ozone_profile(int resolution)
{
atmosphere_O3_profile = new double[resolution];

for (int i=0; i< resolution; i++)
	{
	atmosphere_O3_profile[i] = 0.0;
	}
}

double Atmosphere::compute_chapman_function(int resolution, double lambda)
{
double initial_alt, alt_res, alt;
double intensity, molecule_mass;
double P_abs;

initial_alt = 20.0 * scale_height;
alt_res = initial_alt / resolution;

intensity = 1.0;

for (int i=0; i< resolution; i++)
	{
	alt = initial_alt -  i * alt_res;
	P_abs = 0.0;
	for (int j=0; j< n_components; j++)
		{
		molecule_mass = amu * components[j].molecular_weight;
		P_abs +=  alt_res * gas_properties.lineshape_uv(lambda, components[j].name) * components[j].fraction  * exp(-alt/scale_height) * (column_mass/scale_height) / molecule_mass;
		}
	//P_abs = exp(-alt/scale_height) * column_mass/scale_height;
	//loss = intensity * P_abs;
	//cout << "Alt: " << alt << " P abs: " << P_abs << " loss: " << loss << endl;
	//cout << alt << " " << loss << endl;
	//intensity *= (1.0 - P_abs);
	intensity *= exp(-P_abs);
	}
//cout << "Final intensity: " << intensity << endl;

return intensity;
}


double Atmosphere::compute_chapman_function(int resolution, double lambda, double alt_min)
{
double initial_alt, alt_res, alt;
double intensity, molecule_mass;
double P_abs;

initial_alt = 20.0 * scale_height;
alt_res = initial_alt / resolution;

intensity = 1.0;

for (int i=0; i< resolution; i++)
	{
	alt = initial_alt -  i * alt_res;
	if (alt < alt_min) {break;}
	P_abs = 0.0;
	for (int j=0; j< n_components; j++)
		{
		molecule_mass = amu * components[j].molecular_weight;
		P_abs +=  alt_res * gas_properties.lineshape_uv(lambda, components[j].name) * components[j].fraction  * exp(-alt/scale_height) * (column_mass/scale_height) / molecule_mass;
		}
	intensity *= exp(-P_abs);
	}


return intensity;

}

double* Atmosphere::compute_chapman_heating_profile(int resolution, double lambda, double intensity)
{
double initial_alt, alt_res, alt;
double molecule_mass;
double P_abs;

double* result_array;

result_array = new double[resolution];


initial_alt = 20.0 * scale_height;
alt_res = initial_alt / resolution;


for (int i=0; i< resolution; i++)
	{
	alt = initial_alt -  i * alt_res;
	P_abs = 0.0;
	for (int j=0; j< n_components; j++)
		{
		molecule_mass = amu * components[j].molecular_weight;
		if (components[j].name != "O3")
			{
			P_abs +=  alt_res * gas_properties.lineshape_uv(lambda, components[j].name) * components[j].fraction  * exp(-alt/scale_height) * (column_mass/scale_height) / molecule_mass;
			}
		else 
			{
			P_abs +=  alt_res * gas_properties.lineshape_uv(lambda, components[j].name) * atmosphere_O3_profile[i] / molecule_mass;
			}
		}
	result_array[i] = intensity * (1.0 - exp(-P_abs)) / alt_res;
	intensity *= exp(-P_abs);
	}

return result_array;
}




double* Atmosphere::compute_chapman_o3_profile(int resolution, double lambda, double intensity)
{
double initial_alt, alt_res, alt;
double molecule_mass, fraction;
double P_abs;

double* result_array;

result_array = new double[resolution];


initial_alt = 20.0 * scale_height;
alt_res = initial_alt / resolution;

fraction = 0.0;

for (int i=0; i< n_components; i++)
	{
	if (components[i].name == "O2") 
		{
		fraction = components[i].fraction;
		}
	}

for (int i=0; i< resolution; i++)
	{
	alt = initial_alt -  i * alt_res;
	molecule_mass = amu * 32.0;
	P_abs =  alt_res * gas_properties.lineshape_uv(lambda, "O2") * fraction  * exp(-alt/scale_height) * (column_mass/scale_height) / molecule_mass;
	result_array[i] = intensity * (1.0 - exp(-P_abs)) / alt_res / 39.33 * 6.42e-3;
	atmosphere_O3_profile[i] += result_array[i];
	intensity *= exp(-P_abs);
	}



return result_array;
}


double Atmosphere::get_surface_pressure()
{
return surface_pressure;
}

double Atmosphere::get_cp()
{
return mean_c_p;
}



void Atmosphere::list_properties (void)
{
cout << endl;
cout << "Atmosphere" << endl;
cout << "----------" << endl;
cout << "Surface pressure [mbar]:   " << surface_pressure * Pa_to_mbar << endl;
cout << "Scale height [m]:          " << scale_height << endl;
cout << "Column mass [kg]:          " << column_mass << endl;
cout << "Dry lapse rate [K/km]:     " << adiabatic_lapse_rate << endl;
cout << "Surface temperature [K]:   " << mean_surface_temperature << endl;
cout << "Emissive temp.      [K]:   " << mean_radiative_temperature << endl; 
cout << "Heat capacity CP [J/K*kg]: " << mean_c_p << endl;

//compute_chapman_function(1000, 300.0);

//cout << "Phase: " << gas_properties.get_phase(surface_pressure / 100000.0, 68.0, "N2") << endl;
}

void Atmosphere::list_composition(void)
{
int i;

cout << endl;
cout << "Atmospheric composition" << endl;
cout << "-----------------------" << endl;
cout << "     by volume     by mass" << endl;
for (i=0; i< n_components; i++)
	{
	cout << setw(4) << left <<  components[i].name << ": "    << showpoint << setprecision(4) << setw(7) << components[i].fraction * 100.0 << " %     " << setw(7) << components[i].mass_fraction * 100.0 << " %" << endl; 

	}

}


