// state_vector.hxx -- routines to forward a spacecraft state vector
// and obtain its properties
//
// Written by Thorsten Renk, started 2017
//
// Copyright (C) 2017  Thorsten Renk - thorsten@science-and-fiction.org 
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301




#include <string>

#include "units.hxx"
#include "options.hxx"
#include "burn.hxx"
#include "site.hxx"
#include "celestial_body.hxx"
#include "rendezvous_target.hxx"


class Craft {


	public:
	long double acceleration;
	long double mass;
	long double isp;
	long double propellant_mass;
	long double dry_mass;	
	long double thrust;
	long double cross_section_area;
	
	bool is_defined;

};


class StateVector {
	long double x;
	long double y;
	long double z;

	long double vx;
	long double vy;
	long double vz;

	long double stored_x;
	long double stored_y;
	long double stored_z;

	long double stored_xnorm_tig;
	long double stored_ynorm_tig;
	long double stored_znorm_tig;

	long double stored_vxnorm_tig;
	long double stored_vynorm_tig;
	long double stored_vznorm_tig;

	long double stored_vx;
	long double stored_vy;
	long double stored_vz;

	long double gx;
	long double gy;
	long double gz;

	long double bc_x;
	long double bc_y;
	long double bc_z;

	long double bc_vx;
	long double bc_vy;
	long double bc_vz;

	long double dt;
	long double evolution_time;
	long double time_offset;
	long double stored_evolution_time;
	long double earth_rotation_angle;

	long double stored_earth_rotation_angle;

	long double semi_major;
	long double semi_minor;
	long double eccentricity;
	long double inclination;
	long double ascending_node;
	long double argument_of_periapsis;
	long double true_anomaly;

	long double peg4_rei;

	long double peg4_thetaT_act;
	long double peg4_H_act;
	long double peg4_vspeed_act;
	long double peg4_hspeed_act;
	long double peg4_rei_act;

	long double fit_radius_last;
	long double ei_altitude;

	int gravity_model;
	int earth_model;
	int burn_model;
	int time_format;

	int orbit_counter;
	int low_thrust_counter;

	int plot2d_counter;
	int plot2d_x_option;
	int plot2d_y_option;
	int plot2d_resolution;

	int plot3d_counter;
	int plot3d_x_option;
	int plot3d_y_option;
	int plot3d_z_option;
	int plot3d_resolution;

	int fit_max_iterations;

	bool plot2d_flag;
	bool plot3d_flag;
	bool hypothetical_flag;
	bool fit_log_flag;

	bool list_ei_orbital;
	bool list_ei_sv;
	bool list_ei_pos;

	bool find_ei;

	bool fit_automatic_burn2[10];

	Burn burn_array [10];
	int num_burns;
	int next_burn_index;
	int current_burn_index;

	Site landing_site;
	Site launch_site;
	bool launch_site_defined;

	Craft craft;

	CelestialBody body_array[5];
	BarycenterCoords barycenter;

	bool target_defined;
	RendezvousTarget target;

	int num_bodies;

	Units converter;

	void do_timestep();
	void do_plot2d();
	void do_plot3d();
	void compute_gravity();
	void compute_multibody_gravity();
	void compute_elements();
	void compute_peg4();
	void compute_earth_rotation();
	void list_burn(int i);
	long double earth_radius();
	long double current_thetaT();
	long double course();
	long double radius();
	long double vtot();

	//long double dot_product (long double vec1[], long double vec2[]);
	//long double norm (long double vec[]);
	//long double cross_product_x ( long double vec1[3], long double vec2[3]);
	//long double cross_product_y ( long double vec1[3], long double vec2[3]);
	//long double cross_product_z ( long double vec1[3], long double vec2[3]);
	//long double orthogonalize_x ( long double vec1[3], long double vec2[3]);
	//long double orthogonalize_y ( long double vec1[3], long double vec2[3]);
	//long double orthogonalize_z ( long double vec1[3], long double vec2[3]);
	double get_latitude_rad();

	bool check_condition();


	public:
		
	void init (Units conv, Options options);

	void set_pos (long double x, long double y, long double z);
	void set_vel (long double vx, long double vy, long double vz);

	void add_peg7_burn (long double t, long double dv_prograde, long double dv_radial, long double dv_normal, std::string name);
	void add_peg4_burn (long double t, long double thetaT, long double H, long double c1, long double c2,  std::string name, int flag);
	void add_peg4_burn_optimized (long double t, long double thetaT, long double H, long double c1, long double c2, long double rei, std::string name);
	void add_low_thrust_burn (long double t, long double dv_prograde, long double dv_radial, long double dv_normal, long double tvx, long double tvy, long double tvz, long double tolerance, std::string name);
	void add_lambert_burn (long double t1, long double t2, long double xoffset, long double yoffset, long double zoffset, bool flag, bool flag2, std::string name);

	void add_body(std::string name, std::string ephemeris_file, long double mass);

	void target_init(Options options);
	void target_set_pos(long double x, long double y, long double z);
	void target_set_vel(long double vx, long double vy, long double vz);

	void set_burn_ops(int i, int ops);
	void set_burn_plane_flag(int i, bool flag);

	void evolve_to (long double time);

	double get_longitude();
	double get_latitude();
	double get_altitude();
	double get_earth_radius(double lat);


	void list ();
	void list_orbital();
	void list_position();
	void list_craft();
	void list_bodies();
	void list_lagrange();
	void list_target_sv();

	int plot2d_index;
	double plot2d_x[10000];
	double plot2d_y[10000];

	int plot3d_index;
	double plot3d_x[10000];
	double plot3d_y[10000];
	double plot3d_z[10000];
	
};
