// state_vector.cxx -- routines to forward a spacecraft state vector
// and obtain its properties
//
// Written by Thorsten Renk, started 2017
//
// Copyright (C) 2017  Thorsten Renk - thorsten@science-and-fiction.org 
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301

#include <iostream>
#include <iomanip>
#include <stdlib.h>
#include <math.h>
#include <cmath>

#include "constants.hxx"
#include "state_vector.hxx"
#include "math_helpers.hxx"



using namespace std;
using namespace MathHelpers;


void StateVector::init (Units conv, Options options)
{

converter = conv;


dt = options.timestep;
evolution_time = 0.0;
earth_rotation_angle = 0.0;


gravity_model = options.gravity_model;
earth_model = options.earth_model;
burn_model = options.burn_model;
time_format = options.time_format_flag;


plot2d_flag = options.plot2d_flag;
plot2d_x_option = options.plot2d_x_option;
plot2d_y_option = options.plot2d_y_option;
plot2d_resolution = options.plot2d_resolution;


plot3d_flag = options.plot3d_flag;
plot3d_x_option = options.plot3d_x_option;
plot3d_y_option = options.plot3d_y_option;
plot3d_z_option = options.plot3d_z_option;
plot3d_resolution = options.plot3d_resolution;

fit_log_flag = options.fit_log;
fit_radius_last = 0.0;
fit_max_iterations = options.fit_max_iterations;


for (int i=0; i<10; i++)
	{
	fit_automatic_burn2[i] = options.burn_lambert_automatic_burn2[i];
	}


list_ei_orbital = options.list_ei_orbital;
list_ei_sv = options.list_ei_sv;
list_ei_pos = options.list_ei_pos;

if (options.craft_defined)
	{
	craft.mass = converter.convert_mass_input(options.craft_mass);
	craft.propellant_mass = converter.convert_mass_input(options.craft_mass_propellant);
	craft.dry_mass = craft.mass - craft.propellant_mass;
	craft.isp = options.craft_isp;
	craft.thrust = converter.convert_force_input(options.craft_thrust);	
	craft.cross_section_area = options.craft_area;
	craft.is_defined = true;
	list_craft();
	}
else
	{
	if (options.craft_acceleration_defined)
		{craft.acceleration = converter.convert_length_input(options.craft_acceleration);}
	else
		{craft.acceleration = options.craft_acceleration;}
	craft.is_defined = false;
	}


if (options.target_flag == 1)
	{
	target_defined = true;
	}


if ((list_ei_orbital) || (list_ei_sv) || (list_ei_pos))
	{
	find_ei = true;
	ei_altitude = 121920.0;
	}
else
	{
	find_ei = false;
	}
	
num_bodies = options.num_bodies;

if (num_bodies > 0)
	{
	barycenter.init(body_array, num_bodies);
	}



time_offset = options.initial_met;

orbit_counter = 0;
current_burn_index = 0;
next_burn_index = 0;

//cout << "Burn index is now " << next_burn_index << endl;

hypothetical_flag = false;
if (fit_log_flag)
	{cout << "============ HYPOTHETICAL MODE OFF ===================" << endl;}

if (options.landing_site_set)
	{
	if (landing_site.is_known(options.landing_site_name))
		{
		landing_site.init_by_name (options.landing_site_name);
		}
	else
		{
		landing_site.init(options.landing_site_lat, options.landing_site_lon, options.landing_site_name);
		}
	}

if (options.launch_site_set)
	{

	if (launch_site.is_known(options.launch_site_name))
		{
		launch_site.init_by_name (options.launch_site_name);
		}
	else
		{
		launch_site.init(options.launch_site_lat, options.launch_site_lon, options.launch_site_name);
		}
	launch_site_defined = true;
	}
else
	{
	launch_site_defined = false;
	}

}

void StateVector::add_peg7_burn (long double t, long double dvp, long double dvr, long double dvn, string name)
{
cout << "Adding PEG-7 burn" << endl;


Burn burn;
burn.init(converter.convert_length_input(dvp), converter.convert_length_input(dvr), converter.convert_length_input(dvn), t, name, burn_model);
burn.set_acceleration (craft.acceleration);
burn_array[num_burns] = burn;
num_burns += 1;

}


void StateVector::add_peg4_burn (long double t, long double thetaT, long double H, long double c1, long double c2, string name, int flag)
{
cout << "Adding PEG-4 burn" << endl;

Burn burn;
burn.init_peg4(thetaT, converter.convert_length_input(H), converter.convert_length_input(c1), c2,  t, name, burn_model, flag);
burn.set_acceleration (craft.acceleration);
burn.set_fit_log(fit_log_flag);
burn_array[num_burns] = burn;
num_burns += 1;
}

void StateVector::add_peg4_burn_optimized (long double t, long double thetaT, long double H, long double c1, long double c2, long double rei, string name)
{

if (landing_site.is_defined() == false)
	{
	cout << "Optimized PEG-4 burn is undefined without landing site, aborting..." << endl; 
	exit(0);
	}

cout << "Adding optimized PEG-4 burn" << endl;

Burn burn;
burn.init_peg4_optimized(thetaT, converter.convert_length_input(H), converter.convert_length_input(c1), c2, converter.convert_km_nm_input(rei), t, name, burn_model);
burn.set_acceleration (craft.acceleration);
burn.set_fit_log(fit_log_flag);
burn_array[num_burns] = burn;
num_burns += 1;
}

void StateVector::add_low_thrust_burn (long double t, long double dvp, long double dvr, long double dvn, long double tvx, long double tvy, long double tvz, long double tolerance, string name)
{
cout << "Adding low thrust burn" << endl;

long double tgt_vec[3];

tgt_vec[0] = tvx;
tgt_vec[1] = tvy;
tgt_vec[2] = tvz;
long double threshold = tolerance;

Burn burn;
burn.init_low_thrust(converter.convert_length_input(dvp), converter.convert_length_input(dvr), converter.convert_length_input(dvn), t, tgt_vec, threshold, name, burn_model);
burn.set_acceleration (craft.acceleration);
burn_array[num_burns] = burn;
num_burns += 1;
}


void StateVector::add_lambert_burn (long double t1, long double t2, long double xoffset, long double yoffset, long double zoffset, bool yfit, bool auto2, string name)
{
if (target_defined == false)
	{
	cout << "Lambert burn is undefined without a target, aborting..." << endl; 	
	exit(0);
	}

cout << "Adding Lambert targeted burn sequence" << endl;


Burn burn;
burn.init_lambert(t1, t2, converter.convert_length_input(xoffset), converter.convert_length_input(yoffset), converter.convert_length_input(zoffset), name, burn_model);
burn.set_acceleration (craft.acceleration);
burn.set_fit_log(fit_log_flag);
burn.set_y_option(yfit);
burn_array[num_burns] = burn;
num_burns += 1;

if (auto2) 
	{
	num_burns += 1;
	cout << "Adding placeholder for second Lambert targeted burn" << endl;
	}

}


void StateVector::add_body(string name, string ephemeris_file, long double GM)
{
CelestialBody body;

body.init(name, ephemeris_file, GM);
body_array[num_bodies] = body;

num_bodies += 1;
}


void StateVector::target_init(Options options)
{
target.init(options);

}

void StateVector::target_set_pos(long double x, long double y, long double z)
{
long double x_set = converter.convert_length_input(x);
long double y_set = converter.convert_length_input(y);
long double z_set = converter.convert_length_input(z);

target.set_pos(x_set, y_set, z_set);
}

void StateVector::target_set_vel(long double vx, long double vy, long double vz)
{
long double vx_set = converter.convert_length_input(vx);
long double vy_set = converter.convert_length_input(vy);
long double vz_set = converter.convert_length_input(vz);

target.set_vel(vx_set, vy_set, vz_set);
}

void StateVector::set_burn_ops(int i, int ops)
{
if (launch_site_defined)
	{burn_array[i].set_ops(ops);}
else
	{cout << "Cannot use OPS 1 angle definitions without a launch site! Aborting..." << endl; exit(0);}
}

void StateVector::set_burn_plane_flag(int i, bool flag)
{
burn_array[i].set_plane_flag(flag);
}


void StateVector::set_pos (long double set_x, long double set_y, long double set_z)
{


x = converter.convert_length_input(set_x); 
y = converter.convert_length_input(set_y);
z = converter.convert_length_input(set_z);

if (launch_site_defined)
	{launch_site.set_launch_site(time_offset, get_latitude(), get_longitude());}
}

void StateVector::set_vel (long double set_vx, long double set_vy, long double set_vz)
{

vx = converter.convert_length_input(set_vx);
vy = converter.convert_length_input(set_vy);
vz = converter.convert_length_input(set_vz);

}

bool StateVector::check_condition()
{


	
bool condition = false;

if (hypothetical_flag == false)
	{

	if (find_ei)
		{
		if ((radius() - earth_radius()) < ei_altitude)
			{
			find_ei = false;

			cout << endl;
			cout << "At entry interface: " << endl;

			  if (list_ei_sv)
  				{list();}
  
			  if (list_ei_pos)
				{list_position();}

			  if (list_ei_orbital)
				{list_orbital();}

			cout << endl;
			}
		}




	if (plot2d_index > 9999)
		{
		cout << "2d plot out of points, ending evolution!" << endl;
		return false;
		}
	else if  (plot3d_index > 9999)
		{
		cout << "3d plot out of points, ending evolution!" << endl;
		return false;
		}
	else
		{
		return true;
		}
	}
else if ((burn_array[current_burn_index].get_burn_type() > 0) && (burn_array[current_burn_index].get_burn_type() < 4)) // the condition depends on the chosen fit strategy for PEG 4 fits
	{
	//cout << "PEG BRANCH " << endl;

	int fit_strategy = burn_array[current_burn_index].get_strategy();

	//cout << fit_strategy << endl;


	if (fit_strategy == 0)
		{
		double radius = sqrt(pow(x,2.0) + pow(y,2.0) + pow(z, 2.0));
		double current_alt = radius - earth_radius();
		double H_target = burn_array[current_burn_index].get_H_target();
		if  (current_alt <  H_target)
			{condition = true;}
		else
			{condition = false;}
		}
	else if (fit_strategy == 1)
		{
		double radius = sqrt(pow(x,2.0) + pow(y,2.0) + pow(z, 2.0));
		double current_alt = radius - earth_radius();
		double H_target = burn_array[current_burn_index].get_H_target();
		if  (current_alt >  H_target)
			{condition = true;}
		else
			{condition = false;}
		}
	else if (fit_strategy == 2)
		{
		double radius = sqrt(pow(x,2.0) + pow(y,2.0) + pow(z, 2.0));

		//cout << radius << endl;

		if (radius > fit_radius_last) 
			{
			compute_elements();
			if ( abs(radius - ((1.0 - eccentricity) * semi_major)) > 5000.0)
				{
				//cout << "Radius: " << radius << " apsis: " << (1.0 - eccentricity) * semi_major << endl;
				//cout << "Difference: " << radius - (1.0 - eccentricity) * semi_major << endl;
				//cout << "Spurious apsis, continuing..." << endl;
				condition = false;
				fit_radius_last = 1e20;
				}

			else 
				{
				//cout << "Radius: " << radius << " apsis: " << (1.0 - eccentricity) * semi_major << endl;
				//cout << "Difference: " << radius - (1.0 - eccentricity) * semi_major << endl;
				condition = true;
				fit_radius_last = 1e20;
				}
			}
		else
			{
			condition = false;
			fit_radius_last = radius;
			}

		}
	else if (fit_strategy == 3)
		{
		double radius = sqrt(pow(x,2.0) + pow(y,2.0) + pow(z, 2.0));

		if (radius < fit_radius_last) 
			{
			compute_elements();
			if ( abs(radius - ((1.0 + eccentricity) * semi_major)) > 5000.0)
				{
				//cout << "Radius: " << radius << " apsis: " << (1.0 + eccentricity) * semi_major << endl;
				//cout << "Spurious apsis, continuing..." << endl;
				condition = false;
				fit_radius_last = 0.0;
				}
			else
				{
				condition = true;
				fit_radius_last = 0.0;
				}
			}
		else
			{
			condition = false;
			fit_radius_last = radius;
			}

		}
	else if (fit_strategy == 4)
		{
		if (current_thetaT() >  burn_array[current_burn_index].get_thetaT())
			{
			//cout << "Current thetaT " << current_thetaT() << endl;
			//cout << "Current time " << evolution_time << endl;
			condition = true;
			}
		else
			{
			condition = false;
			}
		}
	}
else if (burn_array[current_burn_index].get_burn_type() == 5) //  Lambert targeting
		{


		if (evolution_time > burn_array[current_burn_index].get_tig2() )
			{
			condition = true;



			if (burn_array[current_burn_index].is_dry_run())
				{				
				burn_array[current_burn_index].set_lambert_target(target.get_x(), target.get_y(), target.get_z());
				burn_array[current_burn_index].set_dry_run(false);
				
				burn_array[current_burn_index].lambert_guess_initial(target.get_prox_x(x,y,z,vx,vy,vz), target.get_prox_y(x,y,z,vx,vy,vz), target.get_prox_z(x,y,z,vx,vy,vz), target.get_orbital_period(), static_cast<int>(0.5 * fit_max_iterations));
				
				x = stored_x;
				y = stored_y;
				z = stored_z;

				vx = stored_vx;
				vy = stored_vy;
				vz = stored_vz;

				target.reset();

				burn_array[current_burn_index].set_propellant (craft.propellant_mass);

				earth_rotation_angle = stored_earth_rotation_angle;
				evolution_time = stored_evolution_time;

				
				return true;
				}

			}
		else
			{return true;}	
	
		}

else {
	cout << "Error: no well defined condition!" << endl;
	exit(0);
	}



// now decide upon restarting fits
	

if ((burn_array[current_burn_index].get_burn_type() > 0) && (burn_array[current_burn_index].get_burn_type() < 4)) // PEG-4 fit
		{
		if (condition)
			{

			compute_peg4();
			burn_array[current_burn_index].peg4_improve(peg4_thetaT_act, peg4_H_act, peg4_vspeed_act, peg4_hspeed_act, peg4_rei_act);

			if (burn_array[current_burn_index].is_converged() == true)
				{
				if (fit_log_flag)
					{cout << "============ HYPOTHETICAL MODE OFF ====================" << endl;}
				hypothetical_flag = false;
				list_burn(current_burn_index);
				}

			x = stored_x;
			y = stored_y;
			z = stored_z;

			vx = stored_vx;
			vy = stored_vy;
			vz = stored_vz;

			target.reset();

			burn_array[current_burn_index].set_propellant (craft.propellant_mass);

			earth_rotation_angle = stored_earth_rotation_angle;
			evolution_time = stored_evolution_time;

			if (fit_log_flag)
				{
				cout << "Iterations: " << burn_array[current_burn_index].get_num_iterations() << endl;
				}
			next_burn_index -=1;
			num_burns += 1;



			if (burn_array[current_burn_index].get_num_iterations() < fit_max_iterations) 
				{
				return true;
				}
			else
				{
				cout << "Fit remains unconverged after " << fit_max_iterations <<  " iterations, aborting!" << endl; exit(0);
				hypothetical_flag = false;
				return true;
				}
			}
		else
			{
			return true;
			}
		}
	else if (burn_array[current_burn_index].get_burn_type() == 5)
		{
		if (condition)
			{
			burn_array[current_burn_index].lambert_improve(target.get_prox_x(x,y,z,vx,vy,vz), target.get_prox_y(x,y,z,vx,vy,vz), target.get_prox_z(x,y,z,vx,vy,vz));
			

			long double vx_prox, vy_prox, vz_prox;



			if (burn_array[current_burn_index].is_converged() == true)
				{
				if (fit_log_flag)
					{cout << "============ HYPOTHETICAL MODE OFF ====================" << endl;}
				hypothetical_flag = false;

				vx_prox = target.get_prox_vx(x,y,z,vx,vy,vz);
				vy_prox = target.get_prox_vy(x,y,z,vx,vy,vz);
 				vz_prox = target.get_prox_vz(x,y,z,vx,vy,vz);

				if (burn_array[current_burn_index].is_yfit_active() == false) {vy_prox = 0.0;}
				burn_array[current_burn_index].set_lambert2_parameters(vx_prox, -vy_prox, vz_prox);
				list_burn(current_burn_index);



				}

			x = stored_x;
			y = stored_y;
			z = stored_z;

			vx = stored_vx;
			vy = stored_vy;
			vz = stored_vz;

			target.reset();


			burn_array[current_burn_index].set_propellant (craft.propellant_mass);

			earth_rotation_angle = stored_earth_rotation_angle;
			evolution_time = stored_evolution_time;


			if (fit_log_flag)
				{
				cout << "Iterations: " << burn_array[current_burn_index].get_num_iterations() << endl;
				}
			next_burn_index -=1;
			num_burns += 1;

			if (burn_array[current_burn_index].is_converged() == true)
				{
				long double tig2 = burn_array[current_burn_index].get_tig2();
				
				if (fit_automatic_burn2[current_burn_index])
					{
					if (burn_model > 0)
						{
						long double vtot = sqrt(pow(vx_prox,2.0) + pow(vy_prox,2.0) + pow(vz_prox,2.0));
						long double delta_tig = vtot/ burn_array[current_burn_index].get_acceleration();
						tig2 -= 0.5*delta_tig;
						}
					// need to insert the burn into the placeholder slot
					int num_burns_tmp = num_burns;

					num_burns = current_burn_index+1;


					add_peg7_burn (tig2, converter.convert_length_output(vx_prox), converter.convert_length_output(vz_prox), converter.convert_length_output(-vy_prox), burn_array[current_burn_index].get_name() + "-2" );
					num_burns = num_burns_tmp;

					}
				}

			if (burn_array[current_burn_index].get_num_iterations() < fit_max_iterations) 
				{
				return true;
				}
			else
				{
				cout << "Fit remains unconverged after " << fit_max_iterations <<  " iterations, aborting!" << endl; exit(0);
				hypothetical_flag = false;
				return true;
				}

			}

		else
			{
			return true;
			}
		
			
		}
	

}


double StateVector::get_latitude ()
{
return get_latitude_rad() * rad_to_deg;
}


double StateVector::get_longitude()
{
double angle = atan2(y,x) * rad_to_deg + earth_rotation_angle;

while (angle < -180.0)  {angle +=360.0;}

//if (angle < -180.0) {angle +=360.0;}

return angle;
}

double StateVector::get_latitude_rad ()
{
double radius = sqrt(pow(x,2.0) + pow(y,2.0) + pow(z, 2.0));
return asin(z/radius);
}


double StateVector::get_altitude()
{
return sqrt(pow(x,2.0) + pow(y,2.0) + pow(z, 2.0)) - earth_radius();
}

void StateVector::list()
{

cout << setprecision(8) ;
cout << endl;
cout << "Current state vector" << endl;
cout << "====================" << endl;

cout << "x [" << converter.get_m_ft() << "]:  " << converter.convert_length_output(x)  << endl;
cout << "y [" << converter.get_m_ft() << "]:  " << converter.convert_length_output(y)  << endl;
cout << "z [" << converter.get_m_ft() << "]:  " << converter.convert_length_output(z)  << endl;

cout << "vx [" << converter.get_m_ft_s() << "]: " << converter.convert_length_output(vx)  << endl;
cout << "vy [" << converter.get_m_ft_s() << "]: " << converter.convert_length_output(vy)  << endl;
cout << "vz [" << converter.get_m_ft_s() << "]: " << converter.convert_length_output(vz)  << endl;

cout << "T:  " << evolution_time << endl;
cout << endl;


}


void StateVector::list_position()
{
double radius = sqrt(pow(x,2.0) + pow(y,2.0) + pow(z, 2.0));
string km_nm_string = converter.get_km_nm();

cout << "Latitude  [deg]:            " << get_latitude() << endl;
cout << "Longitude [deg]:            " << get_longitude() << endl;
cout << "Alt. above surface ["<<km_nm_string<<"]:    " << converter.convert_km_nm_output( (radius - earth_radius()))  << endl;

if (landing_site.is_defined())
	{
	cout << "Dist. to landing site [" << km_nm_string << "]: " << converter.convert_km_nm_output(landing_site.distance_from(get_latitude(), get_longitude())) << endl;
	long double bearing =  landing_site.bearing_from (get_latitude(), get_longitude()) + 180.0;
	if (bearing > 360.0) {bearing -= 360.0;}
	else if (bearing < 0.0) {bearing += 360.0;}
	cout << "Bearing to landing site:    " << bearing << endl;
	cout << "Current course:             " << course() << endl;
	long double deltaAz = course() - bearing;

	cout << "Delta Az to landing site    " << deltaAz << endl;
	}

}

void StateVector::list_orbital()
{

compute_elements();

double radius = sqrt(pow(x,2.0) + pow(y,2.0) + pow(z, 2.0));
double speed = sqrt(pow(vx,2.0) + pow(vy,2.0) + pow(vz, 2.0));


cout << "Radius ["<<converter.get_km_nm()<<"]:             " << converter.convert_km_nm_output(radius) << endl;
cout << "Alt. above surface ["<<converter.get_km_nm()<<"]: " << converter.convert_km_nm_output(radius - earth_radius()) << endl;
cout << "Apoapsis ["<< converter.get_km_nm()<<"]:           " << converter.convert_km_nm_output( ((1.0 + eccentricity) * semi_major) - R_earth) << endl;
cout << "Periapsis ["<< converter.get_km_nm()<<"]:          " << converter.convert_km_nm_output( ((1.0 - eccentricity) * semi_major) - R_earth) << endl;
cout << "Speed ["<<converter.get_m_ft_s()<<"]:             " << converter.convert_length_output(speed) << endl;
cout << "Longitude [deg]:         " << get_longitude() << endl;
cout << "Latitude  [deg]:         " << get_latitude() << endl;
cout << "Semi-major axis ["<<converter.get_km_nm()<<"]:    " << converter.convert_km_nm_output(semi_major) << endl;
cout << "Eccentricity:            " << eccentricity << endl;
cout << "Inclination [deg]:       " << inclination * rad_to_deg << endl;
cout << "Asc. node lon. [deg]:    " << ascending_node * rad_to_deg << endl;
cout << "Arg. of periapsis [deg]: " << argument_of_periapsis * rad_to_deg << endl;
cout << "True anomaly [deg]:      " << true_anomaly * rad_to_deg << endl;
cout << "Orbital period [s]:      " << 2.0 * 3.1415926 * sqrt(pow(semi_major, 3.0) / GM) << endl;
}


void StateVector::list_burn(int i)
{
cout << endl;

cout << "Parameters of burn: " << burn_array[i].get_name() << endl;

if (time_format == 0)
	{
	cout << "TIG             : " << burn_array[i].get_tig() + time_offset << endl;;
	}
else
	{
	cout << "TIG             : " << converter.assemble_timestring(burn_array[i].get_tig() + time_offset) << endl;;
	}

int burn_type = burn_array[i].get_burn_type();

if ((burn_type == 1) || (burn_type == 2))
	{
cout << "PEG-4" << endl;
cout << "H               : " << converter.convert_km_nm_output(burn_array[i].get_H()) << endl;
cout << "theta T         : " << burn_array[i].get_thetaT() << endl;
cout << "c1              : " << converter.convert_length_output(burn_array[i].get_c1()) << endl;
cout << "c2              : " << burn_array[i].get_c2() << endl;
	}
if (burn_type == 0)
	{
	cout << "PEG-7" << endl;
	}
else if (burn_type == 4)
	{
	cout << "Low thrust" << endl;
	}
else if (burn_type == 5)
	{
	cout << burn_array[i].get_name() << "-1" << endl;
	}
cout << "Dx (prograde)   : " << converter.convert_length_output(burn_array[i].get_delta_v_prograde()) << endl;
cout << "Dy (normal)     : " << converter.convert_length_output(burn_array[i].get_delta_v_normal()) << endl;
cout << "Dz (radial)     : " << converter.convert_length_output(burn_array[i].get_delta_v_radial()) << endl;

if (burn_type == 5)
	{
	cout << endl;
	if (time_format == 0)
		{
		cout << "TIG             : " << burn_array[i].get_tig2() + time_offset << endl;;
		}
	else
		{
		cout << "TIG             : " << converter.assemble_timestring(burn_array[i].get_tig2() + time_offset) << endl;;
		}
	cout << burn_array[i].get_name() << "-2" << endl;
	cout << "Dx (prograde)   : " << converter.convert_length_output(burn_array[i].get_delta_v_prograde2()) << endl;
	cout << "Dy (normal)     : " << converter.convert_length_output(burn_array[i].get_delta_v_normal2()) << endl;
	cout << "Dz (radial)     : " << converter.convert_length_output(burn_array[i].get_delta_v_radial2()) << endl;

	}


cout << endl;

}


void StateVector::list_craft()
{
cout << endl;
cout << "Spacecraft properties" << endl;
cout << "total mass [" << converter.get_kg_lb() << "]:      " << converter.convert_mass_output(craft.mass) << endl;
cout << "dry mass [" << converter.get_kg_lb() << "]:        " << converter.convert_mass_output(craft.dry_mass) << endl;
cout << "propellant mass [" << converter.get_kg_lb() << "]: " << converter.convert_mass_output(craft.propellant_mass) << endl;
cout << "thrust [" << converter.get_N_lb() << "]:           " << converter.convert_force_output(craft.thrust) << endl;
cout << "ISP [s]:              " << craft.isp << endl;
cout << "acceleration [g]:     " << craft.thrust/(craft.mass * 9.81) << endl;
cout << endl;
}

void StateVector::list_bodies()
{
cout << endl;
cout << "Celestial body properties" << endl;

for (int i=0; i<num_bodies; i++)
	{
	cout << body_array[i].get_name() << endl;
	cout << "x [" << converter.get_km_nm() << "]:    " << converter.convert_km_nm_output(body_array[i].get_pos_x()) << endl;
	cout << "y [" << converter.get_km_nm() << "]:    " << converter.convert_km_nm_output(body_array[i].get_pos_y()) << endl;
	cout << "z [" << converter.get_km_nm() << "]:    " << converter.convert_km_nm_output(body_array[i].get_pos_z()) << endl;
	cout << "vx [" <<converter.get_m_ft_s() <<"]:  " << converter.convert_length_output(body_array[i].get_vx()) << endl;
	cout << "vy [" <<converter.get_m_ft_s() <<"]:  " << converter.convert_length_output(body_array[i].get_vy()) << endl;
	cout << "vz [" <<converter.get_m_ft_s() <<"]:  " << converter.convert_length_output(body_array[i].get_vz()) << endl;
	}

}

void StateVector::list_lagrange()
{

if (num_bodies > 0)
	{
	cout << endl;

	cout << "Lagrange Points" << endl;
	cout << "===============" << endl;
	cout << endl;

	for (int i=0; i< num_bodies; i++)
		{
		body_array[i].update(evolution_time);
		body_array[i].find_lagrange();

		cout << "with " << body_array[i].get_name() << endl;		

		for (int j=1; j<4; j++)
			{
			cout << "L" << j <<":" << endl;
		
			cout << "x [" << converter.get_m_ft() << "]:  " << converter.convert_length_output(body_array[i].get_Lagrange_pos(j, 0))  << endl;
			cout << "y [" << converter.get_m_ft() << "]:  " << converter.convert_length_output(body_array[i].get_Lagrange_pos(j, 1))  << endl;
			cout << "z [" << converter.get_m_ft() << "]:  " << converter.convert_length_output(body_array[i].get_Lagrange_pos(j, 2))  << endl;

			cout << "vx [" << converter.get_m_ft_s() << "]: " << converter.convert_length_output(body_array[i].get_Lagrange_vel(j, 0))  << endl;
			cout << "vy [" << converter.get_m_ft_s() << "]: " << converter.convert_length_output(body_array[i].get_Lagrange_vel(j, 1))  << endl;
			cout << "vz [" << converter.get_m_ft_s() << "]: " << converter.convert_length_output(body_array[i].get_Lagrange_vel(j, 2))  << endl;


			}
		}


	}

}



void StateVector::list_target_sv()
{

cout << setprecision(8) ;
cout << endl;
cout << "Current target state vector" << endl;
cout << "===========================" << endl;

cout << "x [" << converter.get_m_ft() << "]:  " << converter.convert_length_output(target.get_x())  << endl;
cout << "y [" << converter.get_m_ft() << "]:  " << converter.convert_length_output(target.get_y())  << endl;
cout << "z [" << converter.get_m_ft() << "]:  " << converter.convert_length_output(target.get_z())  << endl;

cout << "vx [" << converter.get_m_ft_s() << "]: " << converter.convert_length_output(target.get_vx())  << endl;
cout << "vy [" << converter.get_m_ft_s() << "]: " << converter.convert_length_output(target.get_vy())  << endl;
cout << "vz [" << converter.get_m_ft_s() << "]: " << converter.convert_length_output(target.get_vz())  << endl;

cout << "T:  " << evolution_time << endl;
cout << endl;


}

void StateVector::compute_earth_rotation()
{
earth_rotation_angle -= earth_rotation_speed * dt;

}


void StateVector::compute_gravity()
{
long double r, g;
long double x2, y2, z2;


x2 = pow(x,2.0);
y2 = pow(y,2.0);
z2 = pow(z,2.0);

r = sqrt(x2 +y2 + z2);
g = -GM /pow(r,2.0);

gx = g * x/r;
gy = g * y/r;
gz = g * z/r;

if ((gravity_model == 1) || (gravity_model == 2))
	{
	// J2 harmonics

	gx += J2 * x/pow(r,7.0) * (6.0 * z2 - 1.5 * (x2 + y2));
	gy += J2 * y/pow(r,7.0) * (6.0 * z2 - 1.5 * (x2 + y2));
	gz += J2 * z/pow(r,7.0) * (3.0 * z2 - 4.5 * (x2 + y2));
	}

if (gravity_model == 2)
	{
	// J3 harmonics

	gx += J3 * x * z / pow(r,9.0) * (10.0 * z2 - 7.5 * (x2 + y2));
	gy += J3 * y * z / pow(r,9.0) * (10.0 * z2 - 7.5 * (x2 + y2));
	gz += J3  / pow(r,9.0) * (4.0 * z2 * (z2 - 3.0 * (x2 + y2)) + 1.5 * pow((x2 + y2),2.0) );

	}
	
if (num_bodies > 0)
	{

	barycenter.update(evolution_time);

	for (int i=0; i< num_bodies; i++)
		{
		body_array[i].update(evolution_time);

		// gravity 

		long double rel_x = x - body_array[i].get_pos_x();
		long double rel_y = y - body_array[i].get_pos_y();
		long double rel_z = z - body_array[i].get_pos_z();

		long double rel_x2 = pow(rel_x, 2.0);	
		long double rel_y2 = pow(rel_y, 2.0);	
		long double rel_z2 = pow(rel_z, 2.0);

		r = sqrt(rel_x2 + rel_y2 + rel_z2);

		//cout << "R to body: " << r << endl;
		
		g = -body_array[i].get_GM()/ pow(r,2.0);

		//cout << "body gravity acc: " << g << endl;		

		gx += g * rel_x/r;
		gy += g * rel_y/r;
		gz += g * rel_z/r;	

		// from this, we need to subtract the gravitational force on our coordinate reference, Earth

		rel_x = - body_array[i].get_pos_x();
		rel_y = - body_array[i].get_pos_y();
		rel_z = - body_array[i].get_pos_z();

		rel_x2 = pow(rel_x, 2.0);
		rel_y2 = pow(rel_y, 2.0);
		rel_z2 = pow(rel_z, 2.0);

		r = sqrt(rel_x2 + rel_y2 + rel_z2);

		g = -body_array[i].get_GM()/ pow(r,2.0);

		//cout << "earth gravity acc: " << g << endl;		

		//r = sqrt(rel_x2 + rel_y2 + rel_z2);

		gx -= g * rel_x/r;
		gy -= g * rel_y/r;
		gz -= g * rel_z/r;




		}
	}


}


long double StateVector::earth_radius ()
{

if (earth_model == 1)
	{
	long double lat = get_latitude_rad();
	long double cl = cos(lat);
	long double sl = sin(lat);

	return sqrt(   (pow(R_eq * R_eq * cl,2.0) + pow(R_polar * R_polar * sl, 2.0)) /  (pow(R_eq * cl, 2.0) + pow(R_polar * sl, 2.0)));

	}
else
	{
	return R_earth;
	}

}


double StateVector::get_earth_radius (double lat)
{
if (earth_model == 1)
	{
	double cl = cos(lat * deg_to_rad);
	double sl = sin(lat * deg_to_rad);

	return sqrt(   (pow(R_eq * R_eq * cl,2.0) + pow(R_polar * R_polar * sl, 2.0)) /  (pow(R_eq * cl, 2.0) + pow(R_polar * sl, 2.0)));

	}
else
	{
	return R_earth;
	}
}


long double StateVector::radius ()
{
return sqrt(x*x + y*y + z*z);
}

long double StateVector::vtot ()
{
return sqrt(vx*vx + vy*vy + vz*vz);
}





void StateVector::compute_elements()
{

long double vel [3], pos [3];
long double h[3], epsilon_vec[3], n_vec[3];

vel[0] = vx;
vel[1] = vy;
vel[2] = vz;

pos[0] = x;
pos[1] = y;
pos[2] = z;


long double r = norm(pos);

long double specific_energy = 0.5 * dot_product(vel,vel) - GM/r;
semi_major = - GM/(2.0 * specific_energy);

long double r_dot_v = dot_product(pos, vel);
h[0] = cross_product_x(pos, vel);
h[1] = cross_product_y(pos, vel);
h[2] = cross_product_z(pos, vel);

long double h_norm = norm(h);

inclination = acos(h[2]/h_norm);

epsilon_vec[0] = 1.0/ GM * cross_product_x(vel, h) - x/r;
epsilon_vec[1] = 1.0/ GM * cross_product_y(vel, h) - y/r;
epsilon_vec[2] = 1.0/ GM * cross_product_z(vel, h) - z/r;

eccentricity = norm(epsilon_vec);



n_vec[0] = -h[1];
n_vec[1] = h[0];
n_vec[2] = 0.0;

long double n_norm = norm(n_vec);
ascending_node = acos(n_vec[0]/n_norm);
if (n_vec[1] < 0.0) {ascending_node = 2.0 * pi - ascending_node;} 

n_vec[0]/= n_norm;
n_vec[1]/= n_norm;
n_vec[2]/= n_norm;

epsilon_vec[0] /= eccentricity;
epsilon_vec[1] /= eccentricity;
epsilon_vec[2] /= eccentricity;

argument_of_periapsis = acos (dot_product(n_vec, epsilon_vec));
if (epsilon_vec[2] < 0.0) {argument_of_periapsis = 2.0 * pi - argument_of_periapsis;}

pos[0] /= r;
pos[1] /= r;
pos[2] /= r;

true_anomaly = acos(dot_product(epsilon_vec,  pos));
if (r_dot_v < 0.0) {true_anomaly = 2.0 * pi - true_anomaly;}

}


long double StateVector::current_thetaT()
{
long double unit_pos_current[3], unit_pos_tig[3];
long double rnorm = radius();
long double current_thetaT;

unit_pos_current[0] = x/rnorm;
unit_pos_current[1] = y/rnorm;
unit_pos_current[2] = z/rnorm;


unit_pos_tig[0] = stored_xnorm_tig;
unit_pos_tig[1] = stored_ynorm_tig;
unit_pos_tig[2] = stored_znorm_tig;

current_thetaT = acos(dot_product(unit_pos_current, unit_pos_tig)) * rad_to_deg;



if (unit_pos_current[0] * stored_vxnorm_tig +unit_pos_current[1] * stored_vynorm_tig + unit_pos_current[2] * stored_vznorm_tig < 0.0)
	{
	current_thetaT = 360.0 - current_thetaT;
	} 

//cout << "Current Theta: " << current_thetaT << endl;

return current_thetaT;

}


void StateVector::compute_peg4()
{

long double unit_pos_current[3], unit_pos_tig[3];

long double rnorm = radius();

unit_pos_current[0] = x/rnorm;
unit_pos_current[1] = y/rnorm;
unit_pos_current[2] = z/rnorm;


unit_pos_tig[0] = stored_xnorm_tig;
unit_pos_tig[1] = stored_ynorm_tig;
unit_pos_tig[2] = stored_znorm_tig;


long double velocity[3];

velocity[0] = vx;
velocity[1] = vy;
velocity[2] = vz;



peg4_thetaT_act = acos(dot_product(unit_pos_current, unit_pos_tig)) * rad_to_deg;
if (unit_pos_current[0] * stored_vxnorm_tig +unit_pos_current[1] * stored_vynorm_tig + unit_pos_current[2] * stored_vznorm_tig < 0.0)
	{
	//cout << "Hemisphere neg " << endl;
	peg4_thetaT_act = 360.0 - peg4_thetaT_act;
	} 
else
	{
	//cout << "Hemisphere pos " << endl;
	}



peg4_H_act = rnorm - earth_radius();
peg4_vspeed_act = dot_product(unit_pos_current, velocity);

velocity[0] -= peg4_vspeed_act * unit_pos_current[0];
velocity[1] -= peg4_vspeed_act * unit_pos_current[1];
velocity[2] -= peg4_vspeed_act * unit_pos_current[2];

peg4_hspeed_act = sqrt(velocity[0]*velocity[0] + velocity[1] * velocity[1] + velocity[2] * velocity[2]);

if  (burn_array[current_burn_index].get_burn_type() == 2)
	{
	peg4_rei_act = landing_site.distance_from (get_latitude(), get_longitude());
	}

//cout << "theta T   : " << peg4_thetaT_act << endl;
//cout << "H         : " << peg4_H_act << endl;
//cout << "vspeed    : " << peg4_vspeed_act << endl;
//cout << "hspeed    : " << peg4_hspeed_act << endl;
//cout << "vspeed_tgt: " << peg4_c2 * v_horiz + peg4_c1 << endl << endl;


}


long double StateVector::course ()
{

long double rnorm = radius();
long double unit_x =  x/rnorm;
long double unit_y =  y/rnorm;
long double unit_z =  z/rnorm;

long double unit_east_x = -unit_y;
long double unit_east_y = unit_x;
long double unit_east_z = 0.0;

long double unit_east_norm = sqrt(unit_east_x * unit_east_x + unit_east_y * unit_east_y);
unit_east_x /= unit_east_norm;
unit_east_y /= unit_east_norm;

long double unit_north_x = unit_y * unit_east_z - unit_z * unit_east_y;
long double unit_north_y = unit_z * unit_east_x - unit_x * unit_east_z;
long double unit_north_z = unit_x * unit_east_y - unit_y * unit_east_x;

long double v_east = vx * unit_east_x + vy * unit_east_y + vz * unit_east_z;
long double v_north = vx * unit_north_x + vy * unit_north_y + vz * unit_north_z;

//long double hspeed = sqrt(v_east * v_east + v_north* v_north);
//v_east/=hspeed;
//v_north/=hspeed;

long double course = atan2(v_east, v_north) * rad_to_deg;

return course;
}


void StateVector::compute_multibody_gravity ()
{

barycenter.update(evolution_time);

	
// first transform to barycenter relative coordinates

bc_x = x + barycenter.get_x();
bc_y = y + barycenter.get_y();
bc_z = z + barycenter.get_z();

bc_vx = vx + barycenter.get_vx();
bc_vy = vy + barycenter.get_vy();
bc_vz = vz + barycenter.get_vz();


// Earth gx, gy and gz have been computed, the relative vector is not different

bc_vx += gx * dt;
bc_vy += gy * dt;
bc_vz += gz * dt;

// now get the other bodies

for (int i=0; i< num_bodies; i++)
	{
	long double rel_x = body_array[i].get_pos_x() - x;
	long double rel_y = body_array[i].get_pos_y() - y;
	long double rel_z = body_array[i].get_pos_z() - z;

	long double rel_x2 = pow(rel_x, 2.0);
	long double rel_y2 = pow(rel_y, 2.0);
	long double rel_z2 = pow(rel_z, 2.0);

	long double r_body = sqrt(rel_x2 +rel_y2 + rel_z2);
	long double g_body = -body_array[i].get_GM() /pow(r_body,2.0);

	long double gx_body = g_body * rel_x/r_body;
	long double gy_body = g_body * rel_y/r_body;
	long double gz_body = g_body * rel_z/r_body;

	bc_vx += gx_body * dt;
	bc_vy += gy_body * dt;
	bc_vz += gz_body * dt;

	}

// and back to Earth-relative coordinates

	vx = bc_vx - barycenter.get_vx();
	vy = bc_vy - barycenter.get_vy();
	vz = bc_vz - barycenter.get_vz();



}


void StateVector::do_timestep ()
{


compute_gravity();
compute_earth_rotation();


vx += gx * dt;
vy += gy * dt;
vz += gz * dt;
	


if (num_burns > 0)
	{
	bool condition;

	// take care of overlapping burns during fit phase
	if (hypothetical_flag == false)
		{
		if (burn_array[next_burn_index].is_active(evolution_time) == true) 
			{condition = true;}
		else
			{condition = false;}
		}
	else
		{
		if ((burn_array[next_burn_index].is_active(evolution_time) == true) && (next_burn_index == current_burn_index))
			{condition = true;}
		else
			{condition = false;}
		}
	

	if (condition)
		{

		int burn_type = burn_array[next_burn_index].get_burn_type();
	
		if ((burn_type == 1) || (burn_type == 2)) // PEG-4 or optimized PEG-4 fit init
			{
			if (burn_array[next_burn_index].is_initialized() == false)
				{
				
				compute_elements();
				compute_peg4();
				long double periapsis = semi_major * (1.0 - eccentricity) -earth_radius();
				long double apoapsis = semi_major * (1.0 + eccentricity) - earth_radius();
				long double vspeed = peg4_vspeed_act;
				long double orbital_speed_diff = sqrt(GM/radius()) - peg4_hspeed_act;

				burn_array[next_burn_index].peg4_guess_initial(radius() - earth_radius(), periapsis, apoapsis, vspeed, orbital_speed_diff, static_cast<int>(0.5 * fit_max_iterations));
				
				
				if (burn_array[next_burn_index].get_strategy() == 2)
					{
					fit_radius_last = 1e20; // need to be sure to not trigger apsis finder while burn is in progress
					}


				stored_x = x;
				stored_y = y;
				stored_z = z;


				if (burn_array[next_burn_index].get_ops() == 3)
					{
					long double v_norm = sqrt(vx*vx + vy*vy + vz*vz);
					long double r_norm = sqrt(x*x + y*y + z*z);

					stored_xnorm_tig = x/r_norm;
					stored_ynorm_tig = y/r_norm;
					stored_znorm_tig = z/r_norm;
					stored_vxnorm_tig = vx/v_norm;
					stored_vynorm_tig = vy/v_norm;
					stored_vznorm_tig = vz/v_norm;
					}
				else
					{
					stored_xnorm_tig = launch_site.get_unit_x();
					stored_ynorm_tig = launch_site.get_unit_y();
					stored_znorm_tig = launch_site.get_unit_z();
					stored_vxnorm_tig = launch_site.get_unit_vx();
					stored_vynorm_tig = launch_site.get_unit_vy();
					stored_vznorm_tig = launch_site.get_unit_vz();

					//cout << "Initial angular offset: " << current_thetaT() << endl;
					}
				stored_vx = vx;
				stored_vy = vy;
				stored_vz = vz;	
				
				target.store();

				stored_evolution_time = evolution_time;			
				stored_earth_rotation_angle = earth_rotation_angle;		


	
				burn_array[next_burn_index].set_spacecraft (craft.mass - craft.propellant_mass, craft.propellant_mass, craft.isp, craft.thrust);

				hypothetical_flag = true;
				if (fit_log_flag)
					{cout << "============ HYPOTHETICAL MODE ON ====================" << endl;}
				current_burn_index = next_burn_index;
				}

			if (burn_array[next_burn_index].tig_changed()) 
				{
				if (burn_array[next_burn_index].get_ops() == 3)
					{
					long double v_norm = sqrt(vx*vx + vy*vy + vz*vz);
					long double r_norm = sqrt(x*x + y*y + z*z);
					stored_xnorm_tig = x/r_norm;
					stored_ynorm_tig = y/r_norm;
					stored_znorm_tig = z/r_norm;
					stored_vxnorm_tig = vx/v_norm;
					stored_vynorm_tig = vy/v_norm;
					stored_vznorm_tig = vz/v_norm;

					}

				burn_array[next_burn_index].new_tig_acknowledge();
				
				if (fit_log_flag)
					{
					cout << "Fit required TIG setting, time is now: " << evolution_time << endl;
					}
				}

			}
		else if ((burn_type == 0) || (burn_type == 3)) // PEG-7 or low thrust
			{
			if (burn_array[next_burn_index].is_initialized() == false)
				{
				list_burn(next_burn_index);
				if (burn_type == 3)
					{
					low_thrust_counter = 0;
					}
				}


			}

		else if (burn_type == 5) // Lambert
			{

			
			if ((burn_array[next_burn_index].is_initialized() == false) && (burn_array[next_burn_index].is_dry_run() == false))
				
				{
				stored_x = x;
				stored_y = y;
				stored_z = z;

				stored_vx = vx;
				stored_vy = vy;
				stored_vz = vz;

				stored_evolution_time = evolution_time;			
				stored_earth_rotation_angle = earth_rotation_angle;		

				target.store();

	
				burn_array[next_burn_index].set_spacecraft (craft.mass - craft.propellant_mass, craft.propellant_mass, craft.isp, craft.thrust);


				burn_array[next_burn_index].set_dry_run(true);
				hypothetical_flag = true;

				if (fit_log_flag)
					{cout << "============ HYPOTHETICAL MODE ON ====================" << endl;}
				
				current_burn_index = next_burn_index;

				cout << "Lambert dry run start." << endl;
				//cout << "Current: " << current_burn_index << " Next: " << next_burn_index << endl;

				}

			}

		
		if ((burn_model == 2) && (burn_array[next_burn_index].is_initialized() == false))
			{

			//cout << "Initializing Spacecraft for burn!" << endl;
			//cout << "Current thrust [" << converter.get_N_lb() << "]: " << converter.convert_force_output(craft.thrust) << endl;
			burn_array[next_burn_index].set_spacecraft (craft.mass - craft.propellant_mass, craft.propellant_mass, craft.isp, craft.thrust);
			}

		// unit vectors

		long double v_tot = sqrt(vx*vx + vy*vy + vz*vz);
		long double r_tot = sqrt(x*x + y*y + z*z);

		long double vnorm_x = vx/v_tot;
		long double vnorm_y = vy/v_tot;
		long double vnorm_z = vz/v_tot;

		long double rnorm_x = x/r_tot;
		long double rnorm_y = y/r_tot;
		long double rnorm_z = z/r_tot;


		if (burn_type == 3)
			{

			burn_array[next_burn_index].start(vnorm_x, vnorm_y, vnorm_z);
			}
		else
			{
			burn_array[next_burn_index].start();
			}
		burn_array[next_burn_index].do_timestep(dt);



		// prograde


		vx += vnorm_x * burn_array[next_burn_index].get_acc_prograde() * dt;		
		vy += vnorm_y * burn_array[next_burn_index].get_acc_prograde() * dt;
		vz += vnorm_z * burn_array[next_burn_index].get_acc_prograde() * dt;

		//radial
		
		long double rnorm [3], vnorm [3], radnorm[3];
		
		rnorm[0] = rnorm_x;
		rnorm[1] = rnorm_y;
		rnorm[2] = rnorm_z;

		vnorm[0] = vnorm_x;
		vnorm[1] = vnorm_y;
		vnorm[2] = vnorm_z;


		radnorm[0] = orthogonalize_x(vnorm,rnorm);
		radnorm[1] = orthogonalize_y(vnorm,rnorm);
		radnorm[2] = orthogonalize_z(vnorm,rnorm);

		long double rad_norm = norm(radnorm);

		radnorm[0]/= rad_norm;
		radnorm[1]/= rad_norm;
		radnorm[2]/= rad_norm;

		vx += radnorm[0] * burn_array[next_burn_index].get_acc_radial() * dt;		
		vy += radnorm[1] * burn_array[next_burn_index].get_acc_radial() * dt;
		vz += radnorm[2] * burn_array[next_burn_index].get_acc_radial() * dt;


		//normal


		long double nx = cross_product_x(vnorm, radnorm);
		long double ny = cross_product_y(vnorm, radnorm);
		long double nz = cross_product_z(vnorm, radnorm);

		vx += nx * burn_array[next_burn_index].get_acc_normal() * dt;		
		vy += ny * burn_array[next_burn_index].get_acc_normal() * dt;
		vz += nz * burn_array[next_burn_index].get_acc_normal() * dt;

		// periodically force ignition ref to be in plane
		if (burn_type == 3)
			{
			if (low_thrust_counter == 100)
				{
				burn_array[next_burn_index].update_low_thrust(nx, ny, nz);
				//cout << "Time is now: " << evolution_time << endl;
				low_thrust_counter = 0;
				}
			else
				{
				low_thrust_counter += 1;
				}
			}



		}
	else if (burn_array[next_burn_index].is_finished())
		{

		if ((craft.is_defined) && (hypothetical_flag == false))
			{
			craft.propellant_mass = burn_array[next_burn_index].get_propellant();
			craft.mass = craft.dry_mass + craft.propellant_mass;
			cout << "Remaining propellant: " << converter.convert_mass_output(craft.propellant_mass) << " " << converter.get_kg_lb() << endl;
			}
		num_burns -= 1;
		next_burn_index += 1;


		//cout << "Burn index is now " << current_burn_index << endl;
		//cout << "Now moving to burn index " << next_burn_index << " with burn " << burn_array[next_burn_index].get_name() << endl;
					

	
		}

	}


x += vx * dt;
y += vy * dt;
z += vz * dt;

evolution_time += dt;
	
}

void StateVector::do_plot2d()
{

double par_x, par_y;



if (plot2d_counter == plot2d_resolution)
	{
	compute_elements();

	switch (plot2d_x_option) {
		case 0: par_x = evolution_time; break;
		case 1: par_x = x; break;
		case 2: par_x = y; break;
		case 3: par_x = z; break;
		case 4: par_x = get_altitude(); break;
		case 5: par_x = semi_major; break;
		case 6: par_x = eccentricity; break;
		case 7: par_x = inclination * rad_to_deg; break;
		case 8: par_x = ascending_node * rad_to_deg; break;
		case 9: par_x = argument_of_periapsis * rad_to_deg; break;
		case 10: par_x = true_anomaly * rad_to_deg; break;
		case 11: par_x = get_latitude(); break;
		case 12: par_x = get_longitude(); break;
		case 13: par_x = radius(); break;
		case 14: par_x = vx; break;
		case 15: par_x = vy; break;
		case 16: par_x = vz; break;
		case 17: par_x = vtot(); break;
		case 18: par_x = body_array[0].get_relpos_x(x); break;
		case 19: par_x = body_array[0].get_relpos_y(y); break;
		case 20: par_x = body_array[0].get_relpos_z(z); break;
		case 21: par_x = body_array[0].get_relv_x(vx); break;
		case 22: par_x = body_array[0].get_relv_y(vy); break;
		case 23: par_x = body_array[0].get_relv_z(vz); break;
		case 27: par_x = sqrt(pow(body_array[0].get_relpos_x(x),2.0) + pow(body_array[0].get_relpos_y(y),2.0)+ pow(body_array[0].get_relpos_z(z),2.0)); break;
		case 28: par_x = course(); break;
		case 40: par_x = target.get_x(); break;
		case 41: par_x = target.get_y(); break;
		case 42: par_x = target.get_z(); break;
		case 43: par_x = target.get_vx(); break;
		case 44: par_x = target.get_vy(); break;
		case 45: par_x = target.get_vz(); break;
		case 46: par_x = target.get_lat(); break;
		case 47: par_x = target.get_lon(); break;
		case 48: par_x = target.get_alt(); break;
		case 49: par_x = target.get_prox_x(x,y,z,vx,vy,vz); break;
		case 50: par_x = target.get_prox_y(x,y,z,vx,vy,vz); break;
		case 51: par_x = target.get_prox_z(x,y,z,vx,vy,vz); break;
		case 52: par_x = target.get_prox_vx(x,y,z,vx,vy,vz); break;
		case 53: par_x = target.get_prox_vy(x,y,z,vx,vy,vz); break;
		case 54: par_x = target.get_prox_vz(x,y,z,vx,vy,vz); break;
		default: cout << "Plot option not impemented!" << endl; exit(0); break;

	}

	switch (plot2d_y_option) {
		case 0: par_y = evolution_time; break;
		case 1: par_y = x; break;
		case 2: par_y = y; break;
		case 3: par_y = z; break;
		case 4: par_y = get_altitude(); break;
		case 5: par_y = semi_major; break;
		case 6: par_y = eccentricity; break;
		case 7: par_y = inclination * rad_to_deg; break;
		case 8: par_y = ascending_node * rad_to_deg; break;
		case 9: par_y = argument_of_periapsis * rad_to_deg; break;
		case 10: par_y = true_anomaly * rad_to_deg; break;
		case 11: par_y = get_latitude(); break;
		case 12: par_y = get_longitude(); break;
		case 13: par_y = radius(); break;
		case 14: par_y = vx; break;
		case 15: par_y = vy; break;
		case 16: par_y = vz; break;
		case 17: par_y = vtot(); break;
		case 18: par_y = body_array[0].get_relpos_x(x); break;
		case 19: par_y = body_array[0].get_relpos_y(y); break;
		case 20: par_y = body_array[0].get_relpos_z(z); break;
		case 21: par_y = body_array[0].get_relv_x(vx); break;
		case 22: par_y = body_array[0].get_relv_y(vy); break;
		case 23: par_y = body_array[0].get_relv_z(vz); break;
		case 27: par_y = sqrt(pow(body_array[0].get_relpos_x(x),2.0) + pow(body_array[0].get_relpos_y(y),2.0)+ pow(body_array[0].get_relpos_z(z),2.0)); break;
		case 28: par_y = course(); break;
		case 40: par_y = target.get_x(); break;
		case 41: par_y = target.get_y(); break;
		case 42: par_y = target.get_z(); break;
		case 43: par_y = target.get_vx(); break;
		case 44: par_y = target.get_vy(); break;
		case 45: par_y = target.get_vz(); break;
		case 46: par_y = target.get_lat(); break;
		case 47: par_y = target.get_lon(); break;
		case 48: par_y = target.get_alt(); break;
		case 49: par_y = target.get_prox_x(x,y,z,vx,vy,vz); break;
		case 50: par_y = target.get_prox_y(x,y,z,vx,vy,vz); break;
		case 51: par_y = target.get_prox_z(x,y,z,vx,vy,vz); break;
		case 52: par_y = target.get_prox_vx(x,y,z,vx,vy,vz); break;
		case 53: par_y = target.get_prox_vy(x,y,z,vx,vy,vz); break;
		case 54: par_y = target.get_prox_vz(x,y,z,vx,vy,vz); break;
		default: cout << "Plot option not impemented!" << endl; exit(0); break;

	}


	plot2d_x[plot2d_index] = par_x;
	plot2d_y[plot2d_index] = par_y;

	//cout << plot2d_index << " " << inclination << endl;

	plot2d_counter = 0;
	plot2d_index += 1;
	
	}
else
	{
	plot2d_counter += 1;
	}


}


void StateVector::do_plot3d()
{

double par_x, par_y, par_z;



if (plot3d_counter == plot3d_resolution)
	{
	compute_elements();

	switch (plot3d_x_option) {
		case 0: par_x = evolution_time; break;
		case 1: par_x = x; break;
		case 2: par_x = y; break;
		case 3: par_x = z; break;
		case 4: par_x = get_altitude(); break;
		case 5: par_x = semi_major; break;
		case 6: par_x = eccentricity; break;
		case 7: par_x = inclination * rad_to_deg; break;
		case 8: par_x = ascending_node * rad_to_deg; break;
		case 9: par_x = argument_of_periapsis * rad_to_deg; break;
		case 10: par_x = true_anomaly * rad_to_deg; break;
		case 11: par_x = get_latitude(); break;
		case 12: par_x = get_longitude(); break;
		case 13: par_x = radius(); break;
		case 14: par_x = vx; break;
		case 15: par_x = vy; break;
		case 16: par_x = vz; break;
		case 17: par_x = vtot(); break;
		case 18: par_x = body_array[0].get_relpos_x(x); break;
		case 19: par_x = body_array[0].get_relpos_y(y); break;
		case 20: par_x = body_array[0].get_relpos_z(z); break;
		case 21: par_x = body_array[0].get_relv_x(vx); break;
		case 22: par_x = body_array[0].get_relv_y(vy); break;
		case 23: par_x = body_array[0].get_relv_z(vz); break;
		case 24: par_x = body_array[0].get_pos_x(); break;
		case 25: par_x = body_array[0].get_pos_y(); break;
		case 26: par_x = body_array[0].get_pos_z(); break;

		case 40: par_x = target.get_x(); break;
		case 41: par_x = target.get_y(); break;
		case 42: par_x = target.get_z(); break;
		case 43: par_x = target.get_vx(); break;
		case 44: par_x = target.get_vy(); break;
		case 45: par_x = target.get_vz(); break;
		case 46: par_x = target.get_lat(); break;
		case 47: par_x = target.get_lon(); break;
		case 48: par_x = target.get_alt(); break;
		case 49: par_x = target.get_prox_x(x,y,z,vx,vy,vz); break;
		case 50: par_x = target.get_prox_y(x,y,z,vx,vy,vz); break;
		case 51: par_x = target.get_prox_z(x,y,z,vx,vy,vz); break;
		case 52: par_x = target.get_prox_vx(x,y,z,vx,vy,vz); break;
		case 53: par_x = target.get_prox_vy(x,y,z,vx,vy,vz); break;
		case 54: par_x = target.get_prox_vz(x,y,z,vx,vy,vz); break;
		default: cout << "Plot option not impemented!" << endl; exit(0); break;

	}

	switch (plot3d_y_option) {
		case 0: par_y = evolution_time; break;
		case 1: par_y = x; break;
		case 2: par_y = y; break;
		case 3: par_y = z; break;
		case 4: par_y = get_altitude(); break;
		case 5: par_y = semi_major; break;
		case 6: par_y = eccentricity; break;
		case 7: par_y = inclination * rad_to_deg; break;
		case 8: par_y = ascending_node * rad_to_deg; break;
		case 9: par_y = argument_of_periapsis * rad_to_deg; break;
		case 10: par_y = true_anomaly * rad_to_deg; break;
		case 11: par_y = get_latitude(); break;
		case 12: par_y = get_longitude(); break;
		case 13: par_y = radius(); break;
		case 14: par_y = vx; break;
		case 15: par_y = vy; break;
		case 16: par_y = vz; break;
		case 17: par_y = vtot(); break;
		case 18: par_y = body_array[0].get_relpos_x(x); break;
		case 19: par_y = body_array[0].get_relpos_y(y); break;
		case 20: par_y = body_array[0].get_relpos_z(z); break;
		case 21: par_y = body_array[0].get_relv_x(vx); break;
		case 22: par_y = body_array[0].get_relv_y(vy); break;
		case 23: par_y = body_array[0].get_relv_z(vz); break;
		case 24: par_y = body_array[0].get_pos_x(); break;
		case 25: par_y = body_array[0].get_pos_y(); break;
		case 26: par_y = body_array[0].get_pos_z(); break;

		case 40: par_y = target.get_x(); break;
		case 41: par_y = target.get_y(); break;
		case 42: par_y = target.get_z(); break;
		case 43: par_y = target.get_vx(); break;
		case 44: par_y = target.get_vy(); break;
		case 45: par_y = target.get_vz(); break;
		case 46: par_y = target.get_lat(); break;
		case 47: par_y = target.get_lon(); break;
		case 48: par_y = target.get_alt(); break;
		case 49: par_y = target.get_prox_x(x,y,z,vx,vy,vz); break;
		case 50: par_y = target.get_prox_y(x,y,z,vx,vy,vz); break;
		case 51: par_y = target.get_prox_z(x,y,z,vx,vy,vz); break;
		case 52: par_y = target.get_prox_vx(x,y,z,vx,vy,vz); break;
		case 53: par_y = target.get_prox_vy(x,y,z,vx,vy,vz); break;
		case 54: par_y = target.get_prox_vz(x,y,z,vx,vy,vz); break;
		default: cout << "Plot option not impemented!" << endl; exit(0); break;

	}

	switch (plot3d_z_option) {
		case 0: par_z = evolution_time; break;
		case 1: par_z = x; break;
		case 2: par_z = y; break;
		case 3: par_z = z; break;
		case 4: par_z = get_altitude(); break;
		case 5: par_z = semi_major; break;
		case 6: par_z = eccentricity; break;
		case 7: par_z = inclination * rad_to_deg; break;
		case 8: par_z = ascending_node * rad_to_deg; break;
		case 9: par_z = argument_of_periapsis * rad_to_deg; break;
		case 10: par_z = true_anomaly * rad_to_deg; break;
		case 11: par_z = get_latitude(); break;
		case 12: par_z = get_longitude(); break;
		case 13: par_z = radius(); break;
		case 14: par_z = vx; break;
		case 15: par_z = vy; break;
		case 16: par_z = vz; break;
		case 17: par_z = vtot(); break;
		case 18: par_z = body_array[0].get_relpos_x(x); break;
		case 19: par_z = body_array[0].get_relpos_y(y); break;
		case 20: par_z = body_array[0].get_relpos_z(z); break;
		case 21: par_z = body_array[0].get_relv_x(vx); break;
		case 22: par_z = body_array[0].get_relv_y(vy); break;
		case 23: par_z = body_array[0].get_relv_z(vz); break;
		case 24: par_z = body_array[0].get_pos_x(); break;
		case 25: par_z = body_array[0].get_pos_y(); break;
		case 26: par_z = body_array[0].get_pos_z(); break;

		case 40: par_z = target.get_x(); break;
		case 41: par_z = target.get_y(); break;
		case 42: par_z = target.get_z(); break;
		case 43: par_z = target.get_vx(); break;
		case 44: par_z = target.get_vy(); break;
		case 45: par_z = target.get_vz(); break;
		case 46: par_z = target.get_lat(); break;
		case 47: par_z = target.get_lon(); break;
		case 48: par_z = target.get_alt(); break;
		case 49: par_z = target.get_prox_x(x,y,z,vx,vy,vz); break;
		case 50: par_z = target.get_prox_y(x,y,z,vx,vy,vz); break;
		case 51: par_z = target.get_prox_z(x,y,z,vx,vy,vz); break;
		case 52: par_z = target.get_prox_vx(x,y,z,vx,vy,vz); break;
		case 53: par_z = target.get_prox_vy(x,y,z,vx,vy,vz); break;
		case 54: par_z = target.get_prox_vz(x,y,z,vx,vy,vz); break;
		default: cout << "Plot option not impemented!" << endl; exit(0); break;

	}


	plot3d_x[plot3d_index] = par_x;
	plot3d_y[plot3d_index] = par_y;
	plot3d_z[plot3d_index] = par_z;


	plot3d_counter = 0;
	plot3d_index += 1;
	
	}
else
	{
	plot3d_counter += 1;
	}


}



void StateVector::evolve_to (long double time)
{

plot2d_index = 0;
plot3d_index = 0;

while (check_condition() && (evolution_time < time) )
	{
	if (plot2d_flag && (hypothetical_flag == false)) {do_plot2d();}
	if (plot3d_flag && (hypothetical_flag == false)) {do_plot3d();}
	do_timestep();
	if (target_defined) {target.do_timestep();}
	}

cout << "Evolution time: " << evolution_time << endl;



}
