
// site.cxx -- routines to compute ground site spherical trigonometry
//
// Written by Thorsten Renk, started 2017
//
// Copyright (C) 2017  Thorsten Renk - thorsten@science-and-fiction.org 
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301

#include <stdlib.h>
#include <iostream>
#include <math.h>

#include "constants.hxx"
#include "site.hxx"


using namespace std;

void Site::init (long double set_lat, long double set_lon, string set_name)
{
lat = set_lat;
lon = set_lon;
name = set_name;

cout << "Adding site "<< name << " at lat " << lat << " and lon " << lon << endl;

defined_flag = true;

}


void Site::init_by_name(string set_name)
{

name = set_name;

if (name == "KSC")
	{
	lat = 28.615;
	lon = -80.695;
	}
else if (name == "VBG")
	{
	lat = 34.722;
	lon = -120.567;
	}
else if (name == "EDW")
	{
	lat = 34.096;
	lon = -117.884;
	}

cout << "Adding site "<< name << " at lat " << lat << " and lon " << lon << " from list of ground sites." << endl;

defined_flag = true;


}

long double Site::distance_from(long double lat2, long double lon2)
{

long double phi1 = lat * deg_to_rad;
long double phi2 = lat2 * deg_to_rad;
long double sdp = sin(0.5 * (phi2 - phi1));
long double sdl = sin(0.5 * (lon2 - lon) * deg_to_rad);

long double a = sdp * sdp + cos(phi1) * cos(phi2) * sdl * sdl; 
long double c = 2.0 * atan2(sqrt(a), sqrt(1.0-a));


return R_earth * c;
}


long double Site::bearing_from(long double lat2, long double lon2)
{
long double delta_lambda = (lon2 - lon) * deg_to_rad;
long double phi1 = lat * deg_to_rad;
long double phi2 = lat2 * deg_to_rad;

long double y = sin(delta_lambda) * cos(phi2);
long double x = cos(phi1) * sin(phi2) - sin(phi1) * cos(phi2) * cos(delta_lambda);

return atan2(y,x) * rad_to_deg;

}

void Site::apply_course_distance(long double course, long double d)
{
long double phi1 = lat * deg_to_rad;
long double lambda1  = lon * deg_to_rad;
long double brg = course * deg_to_rad;

long double phi2 = asin( sin(phi1) * cos(d/R_earth ) + cos(phi1) * sin(d/R_earth) * cos(brg) );
long double lambda2 = lambda1 + atan2(sin(brg) * sin(d/R_earth) * cos(phi1), cos(d/R_earth) - sin(phi1) * sin(phi2) );


lat = phi2 * rad_to_deg;
lon = lambda2 * rad_to_deg;

if (lon > 180.0) {lon = lon - 360.0;}
if (lon < -180.0) {lon = lon + 360.0;}
}


void Site::set_launch_site (long double met, long double slat, long double slon)
{
long double lon_at_launch = lon - met *  earth_rotation_speed;

cout << "Site " << name << " is defined as launch site" << endl;

unit_x =  cos(lat * deg_to_rad) * cos(lon_at_launch * deg_to_rad);
unit_y =  cos(lat * deg_to_rad) * sin(lon_at_launch * deg_to_rad);
unit_z =  sin (lat * deg_to_rad);

long double unit_east_x = -unit_y;
long double unit_east_y = unit_x;
long double unit_east_z = 0.0;

long double unit_east_norm = sqrt(unit_east_x * unit_east_x + unit_east_y * unit_east_y);
unit_east_x /= unit_east_norm;
unit_east_y /= unit_east_norm;

long double unit_north_x = unit_y * unit_east_z - unit_z * unit_east_y;
long double unit_north_y = unit_z * unit_east_x - unit_x * unit_east_z;
long double unit_north_z = unit_x * unit_east_y - unit_y * unit_east_x;

long double course = bearing_from(slat, slon);
if (course > 360.0) {course = course - 360.0;}



long double v_east =  sin(course * deg_to_rad);
long double v_north = cos(course * deg_to_rad);


unit_vx =  unit_east_x * v_east + unit_north_x * v_north;
unit_vy =  unit_east_y * v_east + unit_north_y * v_north;
unit_vz =  unit_east_z * v_east + unit_north_z * v_north;

//cout << "Course: " << course << endl;
//cout << "Unit pos: " << unit_x << " " << unit_y << " " << unit_z << endl;
//cout << "Unit east: " << unit_east_x << " " << unit_east_y << " " << unit_east_z << endl;
//cout << "Unit north: " << unit_north_x << " " << unit_north_y << " " << unit_north_z << endl;
//cout << "Unit vel: " << v_east << " " << v_north << " " << endl;
//cout << "Unit vel: " << unit_vx << " " << unit_vy << " " << unit_vz << endl;
}


bool Site::is_defined()
{
return defined_flag;
}

bool Site::is_known(std::string name)
{
bool flag = false;

if (name == "KSC") {flag = true;}
else if (name == "VBG") {flag = true;}
else if (name == "EDW") {flag = true;}

return flag;
}


long double Site::get_lat()
{
return lat;
}

long double Site::get_lon()
{
return lon;
}

long double Site::get_unit_x()
{
return unit_x;
}

long double Site::get_unit_y()
{
return unit_y;
}

long double Site::get_unit_z()
{
return unit_z;
}

long double Site::get_unit_vx()
{
return unit_vx;
}

long double Site::get_unit_vy()
{
return unit_vy;
}

long double Site::get_unit_vz()
{
return unit_vz;
}


