// rendezvous_target.cxx -- routines to forward a rendezvous target state vector
// and obtain its properties
//
// Written by Thorsten Renk, started 2018
//
// Copyright (C) 2018  Thorsten Renk - thorsten@science-and-fiction.org 
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301

#include <iostream>
#include <iomanip>
#include <stdlib.h>
#include <math.h>
#include <cmath>

#include "constants.hxx"
#include "rendezvous_target.hxx"

using namespace std;
using namespace MathHelpers;


void RendezvousTarget::init( Options options)
{
dt = options.timestep;

x = -1.0;
y = -1.0;
z = -1.0;

vx = -1.0;
vy = -1.0;
vz = -1.0;

earth_rotation_angle = 0.0;


if (options.tgt_gravity_model == -1)
	{
	gravity_model = options.gravity_model;	
	}
else
	{
	gravity_model = options.tgt_gravity_model;	
	}

cout << "Gravity model is now: " << gravity_model << endl;

earth_model = options.earth_model;


}

void RendezvousTarget::set_pos(long double x_in, long double y_in, long double z_in)
{
x = x_in;
y = y_in;
z = z_in;


//cout << "vx is now " << vx << " vy: " << vy << " vz: " << vz << endl;


if ((vx != -1.0) && (vy!= -1.0) && (vz!= -1.0)) {compute_elements();}
}

void RendezvousTarget::set_vel(long double vx_in, long double vy_in, long double vz_in)
{
vx = vx_in;
vy = vy_in;
vz = vz_in;

//cout << "x is now " << x << " y: " << y << " z: " << z << endl;

if ((x != -1.0) && (y!= -1.0) && (z!= -1.0)) {compute_elements();}

}


long double RendezvousTarget::earth_radius ()
{

if (earth_model == 1)
	{
	long double lat = get_latitude_rad();
	long double cl = cos(lat);
	long double sl = sin(lat);

	return sqrt(   (pow(R_eq * R_eq * cl,2.0) + pow(R_polar * R_polar * sl, 2.0)) /  (pow(R_eq * cl, 2.0) + pow(R_polar * sl, 2.0)));

	}
else
	{
	return R_earth;
	}

}


void RendezvousTarget::compute_gravity()
{
long double r, g;
long double x2, y2, z2;


x2 = pow(x,2.0);
y2 = pow(y,2.0);
z2 = pow(z,2.0);

r = sqrt(x2 +y2 + z2);
g = -GM /pow(r,2.0);

gx = g * x/r;
gy = g * y/r;
gz = g * z/r;


if ((gravity_model == 1) || (gravity_model == 2))
	{
	// J2 harmonics

	gx += J2 * x/pow(r,7.0) * (6.0 * z2 - 1.5 * (x2 + y2));
	gy += J2 * y/pow(r,7.0) * (6.0 * z2 - 1.5 * (x2 + y2));
	gz += J2 * z/pow(r,7.0) * (3.0 * z2 - 4.5 * (x2 + y2));
	}

if (gravity_model == 2)
	{
	// J3 harmonics

	gx += J3 * x * z / pow(r,9.0) * (10.0 * z2 - 7.5 * (x2 + y2));
	gy += J3 * y * z / pow(r,9.0) * (10.0 * z2 - 7.5 * (x2 + y2));
	gz += J3  / pow(r,9.0) * (4.0 * z2 * (z2 - 3.0 * (x2 + y2)) + 1.5 * pow((x2 + y2),2.0) );

	}
	
if (num_bodies > 0)
	{


	for (int i=0; i< num_bodies; i++)
		{
		// gravity 

		long double rel_x = x - body_array[i].get_pos_x();
		long double rel_y = y - body_array[i].get_pos_y();
		long double rel_z = z - body_array[i].get_pos_z();

		long double rel_x2 = pow(rel_x, 2.0);	
		long double rel_y2 = pow(rel_y, 2.0);	
		long double rel_z2 = pow(rel_z, 2.0);

		r = sqrt(rel_x2 + rel_y2 + rel_z2);

		//cout << "R to body: " << r << endl;
		
		g = -body_array[i].get_GM()/ pow(r,2.0);

		//cout << "body gravity acc: " << g << endl;		

		gx += g * rel_x/r;
		gy += g * rel_y/r;
		gz += g * rel_z/r;	

		// from this, we need to subtract the gravitational force on our coordinate reference, Earth

		rel_x = - body_array[i].get_pos_x();
		rel_y = - body_array[i].get_pos_y();
		rel_z = - body_array[i].get_pos_z();

		rel_x2 = pow(rel_x, 2.0);
		rel_y2 = pow(rel_y, 2.0);
		rel_z2 = pow(rel_z, 2.0);

		r = sqrt(rel_x2 + rel_y2 + rel_z2);

		g = -body_array[i].get_GM()/ pow(r,2.0);

		//cout << "earth gravity acc: " << g << endl;		

		//r = sqrt(rel_x2 + rel_y2 + rel_z2);

		gx -= g * rel_x/r;
		gy -= g * rel_y/r;
		gz -= g * rel_z/r;




		}
	}


}


void RendezvousTarget::compute_earth_rotation()
{
earth_rotation_angle -= earth_rotation_speed * dt;

}

void RendezvousTarget::compute_elements()
{

long double r = sqrt(pow(x,2.0) + pow(y, 2.0) + pow(z,2.0));


long double specific_energy = 0.5 * (vx * vx + vy * vy + vz * vz) - GM/r;


semi_major = - GM/(2.0 * specific_energy);
orbital_period = 2.0 * pi * sqrt(pow(semi_major, 3.0) / GM);

}


void RendezvousTarget::do_timestep ()
{


compute_gravity();
compute_earth_rotation();


vx += gx * dt;
vy += gy * dt;
vz += gz * dt;
	
x += vx * dt;
y += vy * dt;
z += vz * dt;

	
}

void RendezvousTarget::store()
{

stored_x = x;
stored_y = y;
stored_z = z;

stored_vx = vx;
stored_vy = vy;
stored_vz = vz;

stored_earth_rotation_angle = earth_rotation_angle;
}

void RendezvousTarget::reset()
{

x = stored_x;
y = stored_y;
z = stored_z;

vx = stored_vx;
vy = stored_vy;
vz = stored_vz;

earth_rotation_angle = stored_earth_rotation_angle;
}


long double RendezvousTarget::get_bearing(long double lat2, long double lon2)
{
long double lon1 = get_lon() * deg_to_rad;
long double phi1 = get_lat() * deg_to_rad;

long double delta_lambda = lon2 - lon1;
long double phi2 = lat2;

long double y = sin(delta_lambda) * cos(phi2);
long double x = cos(phi1) * sin(phi2) - sin(phi1) * cos(phi2) * cos(delta_lambda);

return atan2(y,x);
}

long double RendezvousTarget::get_ang_distance(long double lat2, long double lon2)
{

long double lat1 = get_lat() * deg_to_rad;
long double lon1 = get_lon() * deg_to_rad;

long double delta_phi = lat2 - lat1 ;
long double delta_lambda = lon2 - lon1; 

long double a = pow(sin(0.5 * delta_phi), 2.0) + cos(lat1) * cos(lat2) * pow(sin(0.5 * delta_lambda), 2.0);
return 2.0 * atan2(sqrt(a), sqrt(1.0-a));

}


long double RendezvousTarget::get_delta_crosstrack(long double lat2, long double lon2, long double course2)
{

long double delta_13 = get_ang_distance(lat2, lon2);
long double theta_12 = course2;
long double theta_13 = get_bearing(lat2, lon2);

return theta_13 - theta_12;


//return asin(sin(delta_13) * sin(theta_13 - theta_12) );
}


double RendezvousTarget::get_prox_x (long double x_ch, long double y_ch, long double z_ch, long double vx_ch, long double vy_ch, long double vz_ch)
{

long double rnorm_ch = sqrt(pow(x_ch, 2.0) + pow(y_ch, 2.0) + pow(z_ch, 2.0));
long double vnorm_ch = sqrt(pow(vx_ch, 2.0) + pow(vy_ch, 2.0) + pow(vz_ch, 2.0));

long double x_ch_norm = x_ch/rnorm_ch;
long double y_ch_norm = y_ch/rnorm_ch;
long double z_ch_norm = z_ch/rnorm_ch;

long double vx_ch_norm = vx_ch/vnorm_ch;
long double vy_ch_norm = vy_ch/vnorm_ch;
long double vz_ch_norm = vz_ch/vnorm_ch;

long double n_ch_x = y_ch_norm * vz_ch_norm - z_ch_norm * vy_ch_norm;
long double n_ch_y = z_ch_norm * vx_ch_norm - x_ch_norm * vz_ch_norm;
long double n_ch_z = x_ch_norm * vy_ch_norm - y_ch_norm * vx_ch_norm;


long double r =  sqrt(pow(x, 2.0) + pow(y, 2.0) + pow(z, 2.0));

long double x_norm = x/r;
long double y_norm = y/r;
long double z_norm = z/r;

long double ang_xt = 0.5 * pi - acos(x_norm * n_ch_x + y_norm * n_ch_y + z_norm * n_ch_z);
long double delta_ang = acos(x_norm * x_ch_norm + y_norm * y_ch_norm + z_norm * z_ch_norm);


long double ang = acos(cos(delta_ang) / cos(ang_xt));

long double sign = 1.0;

if ( (x-x_ch) * vx_ch_norm + (y - y_ch) * vy_ch_norm + (z-z_ch) * vz_ch_norm < 0.0)
	{
	sign = -1.0;
	}


return  sign *  ang * rnorm_ch;

}

double RendezvousTarget::get_prox_z (long double x_ch, long double y_ch, long double z_ch, long double vx_ch, long double vy_ch, long double vz_ch)
{

long double r_ch = sqrt(pow(x_ch, 2.0)+ pow(y_ch, 2.0) + pow (z_ch, 2.0));
long double r = sqrt(pow(x, 2.0)+ pow(y, 2.0) + pow (z, 2.0));



return r_ch - r;

}


double RendezvousTarget::get_prox_y (long double x_ch, long double y_ch, long double z_ch, long double vx_ch, long double vy_ch, long double vz_ch)
{


long double rnorm_ch = sqrt(pow(x_ch, 2.0) + pow(y_ch, 2.0) + pow(z_ch, 2.0));
long double vnorm_ch = sqrt(pow(vx_ch, 2.0) + pow(vy_ch, 2.0) + pow(vz_ch, 2.0));

long double x_ch_norm = x_ch/rnorm_ch;
long double y_ch_norm = y_ch/rnorm_ch;
long double z_ch_norm = z_ch/rnorm_ch;

long double vx_ch_norm = vx_ch/vnorm_ch;
long double vy_ch_norm = vy_ch/vnorm_ch;
long double vz_ch_norm = vz_ch/vnorm_ch;

long double n_ch_x = y_ch_norm * vz_ch_norm - z_ch_norm * vy_ch_norm;
long double n_ch_y = z_ch_norm * vx_ch_norm - x_ch_norm * vz_ch_norm;
long double n_ch_z = x_ch_norm * vy_ch_norm - y_ch_norm * vx_ch_norm;


long double r =  sqrt(pow(x, 2.0) + pow(y, 2.0) + pow(z, 2.0));

long double x_norm = x/r;
long double y_norm = y/r;
long double z_norm = z/r;

long double ang = 0.5 * pi - acos(x_norm * n_ch_x + y_norm * n_ch_y + z_norm * n_ch_z);

return rnorm_ch * ang;

}


double RendezvousTarget::get_prox_vx (long double x_ch, long double y_ch, long double z_ch, long double vx_ch, long double vy_ch, long double vz_ch)
{

long double vnorm_ch = sqrt(pow(vx_ch, 2.0) + pow(vy_ch,2.0) + pow(vz_ch, 2.0));
long double vnorm = sqrt(pow(vx,2.0) + pow(vy, 2.0) + pow(vz,2.0));

long double vx_ch_norm = vx_ch/vnorm_ch;
long double vy_ch_norm = vy_ch/vnorm_ch;
long double vz_ch_norm = vz_ch/vnorm_ch;

long double vx_norm = vx / vnorm;
long double vy_norm = vy / vnorm;
long double vz_norm = vz / vnorm;

return vx * vx_norm - vx_ch * vx_ch_norm + vy * vy_norm - vy_ch * vy_ch_norm + vz * vz_norm - vz_ch * vz_ch_norm; 

}

double RendezvousTarget::get_prox_vy (long double x_ch, long double y_ch, long double z_ch, long double vx_ch, long double vy_ch, long double vz_ch)
{

long double rnorm_ch = sqrt(pow(x_ch, 2.0) + pow(y_ch, 2.0) + pow(z_ch, 2.0));
long double vnorm_ch = sqrt(pow(vx_ch, 2.0) + pow(vy_ch, 2.0) + pow(vz_ch, 2.0));

long double x_ch_norm = x_ch/rnorm_ch;
long double y_ch_norm = y_ch/rnorm_ch;
long double z_ch_norm = z_ch/rnorm_ch;

long double vx_ch_norm = vx_ch/vnorm_ch;
long double vy_ch_norm = vy_ch/vnorm_ch;
long double vz_ch_norm = vz_ch/vnorm_ch;

long double n_ch_x = y_ch_norm * vz_ch_norm - z_ch_norm * vy_ch_norm;
long double n_ch_y = z_ch_norm * vx_ch_norm - x_ch_norm * vz_ch_norm;
long double n_ch_z = x_ch_norm * vy_ch_norm - y_ch_norm * vx_ch_norm;

return (vx - vx_ch) * n_ch_x + (vy - vy_ch) * n_ch_y + (vz - vz_ch) * n_ch_z; 

}


double RendezvousTarget::get_prox_vz (long double x_ch, long double y_ch, long double z_ch, long double vx_ch, long double vy_ch, long double vz_ch)
{

long double rnorm_ch = sqrt(pow(x_ch, 2.0) + pow(y_ch, 2.0) + pow(z_ch, 2.0));
long double rnorm = sqrt(pow(x, 2.0) + pow (y, 2.0) + pow(z,2.0));

long double x_ch_norm = x_ch/rnorm_ch;
long double y_ch_norm = y_ch/rnorm_ch;
long double z_ch_norm = z_ch/rnorm_ch;

long double x_norm = x / rnorm;
long double y_norm = y / rnorm;
long double z_norm = z / rnorm;


return vx * x_norm - vx_ch * x_ch_norm + vy * y_norm - vy_ch * y_ch_norm + vz * z_norm - vz_ch * z_ch_norm; 

}

double RendezvousTarget::get_prox_x (long double lat_ch, long double lon_ch, long double course_ch)
{


long double delta_13 = get_ang_distance(lat_ch * deg_to_rad, lon_ch * deg_to_rad);
long double delta_xt = get_delta_crosstrack(lat_ch * deg_to_rad, lon_ch * deg_to_rad, course_ch * deg_to_rad);
long double r = sqrt(pow(x, 2.0)+ pow(y, 2.0) + pow (z, 2.0));

return r * acos( cos(delta_13) / cos(delta_xt) );



} 


double RendezvousTarget::get_prox_y (long double lat_ch, long double lon_ch, long double course_ch)
{



long double delta_xt = get_delta_crosstrack(lat_ch * deg_to_rad, lon_ch * deg_to_rad, course_ch * deg_to_rad);
long double r = sqrt(pow(x, 2.0)+ pow(y, 2.0) + pow (z, 2.0));


return delta_xt;

return  delta_xt * r;

} 




double RendezvousTarget::get_x()
{
return x;
}

double RendezvousTarget::get_y()
{
return y;
}

double RendezvousTarget::get_z()
{
return z;
}


double RendezvousTarget::get_vx()
{
return vx;
}

double RendezvousTarget::get_vy()
{
return vy;
}

double RendezvousTarget::get_vz()
{
return vz;
}

double RendezvousTarget::get_alt()
{
return sqrt(pow(x,2.0) + pow(y,2.0) + pow(z, 2.0)) - earth_radius();
}


double RendezvousTarget::get_lat ()
{
return get_latitude_rad() * rad_to_deg;
}


double RendezvousTarget::get_lon()
{
double angle = atan2(y,x) * rad_to_deg + earth_rotation_angle;

while (angle < -180.0)  {angle +=360.0;}


return angle;
}



double RendezvousTarget::get_latitude_rad ()
{
double radius = sqrt(pow(x,2.0) + pow(y,2.0) + pow(z, 2.0));
return asin(z/radius);
}


double RendezvousTarget::get_orbital_period()
{
return orbital_period;
}
