// math_helpers.cxx -- routines to do simple math
//
// Written by Thorsten Renk, started 2017
//
// Copyright (C) 2017  Thorsten Renk - thorsten@science-and-fiction.org 
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301

#include <math.h>
#include <cmath>
#include <iostream>

#include "math_helpers.hxx"

using namespace std;

namespace MathHelpers {

long double dot_product(long double vec1[], long double vec2[])
{
return vec1[0] * vec2[0] + vec1[1] * vec2[1] + vec1[2] * vec2[2];
}

long double norm (long double vec[])
{
return sqrt(vec[0] * vec[0] + vec[1] * vec[1] + vec[2] * vec[2]);
}


long double cross_product_x (long double vec1[3], long double vec2[3])
{
return  vec1[1] * vec2[2] - vec1[2] * vec2[1];
}

long double cross_product_y (long double vec1[3], long double vec2[3])
{
return vec1[2] * vec2[0] - vec1[0] * vec2[2];
}

long double cross_product_z (long double vec1[3], long double vec2[3])
{
return vec1[0] * vec2[1] - vec1[1] * vec2[0];
}


long double orthogonalize_x (long double vec1[3], long double vec2[3])
{
long double angle = dot_product(vec1, vec2);

return vec2[0] - angle * vec1[0];
}

long double orthogonalize_y (long double vec1[3], long double vec2[3])
{
long double angle = dot_product(vec1, vec2);

return vec2[1] - angle * vec1[1];
}

long double orthogonalize_z (long double vec1[3], long double vec2[3])
{
long double angle = dot_product(vec1, vec2);

return vec2[2] - angle * vec1[2];
}


long double sgn(long double x)
{
if (x>0) {return 1.0;}
else if (x < 0) {return -1.0;}
else {return 0.0;}
}




void Spline::init  (int k, long double x_in[1001], long double y[1001])
{

size = k-1;

for (int i=0; i<k+1; i++)
	{
	a[i] = y[i];
	x[i] = x_in[i];
	}

for (int i=0; i< k; i++)
	{
	h[i] = x[i+1] - x[i];
	}

for (int i=1; i< k; i++)
	{
	alpha[i] = 3.0/h[i] * (a[i+1] - a[i]) - 3.0/h[i-1] * (a[i] - a[i-1]);
	} 

l[0] = 1;
mu[0] = 0;
z[0] = 1;

for (int i=1; i<k;i++)
	{
	l[i] = 2.0 * (x[i+1] - x[i-1]) - h[i-1] * mu[i-1];
	mu[i] = h[i]/l[i];
	z[i] = (alpha[i] - h[i-1] * z[i-1])/l[i];
	}
l[k] = 1;
z[k] = 0;
c[k] = 0;

for (int j=k-1; j>-1; j--)
	{
	c[j] = z[j] - mu[j] * c[j+1];
	b[j] = (a[j+1] - a[j])/ h[j] -( h[j] * (c[j+1] + 2.0 * c[j]))/3.0;
	d[j] = (c[j+1] - c[j])/(3.0 * h[j]);
	}


//cout << "Spline: "<<  x[0] << " " << x[1] << endl;

}


long double Spline::interpolate(long double x_in)
{

int i;

for (i=0; i<size; i++)
	{
	//cout << x_in << " " << x[i] << endl;
	if (x_in < x[i]) {break;}
	}

i=i-1;

//cout << "i is now: " << i << endl;

long double res = a[i];

res += b[i] * (x_in - x[i]);
//res += c[i] * (x_in - x[i]) * (x_in - x[i]); 
//res += d[i] * (x_in - x[i]) * (x_in - x[i]) * (x_in - x[i]);
return res;
}


long double Spline::interpolate (long double x_in, int i)
{

long double res = a[i];

res += b[i] * (x_in - x[i]);
res += c[i] * (x_in - x[i]) * (x_in - x[i]); 
res += d[i] * (x_in - x[i]) * (x_in - x[i]) * (x_in - x[i]);

return res;

}


}
