// celestial_body.hxx -- routines to compute a secondary body gravitational
// influence for perturbation analysis
//
// Written by Thorsten Renk, started 2018
//
// Copyright (C) 2018  Thorsten Renk - thorsten@science-and-fiction.org 
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301


#ifndef CELESTIAL_H
#define CELESTIAL_H

#include <string>

#include "math_helpers.hxx"

class Ephemeris {

	long double time[1000];
	long double dec_angle[1000];
	long double ra_angle[1000];
	long double distance[1000];


	int num_entries;

	long double current_dec_angle;
	long double current_ra_angle;
	long double current_distance;

	MathHelpers::Spline dist_spline;
	

	public:

	void init (std::string filename);
	void interpolate(long double time);

	long double get_ra_angle();
	long double get_dec_angle();
	long double get_distance();


};



class CelestialBody {

	long double GM;
	long double J2;
	long double J3;
	long double x;
	long double y;
	long double z;
	long double ux;
	long double uy;
	long double uz;
	long double vx;
	long double vy;
	long double vz;
	long double sidereal_angle;
	long double omega;
	long double distance;

	long double L1_pos[3];
	long double L2_pos[3];
	long double L3_pos[3];
	long double L4_pos[3];
	long double L5_pos[3];

	long double L1_vel[3];
	long double L2_vel[3];
	long double L3_vel[3];
	long double L4_vel[3];
	long double L5_vel[3];

	long double get_radial_acc(long double r);

	std::string name;
	std::string ephemeris_file;

	Ephemeris ephemeris;

	int update_counter;
	bool high_accuracy;
	bool J3_defined;


	public:

	void init (std::string name, std:: string ephemeris_file, long double GM);
	void set_high_accuracy(long double J2, long double J3);
	void update( long double time);
	void find_lagrange();


	long double get_pos_x();
	long double get_pos_y();
	long double get_pos_z();

	long double get_distance();

	long double get_unit_x();
	long double get_unit_y();
	long double get_unit_z();

	long double get_vx();
	long double get_vy();
	long double get_vz();

	long double get_Lagrange_pos(int i, int j);
	long double get_Lagrange_vel(int i, int j);

	long double get_relpos_x(long double x);
	long double get_relpos_y(long double y);
	long double get_relpos_z(long double z);

	long double get_relv_x(long double vx);
	long double get_relv_y(long double vy);
	long double get_relv_z(long double vz);

	long double get_GM();
	long double get_J2();
	long double get_J3();
	long double get_omega();
	std::string get_name();

};


class BarycenterCoords {

	int num_bodies;

	long double x;
	long double y;
	long double z;
	long double vx;
	long double vy;
	long double vz;

	long double x_body[5];
	long double y_body[5];
	long double z_body[5];

	long double x_up[5];
	long double y_up[5];
	long double z_up[5];


	long double vx_body[5];
	long double vy_body[5];
	long double vz_body[5];

	CelestialBody body_array[5];

	public:

	void init (CelestialBody body_array[5], int num_bodies);
	void update( long double time);

	long double get_x();
	long double get_y();
	long double get_z();

	long double get_x(int i);
	long double get_y(int i);
	long double get_z(int i);

	long double get_vx();
	long double get_vy();
	long double get_vz();	

	long double get_vx(int i);
	long double get_vy(int i);
	long double get_vz(int i);

	long double get_up_x(int i);
	long double get_up_y(int i);
	long double get_up_z(int i);

	

};


#endif
