// celestial_body.cxx -- routines to compute a secondary body gravitational
// influence for perturbation analysis
//
// Written by Thorsten Renk, started 2018
//
// Copyright (C) 2018  Thorsten Renk - thorsten@science-and-fiction.org 
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301

#include <iostream>
#include <iomanip>
#include <fstream>
#include <stdlib.h>
#include <math.h>
#include <cmath>

#include "constants.hxx"
#include "celestial_body.hxx"


using namespace std;
using namespace MathHelpers;


// Ephemeris class

void Ephemeris::init(string filename)
{

int inner_counter;
double value;
string element;

inner_counter = 0;
num_entries = 0;


ifstream ephfile (filename.c_str());
if (ephfile.is_open())
  {
    while (!ephfile.eof() && (num_entries < 1000))
    {
      ephfile >> element;
      value = atof (element.c_str());

	//cout << "Current value: " << value << endl;

      switch (inner_counter) {
		case 0: time[num_entries] = value;  break;
		case 1: ra_angle[num_entries] = value; break;
		case 2: dec_angle[num_entries] = value; break;
		case 3: distance[num_entries] = value; num_entries++; break;

		}
		inner_counter++;
		if (inner_counter == 4) {inner_counter = 0;}

    
    }
    ephfile.close();

    cout << num_entries << " ephemeris entries read." << endl;
    current_dec_angle = dec_angle[0];
    current_ra_angle = ra_angle[0];
    current_distance = distance[0];


 //   dist_spline.init(num_entries, time, distance);

   // for (int i=0; i< 300; i++)
//	{
//	long double test_time = 100.0 * i;
//	cout << test_time << " " << dist_spline.interpolate(test_time) << endl;
//	}

  }

  else 
	{
	cout << "Unable to open file " << filename << " !" << endl;
  	exit(0);
	}
}


void Ephemeris::interpolate(long double current_time)
{

double int_factor;

if (current_time < time[0])
	{
	cout << "Time outside ephemeris table!" << endl;
	current_dec_angle = dec_angle[0];
    	current_ra_angle = ra_angle[0];
   	current_distance = distance[0];
	} 
else if (current_time > time[num_entries -1])
	{
	cout << "Time outside ephemeris table!" << endl;
	current_dec_angle = dec_angle[num_entries-1];
    	current_ra_angle = ra_angle[num_entries-1];
   	current_distance = distance[num_entries-1];
	}
else
	{
	for (int i=1; i< num_entries; i++)
		{
		if (current_time < time[i])
			{
			int_factor = (current_time - time[i-1])/(time[i] - time[i-1]);

			current_dec_angle = dec_angle[i-1] + int_factor * (dec_angle[i] - dec_angle[i-1]);
			current_ra_angle = ra_angle[i-1] + int_factor * (ra_angle[i] - ra_angle[i-1]);
			current_distance = distance[i-1] + int_factor * (distance[i] - distance[i-1]);

			// need to separate the case when RA crosses 360 to 0 deg

			if ((ra_angle[i-1] > 330.0) && (ra_angle[i] < 30.0))
				{
				current_ra_angle = ra_angle[i-1] + int_factor * (ra_angle[i] + 360.0 - ra_angle[i-1]);
				}
			else if ((ra_angle[i-1] < 30.0) && (ra_angle[i] > 330.0))
				{
				current_ra_angle = ra_angle[i-1] - 360.0 + int_factor * (ra_angle[i] - ra_angle[i-1] - 360.0);
				}

			break;
			}

		}
	}

}


long double Ephemeris::get_ra_angle()
{
return -current_ra_angle;
}

long double Ephemeris::get_dec_angle()
{
return current_dec_angle;
}

long double Ephemeris::get_distance()
{
return current_distance;
}



//Celestial body class



void CelestialBody::init(string set_name, string set_ephemeris_file, long double set_GM)
{

cout << "Adding celestial object " << set_name << "." << endl;

GM = set_GM;

//cout << "GM is now: " << GM << endl;

name = set_name;
ephemeris_file = set_ephemeris_file;
ephemeris.init(ephemeris_file);
update_counter = 0;
sidereal_angle = 0.0;

high_accuracy = false;
J3_defined = false;
J2 = 0.0;
J3 = 0.0;

update(0.0);

}

void CelestialBody::set_high_accuracy(long double set_J2, long double set_J3)
{
J2 = set_J2;

if (set_J3 != 0.0)	
	{
	J3 = set_J3;
	J3_defined = true;
	}

}



void CelestialBody::update(long double time)
{

if (update_counter == 0)
	{	
	ephemeris.interpolate(time + 1.0);
	//ephemeris.interpolate(0);

	long double ra_ang_f = (ephemeris.get_ra_angle() + sidereal_angle) * deg_to_rad;
	long double dec_ang_f = ephemeris.get_dec_angle() * deg_to_rad;
	distance = ephemeris.get_distance() * 1000.0;

	

	long double xf = distance * cos(dec_ang_f) * cos(ra_ang_f);
	long double yf = distance * cos(dec_ang_f) * sin(ra_ang_f);
	long double zf = distance * sin (dec_ang_f);	 




	ephemeris.interpolate(time);
	//ephemeris.interpolate(0);

	long double ra_ang = (ephemeris.get_ra_angle() + sidereal_angle) * deg_to_rad;
	long double dec_ang = ephemeris.get_dec_angle() * deg_to_rad;

	long double delta_ra_ang = ra_ang_f - ra_ang;
	long double delta_dec_ang = dec_ang_f - dec_ang;
	
	omega = sqrt(delta_ra_ang * delta_ra_ang + delta_dec_ang * delta_dec_ang);

	distance = ephemeris.get_distance() * 1000.0;



	ux = cos(dec_ang) * cos(ra_ang);
	uy = cos(dec_ang) * sin(ra_ang);
	uz = sin(dec_ang);

	x = distance * ux;
	y = distance * uy;
	z = distance * uz;




	vx = xf - x;
	vy = yf - y;
 	vz = zf - z;

	}

if (!high_accuracy) {update_counter++;}
if (update_counter == 100) {update_counter = 0;}
}

long double CelestialBody::get_radial_acc(long double r)
{

return -GM_Earth/pow(r, 2.0) * sgn(r) + GM/pow((distance - r),2.0) * sgn(distance - r) + (GM_Earth *r + GM * r - GM * distance)/pow(distance, 3.0);
}


void CelestialBody::find_lagrange()
{

//long double start_x = - 1.5 * x;
//long double start_y = - 1.5 * y;
//long double start_z = - 1.5 * z;

//long double end_x = 1.5 * x;
//long double end_y = 1.5 * y;
//long double end_z = 1.5 * z;

//long double delta_x = end_x - start_x;
//long double delta_y = end_y - start_y;
//long double delta_z = end_z - start_z;

//long double search_length = sqrt(delta_x * delta_x + delta_y * delta_y + delta_z * delta_z);

long double search_length = 3.0 * distance;

long double r;
long double a, a_last;

int counter;
long double crossing[3];

counter = 0;

for (int i=0; i< 1000; i++)
	{
	r = -1.5 * distance + i * search_length/1000.0;

	//cout << "R: " << r << endl;

	if (abs(r) < 1.0) {r = 1.0;} 
	if (abs (distance-r) < 1.0) {r = distance + 1.0;}

	a = get_radial_acc(r);


	//cout << "a: " << a << endl;

	if ((i>0) && (sgn(a) != sgn(a_last) && (abs(a) < 1.0)))
		{
		//cout << "zero crossing at " << r/distance << endl;
		//cout << "a is now" << a << endl;
		//cout << "counter: " << counter << endl;
		crossing[counter] = r;
		counter++;
		}
	a_last = a;

	}



for (int j=0; j< 3; j++)
	{
	for (int i=0; i< 1000; i++)
		{
		
		r = crossing[j] - i * search_length/1.0e6;

		if (abs(r) < 1.0) {r = 1.0;} 
		if (abs (distance-r) < 1.0) {r = distance + 1.0;}

		a = get_radial_acc(r);

		if ((i>0) && (sgn(a) != sgn(a_last) && (abs(a) < 1.0)))
			{
			if (j == 0) 
				{
				//cout << "L3 at " << r/distance << " radii" << endl;
				L3_pos[0] = ux * r;
				L3_pos[1] = uy * r;
				L3_pos[2] = uz * r;

				L3_vel[0] = vx * r/distance;
				L3_vel[1] = vy * r/distance;				
				L3_vel[2] = vz * r/distance;								
				}
			else if (j == 1)
				{
				//cout << "L1 at " << r/distance << " radii" << endl;
				L1_pos[0] = ux * r;
				L1_pos[1] = uy * r;
				L1_pos[2] = uz * r;

				L1_vel[0] = vx * r/distance;
				L1_vel[1] = vy * r/distance;				
				L1_vel[2] = vz * r/distance;
				}
			else if (j == 2)
				{
				//cout << "L2 at " << r/distance << " radii" << endl;
				L2_pos[0] = ux * r;
				L2_pos[1] = uy * r;
				L2_pos[2] = uz * r;

				L2_vel[0] = vx * r/distance;
				L2_vel[1] = vy * r/distance;				
				L2_vel[2] = vz * r/distance;
				}
			break;
			}

		}	

	}

}


long double CelestialBody::get_Lagrange_pos(int i, int j)
{
if (i==1) {return L1_pos[j];}
else if (i==2) {return L2_pos[j];}
else if (i==3) {return L3_pos[j];} 
else if (i==4) {return L4_pos[j];} 
else if (i==5) {return L5_pos[j];} 

}

long double CelestialBody::get_Lagrange_vel(int i, int j)
{
if (i==1) {return L1_vel[j];}
else if (i==2) {return L2_vel[j];}
else if (i==3) {return L3_vel[j];} 
else if (i==4) {return L4_vel[j];} 
else if (i==5) {return L5_vel[j];} 

}

long double CelestialBody::get_pos_x()
{
return x;
}

long double CelestialBody::get_pos_y()
{
return y;
}

long double CelestialBody::get_pos_z()
{
return z;
}

long double CelestialBody::get_distance()
{
return distance;
}

long double CelestialBody::get_unit_x()
{
return ux;
}

long double CelestialBody::get_unit_y()
{
return uy;
}

long double CelestialBody::get_unit_z()
{
return uz;
}

long double CelestialBody::get_vx()
{
return vx;
}

long double CelestialBody::get_vy()
{
return vy;
}

long double CelestialBody::get_vz()
{
return vz;
}

long double CelestialBody::get_relpos_x(long double x_in)
{
return x_in - x; 
}

long double CelestialBody::get_relpos_y(long double y_in)
{
return y_in - y; 
}

long double CelestialBody::get_relpos_z(long double z_in)
{
return z_in - z; 
}

long double CelestialBody::get_relv_x(long double vx_in)
{
return vx_in - vx; 
}

long double CelestialBody::get_relv_y(long double vy_in)
{
return vy_in - vy; 
}

long double CelestialBody::get_relv_z(long double vz_in)
{
return vz_in - vz; 
}

long double CelestialBody::get_GM()
{
return GM;
}

long double CelestialBody::get_omega()
{
return omega;
}

string CelestialBody::get_name()
{
return name;
}


// Barycenter coords class


void BarycenterCoords::init (CelestialBody body_array_in[5], int num_bodies_in)
{

cout << "Initializing barycenter for " << num_bodies_in + 1 << " objects." << endl;

for (int i=0; i< num_bodies_in; i++)
	{
	body_array[i] = body_array_in[i];
	x_body[i] = 0.0;
	y_body[i] = 0.0;
	z_body[i] = 0.0;

	vx_body[i] = 0.0;
	vy_body[i] = 0.0;
	vz_body[i] = 0.0;

	}

num_bodies = num_bodies_in;
}


void BarycenterCoords::update (long double time)
{

long double GM_tot = GM;

x = 0.0;
y = 0.0;
z = 0.0;

vx = 0.0;
vy = 0.0;
vz = 0.0;

for (int i=0; i< num_bodies; i++)
	{
	body_array[i].update(time);
	GM_tot += body_array[i].get_GM();
	}

for (int i=0; i< num_bodies; i++)
	{

	// system barycenter

	long double weight = body_array[i].get_GM() / GM_tot;

	x += body_array[i].get_pos_x() * weight;	
	y += body_array[i].get_pos_y() * weight;	
	z += body_array[i].get_pos_z() * weight;	

	vx += body_array[i].get_vx() * weight;
	vy += body_array[i].get_vy() * weight;
	vz += body_array[i].get_vz() * weight;

	// 2-body barycenter with Earth

	weight =  body_array[i].get_GM() / (body_array[i].get_GM() + GM);

	x_body[i] = body_array[i].get_pos_x() * weight;
	y_body[i] = body_array[i].get_pos_y() * weight;
	z_body[i] = body_array[i].get_pos_z() * weight;

	vx_body[i] = body_array[i].get_vx() * weight;
	vy_body[i] = body_array[i].get_vy() * weight;
	vz_body[i] = body_array[i].get_vz() * weight;
	
	// and 2-body up-vector out of plane

	long double xvec_body[3];
	xvec_body[0] = x_body[i];
	xvec_body[1] = y_body[i];
	xvec_body[2] = z_body[i];

	long double xnorm = norm(xvec_body);
	xvec_body[0] /= xnorm;
	xvec_body[1] /= xnorm;
	xvec_body[2] /= xnorm;

	long double vvec_body[3];
	vvec_body[0] = vx_body[i];
	vvec_body[1] = vy_body[i];
	vvec_body[2] = vz_body[i];

	long double vnorm = norm(vvec_body);
	vvec_body[0] /= vnorm;
	vvec_body[1] /= vnorm;
	vvec_body[2] /= vnorm;


	x_up[i] =  cross_product_x (xvec_body, vvec_body);
	y_up[i] =  cross_product_y (xvec_body, vvec_body);
	z_up[i] =  cross_product_z (xvec_body, vvec_body);

	long double upnorm = sqrt(x_up[i] * x_up[i] + y_up[i] * y_up[i] + z_up[i] * z_up[i]);
	x_up[i] /= upnorm;
	y_up[i] /= upnorm;
	z_up[i] /= upnorm;


	}

x = -x;
y = -y;
z = -z;

vx = -vx;
vy = -vy;
vz = -vz;


//cout << "Barycenter pos: " << x << " " << y << " " << z << endl;
//cout << "Barycenter vel: " << vx << " " << vy << " " << vz << endl;

}


long double BarycenterCoords::get_x()
{
return x;
}

long double BarycenterCoords::get_x(int i)
{
return x_body[i];
}

long double BarycenterCoords::get_y()
{
return y;
}

long double BarycenterCoords::get_y(int i)
{
return y_body[i];
}

long double BarycenterCoords::get_z()
{
return z;
}

long double BarycenterCoords::get_z(int i)
{
return z_body[i];
}

long double BarycenterCoords::get_vx()
{
return vx;
}

long double BarycenterCoords::get_vx(int i)
{
return vx_body[i];
}

long double BarycenterCoords::get_vy()
{
return vy;
}

long double BarycenterCoords::get_vy(int i)
{
return vy_body[i];
}

long double BarycenterCoords::get_vz()
{
return vz;
}

long double BarycenterCoords::get_vz(int i)
{
return vz_body[i];
}

long double BarycenterCoords::get_up_x(int i)
{
return x_up[i];
}

long double BarycenterCoords::get_up_y(int i)
{
return y_up[i];
}

long double BarycenterCoords::get_up_z(int i)
{
return z_up[i];
}
