// burn.hxx -- routines to simulate an untargeted or targeted orbital
// correction burn
//
// Written by Thorsten Renk, started 2017
//
// Copyright (C) 2017  Thorsten Renk - thorsten@science-and-fiction.org 
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301


#ifndef BURN_H
#define BURN_H

#include <string>

class Burn {

	// PEG-7
	long double delta_v_prograde;
	long double delta_v_radial;
	long double delta_v_normal;


	// low thrust
	long double tgt_x;
	long double tgt_y;
	long double tgt_z;
	long double tgt_threshold;
	bool force_plane;


	// PEG-4
	long double peg4_thetaT;
	long double peg4_H;
	long double peg4_c1;
	long double peg4_c2;
	int peg4_ops;

	// optimized PEG-4

	long double peg4_rei;


	// Lambert

	long double lambert_t1;
	long double lambert_t2;
	long double lambert_delta_t;

	long double lambert_offset_x;
	long double lambert_offset_y;
	long double lambert_offset_z;

	long double lambert_target_x;
	long double lambert_target_y;
	long double lambert_target_z;

	long double lambert_transfer_angle;

	double lambert2_delta_v_prograde;
	double lambert2_delta_v_radial;
	double lambert2_delta_v_normal;



	//fit

	long double fit_last_inc_radial;
	long double fit_last_inc_prograde;
	long double fit_last_y_error;

	long double fit_radial_limit;
	long double fit_prograde_limit;
	long double fit_gain_prograde;
	long double fit_gain_radial;
	long double fit_lambert_accept;
	long double fit_accept;

	int num_iterations;
	int num_precise_iterations;
	int fit_strategy;
	bool fit_initialized;
	bool fit_lambert_dry_run;
	bool fit_lambert_y;
	bool fit_new_tig;
	bool fit_converged;
	bool fit_log;
	bool fit_theta;

	//spacecraft
	
	long double craft_acceleration;
	long double craft_dry_mass;
	long double craft_propellant_mass;
	long double craft_isp;
	long double craft_thrust;
	long double craft_mass_flow;

	bool craft_defined;

	// general
	long double delta_v_tot;
	long double delta_v_impulse;

	long double unit_prograde;
	long double unit_radial;
	long double unit_normal;

	long double ignition_time;
	long double burn_time;
	long double acceleration;

	bool burn_active;
	bool burn_finished;

	int burn_model;
	int burn_type;

	std::string designation;


	long double clamp(long double arg, long double limit);

	bool check_signflip(long double arg, long double lastarg);

	void compute_vvectors();
	
	public:

	void init (long double v_prog, long double v_rad, long double v_norm, long double tig, std::string designation, int burn_model);
	void init_peg4 (long double theta_T, long double H, long double c1, long double c2, long double tig, std::string designation, int burn_model, int flag);
	void init_peg4_optimized (long double theta_T, long double H, long double c1, long double c2, long double rei, long double tig, std::string designation, int burn_model);
	void init_low_thrust (long double v_prog, long double v_rad, long double v_norm, long double tig, long double v_tgt[], long double threshold, std::string designation, int burn_model);
	void init_lambert (long double t1, long double t2, long double x_offset, long double y_offset, long double z_offset, std::string designation, int burn_model);
	void update_low_thrust(long double nx, long double ny, long double nz);	
	void start();
	void start(long double vx, long double vy, long double vz);
	void do_timestep(long double timestep);
	void new_tig_acknowledge();
	void set_fit_log(bool flag);	
	void set_ops(int ops);
	void set_acceleration(long double acceleration);
	void set_spacecraft(long double mass_dry, long double mass_propellant, long double isp, long double thrust);
	void set_propellant(long double propellant_mass);
	void set_vnorm(long double vnorm []);
	void set_plane_flag (bool plane_flag);
	void set_dry_run(bool flag);
	void set_y_option(bool flag);
	void set_lambert_target(long double x, long double y, long double z);
	void set_lambert2_parameters(long double dvx, long double dvy, long double dvz);

	void peg4_guess_initial (long double altitude, long double apoapsis, long double periapsis, long double vspeed, long double orbital_speed_diff, int num_precise_iterations);
	void peg4_improve (long double thetaT, long double H, long double vspeed, long double hspeed, long double rei);

	void lambert_guess_initial(long double prox_x, long double prox_y, long double prox_z, long double tgt_period, int num_precise_iterations);
	void lambert_improve(long double prox_x, long double prox_y, long double prox_z);

	long double get_acc_prograde();		
	long double get_acc_radial();
	long double get_acc_normal();
	long double get_acceleration();
	long double get_H_target();

	long double get_tig();
	long double get_delta_v_prograde();
	long double get_delta_v_radial();
	long double get_delta_v_normal();
	long double get_delta_v_prograde2();
	long double get_delta_v_radial2();
	long double get_delta_v_normal2();
	long double get_c1();
	long double get_c2();
	long double get_thetaT();
	long double get_H();
	long double get_tig2();

	long double get_propellant();

	std::string get_name();
	
	int get_burn_type();
	int get_num_iterations();
	int get_strategy();
	int get_ops();

	bool is_active(long double time);
	bool is_finished();
	bool is_dry_run();
	bool is_initialized();
	bool is_converged();	
	bool is_yfit_active();
	bool tig_changed();




};




#endif
