// burn.cxx -- routines to simulate an untargeted or targeted orbital
// correction burn
//
// Written by Thorsten Renk, started 2017
//
// Copyright (C) 2017  Thorsten Renk - thorsten@science-and-fiction.org 
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301

#include <stdlib.h>
#include <iostream>
#include <math.h>
#include <cmath>

#include "constants.hxx"
#include "burn.hxx"
#include "rendezvous_target.hxx"



using namespace std;




void Burn::init (long double v_prog, long double v_rad, long double v_norm, long double tig, std::string set_designation, int set_burn_model)
{

burn_type = 0;


designation = set_designation;

delta_v_prograde = v_prog;
delta_v_radial = v_rad;
delta_v_normal = v_norm;
burn_model = set_burn_model;

ignition_time = tig;
burn_time = 0.0;
burn_active = false;
burn_finished = false;


fit_log = false;
fit_strategy = -1;
fit_lambert_dry_run = false;

craft_acceleration = 12174.0 / 254600.0 * 9.81;
acceleration = get_acceleration();

if ((set_burn_model == 0) || (set_burn_model == 1))
	{
	craft_propellant_mass = 0.0;
	}

compute_vvectors();

}

void Burn::init_low_thrust (long double v_prog, long double v_rad, long double v_norm, long double tig, long double v_tgt[], long double threshold, std::string set_designation, int set_burn_model)
{

burn_type = 3;


designation = set_designation;

delta_v_prograde = v_prog;
delta_v_radial = v_rad;
delta_v_normal = v_norm;
burn_model = set_burn_model;

ignition_time = tig;
burn_time = 0.0;
burn_active = false;
burn_finished = false;
force_plane = true;

fit_log = false;
fit_strategy = -1;
fit_lambert_dry_run = false;

tgt_x = v_tgt[0];
tgt_y = v_tgt[1];
tgt_z = v_tgt[2];
tgt_threshold = threshold; 

craft_acceleration = 12174.0 / 254600.0 * 9.81;
acceleration = get_acceleration();

if ((set_burn_model == 0) || (set_burn_model == 1))
	{
	craft_propellant_mass = 0.0;
	}

compute_vvectors();

}


void Burn::init_peg4 (long double set_theta_T, long double set_H, long double set_c1, long double set_c2, long double set_tig, std::string set_designation, int set_burn_model, int flag)
{
burn_type = 1;

designation = set_designation;

peg4_thetaT = set_theta_T;
peg4_H = set_H;
peg4_c1 = set_c1;
peg4_c2 = set_c2;
peg4_ops = 3;

cout << "PEG-4 targets: thetaT " << peg4_thetaT << " H " << peg4_H << " c1 " << peg4_c1 << " c2 " << peg4_c2 << endl;

ignition_time = set_tig;
burn_time = 0.0;
num_iterations = 0;
fit_converged = false;
fit_initialized = false;
burn_active = false;
burn_finished = false;
fit_lambert_dry_run = false;

if (flag == 0)
	{
	fit_theta = false;
	}
else
	{
	fit_theta = true;
	}

burn_model = set_burn_model;

craft_acceleration = 12174.0 / 254600.0 * 9.81;
acceleration = get_acceleration();

if ((set_burn_model == 0) || (set_burn_model == 1))
	{
	craft_propellant_mass = 0.0;
	}

}


void Burn::init_peg4_optimized (long double set_theta_T, long double set_H, long double set_c1, long double set_c2, long double set_rei, long double set_tig, std::string set_designation, int set_burn_model)
{
burn_type = 2;

designation = set_designation;

peg4_thetaT = set_theta_T;
peg4_H = set_H;
peg4_c1 = set_c1;
peg4_c2 = set_c2;
peg4_rei = set_rei;
peg4_ops = 3;


ignition_time = set_tig;
burn_time = 0.0;
num_iterations = 0;
fit_converged = false;
fit_new_tig = false;
fit_initialized = false;
burn_active = false;
burn_finished = false;
fit_theta = false;
fit_lambert_dry_run = false;

burn_model = set_burn_model;

craft_acceleration = 12174.0 / 254600.0 * 9.81;
acceleration = get_acceleration();

if ((set_burn_model == 0) || (set_burn_model == 1))
	{
	craft_propellant_mass = 0.0;
	}

}

void Burn::init_lambert(long double set_t1, long double set_t2, long double set_xoffset, long double set_yoffset, long double set_zoffset, std::string set_designation, int set_burn_model)
{
burn_type = 5;

lambert_t1 = set_t1;
lambert_t2 = set_t2;
lambert_offset_x = set_xoffset;
lambert_offset_y = set_yoffset;
lambert_offset_z = set_zoffset;

delta_v_prograde = 0.0;
delta_v_radial = 0.0;
delta_v_normal = 0.0;


designation = set_designation;

burn_model = set_burn_model;

ignition_time = set_t1;
burn_time = 0.0;
num_iterations = 0;
fit_converged = false;
fit_initialized = false;
fit_new_tig = false;
burn_active = false;
burn_finished = false;
fit_lambert_dry_run = false;

compute_vvectors();
}  

void Burn::set_ops(int ops)
{

if ((ops !=1) && (ops !=3))
	{
	cout << "Only OPS 1 and 3 angular definitions exist! Aborting..." << endl; exit(0);

	}

cout << "Using OPS " << ops << " angle definitions for burn " << designation << endl;

peg4_ops = ops;
}





void Burn::compute_vvectors()
{
delta_v_tot = sqrt(delta_v_prograde * delta_v_prograde + delta_v_radial * delta_v_radial + delta_v_normal * delta_v_normal);
delta_v_impulse = delta_v_tot;


unit_prograde = delta_v_prograde/delta_v_tot;
unit_radial = delta_v_radial/delta_v_tot;
unit_normal = delta_v_normal/delta_v_tot;
}


void Burn::peg4_guess_initial(long double altitude, long double periapsis, long double apoapsis, long double vspeed, long double orbital_speed_diff, int num_precise_iterations_in)
{

fit_radial_limit = 10.0;
fit_prograde_limit = 40.0;

fit_last_inc_radial = 0.0;
fit_last_inc_prograde = 0.0;

fit_gain_radial = 3.0;
fit_gain_prograde = 0.003;

num_precise_iterations = num_precise_iterations_in;
fit_accept = 1.0;


if ((altitude > peg4_H + 15000.0) && (peg4_c1 > 0.0)) // intersection with lower altitude
	{
	fit_strategy = 0;
	}
else if ((altitude < peg4_H - 15000.0) && (peg4_c1 < 0.0)) // intersection with higher altitude
	{
	fit_strategy = 1;
	}	
else if ((altitude > peg4_H + 15000.0) && (peg4_c1 == 0.0)) // apsis at lower altitude
	{
	fit_strategy = 2;
	}
else if ((altitude < peg4_H - 15000.0) && (peg4_c1 == 0.0)) // apsis at higher altitude
	{
	fit_strategy = 3;
	}
else //circularization burn
	{
	fit_strategy = 4;
	}


long double phase = acos(2.0 * (((altitude - periapsis) / (apoapsis - periapsis)) - 0.5));
if (vspeed > 0.0) {phase = phase + pi;}

long double phase_error = 0.0;

if (fit_strategy == 3)
	{
	phase_error = 2.0 * pi - phase + deg_to_rad * peg4_thetaT;
	}


while (phase_error > 2.0*pi) 
	{phase_error -=2.0 * pi;}
phase_error -= 2.0 * pi;




long double expected_alt = periapsis + (apoapsis - periapsis) * 0.5 * (cos(phase + deg_to_rad * peg4_thetaT) + 1.0);

//cout << "apoapsis: " << apoapsis << " periapsis: " << periapsis << endl;
//cout << "PEG-4 H : " << peg4_H << endl;
//cout << "expected alt: " << expected_alt << endl;

if (fit_strategy < 2)
	{
	delta_v_prograde = (peg4_H - expected_alt)/2500.0;
	delta_v_radial = 0.0;
	}
else if (fit_strategy == 2)
	{
	delta_v_prograde = (peg4_H - expected_alt)/3500.0;
	delta_v_radial = 0.0;
	}
else if (fit_strategy == 3)
	{
	delta_v_prograde = (peg4_H - expected_alt)/3500.0;
	delta_v_radial = 0.0 * phase_error;
	}
else if (fit_strategy == 4)
	{
	delta_v_prograde = orbital_speed_diff;
	delta_v_radial = - 0.73 * vspeed;
	}

delta_v_normal = 0.0;


if (fit_log)
	{
	cout << "PEG-4 initial guess for burn " << designation << "  vx: " << delta_v_prograde  << endl;
	cout << "Using fit strategy " << fit_strategy << endl;
	}

compute_vvectors();

fit_initialized = true;
}


void Burn::peg4_improve (long double thetaT, long double H, long double vspeed, long double hspeed, long double rei)
{

long double rei_error;
long double increment;


long double vspeed_error = (peg4_c2 * hspeed  + peg4_c1) - vspeed;
long double H_error = peg4_H - H;
long double thetaT_error = ( peg4_thetaT - thetaT);


//cout << "Theta T is now: " << thetaT << endl;
//cout << "Theta T error is now: " << thetaT_error << endl;

if (abs(thetaT_error) > 180.0)
	{
	thetaT_error = thetaT_error - 360.0;
	//cout << "ThetaT error periodic correction!" << endl;
	}



bool inner_fit_converged = false;

if (fit_theta)
	{
	peg4_thetaT = thetaT;
	thetaT_error = 0.0;
	}


if (fit_log)
	{
	cout << "vspeed error: " << vspeed_error << endl;
	cout << "H error     : " << H_error << endl;
	cout << "thetaT error: " << thetaT_error << endl;
	}


long double H_accept_factor = 1.0;
long double thetaT_accept_factor = 1.0;

if ((fit_strategy == 2) || (fit_strategy == 3))
	{
	H_accept_factor = 5.0;
	thetaT_accept_factor = 2.0;
	}
else if (fit_strategy == 4)
	{
	H_accept_factor = 20.0;
	}	

if (num_iterations > num_precise_iterations) 
	{
	fit_accept *= 1.2;
	}


if ((abs(vspeed_error) < 1.0) && (abs(H_error) < 10.0 * H_accept_factor * fit_accept) && (abs(thetaT_error) < 0.1 * thetaT_accept_factor * fit_accept) )

	{
	vspeed_error = 0.0;
	H_error = 0.0;
	thetaT_error = 0.0;
	inner_fit_converged = true;
	if (fit_log)
		{cout << "***************** Inner fit converged ***************** "<< endl;}
	}


if (num_iterations > 1.0)


if (fit_strategy == 0)
	{
	delta_v_radial += 3.0 * thetaT_error;
	delta_v_prograde += 0.2 * vspeed_error;
	}
else if (fit_strategy == 1)
	{
	delta_v_radial -= 3.0 * thetaT_error;
	delta_v_prograde += 0.2 * vspeed_error;
	}
else if (fit_strategy == 2)
	{

	increment = fit_gain_prograde * H_error;
	if (check_signflip (increment, fit_last_inc_prograde))
		{
		fit_prograde_limit *=0.5;
		}
	increment = clamp(increment, fit_prograde_limit);
	fit_last_inc_prograde = increment;
	delta_v_prograde += increment;


	increment =  fit_gain_radial * thetaT_error;
	if (check_signflip (increment, fit_last_inc_radial))
		{
		fit_radial_limit *=0.5;
		fit_gain_radial *=1.5;
		}
	increment = clamp(increment, fit_radial_limit);
	fit_last_inc_radial = increment;	
	delta_v_radial += increment;

	}
else if (fit_strategy == 3)
	{

	increment = fit_gain_prograde * H_error;
	if (check_signflip (increment, fit_last_inc_prograde))
		{
		fit_prograde_limit *=0.5;

		}
	increment = clamp(increment, fit_prograde_limit);
	fit_last_inc_prograde = increment;
	delta_v_prograde += increment;

	increment =  fit_gain_radial * thetaT_error;
	if (check_signflip (increment, fit_last_inc_radial))
		{
		fit_radial_limit *=0.5;
		fit_gain_radial *=1.5;
		}
	increment = clamp(increment, fit_radial_limit);
	fit_last_inc_radial = increment;	

	delta_v_radial -= increment;
	}
else if (fit_strategy == 4)
	{
	increment = fit_gain_prograde * H_error;
	if (check_signflip (increment, fit_last_inc_prograde))
		{
		fit_prograde_limit *=0.5;

		}
	increment = clamp(increment, fit_prograde_limit);
	fit_last_inc_prograde = increment;
	delta_v_prograde += increment;

	increment =  fit_gain_radial * vspeed_error;
	if (check_signflip (increment, fit_last_inc_radial))
		{
		fit_radial_limit *=0.5;
		fit_gain_radial *=1.5;
		}
	increment = clamp(increment, fit_radial_limit);
	fit_last_inc_radial = increment;	

	delta_v_radial -= increment;
	}

delta_v_normal = 0.0;

if ((burn_type == 2) && (inner_fit_converged))
	{
	if ((fit_strategy == 0) || (fit_strategy == 1))
		{
		rei_error = peg4_rei - rei;
		if (fit_log)
			{cout << "REI error [nm]: " << rei_error / 1852.0 << endl;}
		ignition_time -= rei_error /10000;
		peg4_thetaT -= 0.05 * delta_v_radial;
		fit_new_tig = true;
		}
	else if ((fit_strategy == 2) || (fit_strategy == 3) || (fit_strategy == 4))
		{

		ignition_time -=  8.0 * delta_v_radial;
		
		//cout<< "==========================" << endl;
		//cout << "Delta V rad: " << delta_v_radial << endl;
		//cout<< "==========================" << endl;
		rei_error = 0.0;
		fit_radial_limit = 4.0;
		fit_prograde_limit = 10.0;

		fit_last_inc_radial = 0.0;
		fit_last_inc_prograde = 0.0;

		fit_gain_radial = 3.0;
		fit_gain_prograde = 0.003;
		fit_new_tig = true;	
		}

	}
else 
	{
	rei_error = 0.0;
	}

if ((inner_fit_converged) && (abs(rei_error) < 30000.0) && (abs(delta_v_radial) < 2.0) && (burn_type == 2))
	{
	fit_converged = true;
	if (fit_log)
		{
		cout << "Fit converged!" << endl;
		}
	}
else if ((inner_fit_converged) && (burn_type == 1))
	{
	fit_converged = true;
	if (fit_log)
		{
		cout << "Fit converged!" << endl;
		}
	}

if (fit_log)
	{
	cout << "New dvx: " << delta_v_prograde << endl;
	cout << "New dvz: " << delta_v_radial << endl;

	if (burn_type == 2)
		{
		cout << "New TIG: " << ignition_time << endl;
		}

	}
compute_vvectors();

burn_time = 0.0;
burn_active = false;
burn_finished = false;

num_iterations++;

}


void Burn::lambert_guess_initial(long double prox_x, long double prox_y, long double prox_z, long double tgt_period, int num_precise_iterations_in)
{

lambert_transfer_angle = 2.0 * pi * ((lambert_t2 - lambert_t1) / tgt_period);

if (abs(sin(lambert_transfer_angle)) < 0.05)
	{
	cout << "Lambert transfer angle unsuitable to change to target plane." << endl;
	cout << "Disabling y-fit for burn " << designation << "." << endl;
	fit_lambert_y = false;

	}

num_precise_iterations = num_precise_iterations_in;

//cout << "Tranfer angle: " << lambert_transfer_angle * 180.0/pi << endl; 

long double x_error = prox_x - lambert_offset_x;
long double y_error = prox_y - lambert_offset_y;
long double z_error = prox_z - lambert_offset_z;

delta_v_prograde = -prox_z/2500.0;
delta_v_radial = prox_x/4000;
delta_v_normal = 0.0;


fit_last_y_error = 1e10;
fit_gain_prograde = 1.0;
fit_gain_radial = 1.0;
fit_lambert_accept = 1.0;

if (fit_log)
	{

	cout << "Lambert initial guess for burn " << designation << "  vx: " << delta_v_prograde  << " vz " << delta_v_radial << endl;
	cout << "Transfer angle is " << lambert_transfer_angle * 180.0/3.1415926535 << " degrees." << endl;
	cout << "Dry run errors x: " << x_error << " z: " << z_error << endl; 
	//cout << "Using fit strategy " << fit_strategy << endl;
	}

compute_vvectors();

fit_initialized = true;

}


void Burn::lambert_improve(long double prox_x_in, long double prox_y_in, long double prox_z_in)
{



long double x_error = prox_x_in - lambert_offset_x;
long double y_error = prox_y_in - lambert_offset_y;
long double z_error = prox_z_in - lambert_offset_z;

if (fit_lambert_y == false)
	{
	y_error = 0.0;
	}

long double sign = 1.0;

//if (abs(y_error) > abs(fit_last_y_error)) {sign = -1; cout << "Sign reversal!" << endl;}
//fit_last_y_error = y_error;

if (fit_log)
	{
	cout << "x error: " << x_error << endl;
	cout << "y error: " << y_error << endl;
	cout << "z error: " << z_error << endl;

	}


if ((abs(x_error) < 10.0 * fit_lambert_accept) && (abs(y_error) < 10.0 * fit_lambert_accept) && (abs(z_error) < 10.0 * fit_lambert_accept))
	{
	fit_converged = true;
	if (fit_log)
		{
		cout << "Fit converged!" << endl;
		}
	}

long double y_gain = 0.0012 * 1.0/sin(lambert_transfer_angle);



delta_v_prograde -= z_error / 6000.0 * fit_gain_prograde;
delta_v_radial -= x_error / 4000.0 * fit_gain_radial;
//delta_v_normal -=  -y_error /200.;

delta_v_normal -= y_gain * y_error;



if (num_iterations > num_precise_iterations) 
	{
	fit_lambert_accept *=1.2;
	fit_gain_prograde *=0.96;
	fit_gain_radial *=0.96;
	}
if ((fit_log) &&  (num_iterations == (num_precise_iterations + 1)))
	{
	cout << "Relaxing fit precision requirements now." << endl;
	}

if (fit_log)
	{
	cout << "New dvx: " << delta_v_prograde << endl;
	cout << "New dvy: " << delta_v_normal << endl;
	cout << "New dvz: " << delta_v_radial << endl;
	}

compute_vvectors();

burn_time = 0.0;
burn_active = false;
burn_finished = false;

num_iterations++;
}


long double Burn::get_acceleration ()
{
// Shuttle OMS dual engine burn

//return  12174.0 / 254600.0 * 9.81;

if ((burn_model == 2) && (craft_defined))
	{
	craft_acceleration = craft_thrust/(craft_propellant_mass + craft_dry_mass);
	}

//cout << "Acceleration is now " << craft_acceleration << endl;
//cout << "Thrust: " << craft_thrust << endl;
return craft_acceleration;

}

void Burn::set_acceleration (long double acc)
{

craft_acceleration = acc;
acceleration = craft_acceleration;
}


void Burn::set_plane_flag( bool flag)
{
force_plane = flag;
}


void Burn::set_lambert_target(long double set_x, long double set_y, long double set_z)
{
lambert_target_x = set_x + lambert_offset_x;
lambert_target_y = set_y + lambert_offset_y;
lambert_target_z = set_z + lambert_offset_z;


}

void Burn::update_low_thrust(long double nx, long double ny, long double nz)
{

if (force_plane == false) {return;}
	
long double n_dot_t = tgt_x * nx + tgt_y * ny + tgt_z * nz;

tgt_x -= nx * n_dot_t;
tgt_y -= ny * n_dot_t;
tgt_z -= nz * n_dot_t;


long double tgt_norm = sqrt(tgt_x * tgt_x + tgt_y * tgt_y + tgt_z * tgt_z);

tgt_x /= tgt_norm;
tgt_y /= tgt_norm;
tgt_z /= tgt_norm;


}


void Burn::set_spacecraft(long double mass_dry, long double mass_propellant, long double isp, long double thrust)
{

craft_dry_mass = mass_dry;
craft_propellant_mass = mass_propellant;
craft_isp = isp;
craft_thrust = thrust;
craft_mass_flow = craft_thrust / (9.81 * craft_isp);


//cout << "Dry mass: " << craft_dry_mass << " propellant mass: " << craft_propellant_mass << endl;
//cout << "ISP: " << craft_isp << " thrust: " << craft_thrust << endl;
//cout << "mass flow: " << craft_mass_flow << endl;

craft_defined = true;
}

void Burn::set_propellant(long double propellant_mass)
{
craft_propellant_mass = propellant_mass;
}

void Burn::set_dry_run (bool flag)
{
fit_lambert_dry_run = flag;
}


void Burn::set_y_option (bool flag)
{
fit_lambert_y = flag;
}

void Burn::set_lambert2_parameters(long double dvx, long double dvy, long double dvz)
{
lambert2_delta_v_prograde = dvx;
lambert2_delta_v_normal = dvy;
lambert2_delta_v_radial = dvz;

}


void Burn::do_timestep (long double timestep)
{
if (burn_active)
	{
	if (burn_model == 0)
		{
		delta_v_tot -= delta_v_impulse;
		delta_v_impulse /=timestep;
		}
	else if (burn_model == 1)
		{
		delta_v_tot -=  acceleration * timestep;

		burn_time += timestep;
		}
	else if (burn_model == 2)
		{
		craft_propellant_mass -= craft_mass_flow * timestep;
		acceleration = get_acceleration();
		delta_v_tot -= acceleration * timestep;
		burn_time += timestep;
		//cout << "Propellant mass is now: " << craft_propellant_mass << endl;
		}

	if ((delta_v_tot < 0.0) || (craft_propellant_mass < 0.0))
		{
		burn_active = false; 
		burn_finished = true;
		if ((fit_log) || (fit_converged) || (burn_type == 0))
			{
			if (burn_model == 0)
				{
				cout << "Burn "<< designation << " finished." << endl;
				}
			else if ((burn_model == 1) || (burn_model == 2))
				{
				cout << "Burn "<< designation << " finished, burn duration was " << burn_time << " seconds." << endl;
				if (burn_model == 2) 
					{	
					if (craft_propellant_mass < 0.0) {cout << "Out of propellant." << endl;} 
					//else {cout << "Remaining propellant: " << craft_propellant_mass << endl;}
					}
				}
			}
		}
	}
else
	{return;}
}

long double Burn::get_acc_prograde ()
{



if (burn_active)
	{
	if (burn_model == 0)
		{
		return unit_prograde * delta_v_impulse;
		}
	else
		{
		return unit_prograde * acceleration;
		}

	}
else
	{
	return 0.0;
	}
}

long double Burn::get_acc_radial ()
{

if (burn_active)
	{
	if (burn_model == 0)
		{
		return unit_radial * delta_v_impulse;
		}
	else
		{
		return unit_radial * acceleration;
		}

	}
else
	{
	return 0.0;
	}
}

long double Burn::get_acc_normal ()
{


if (burn_active)
	{
	if (burn_model == 0)
		{
		return unit_normal * delta_v_impulse;
		}
	else
		{
		return unit_normal * acceleration;
		}

	}
else
	{
	return 0.0;
	}
}


long double Burn::clamp(long double arg, long double limit)
{
if (arg > limit)
	{arg = limit;}
else if (arg < -limit)
	{arg = -limit;}
return arg;

}



void Burn::new_tig_acknowledge()
{
fit_new_tig = false;
}

int Burn::get_burn_type ()
{
return burn_type;
}

int Burn::get_num_iterations ()
{
return num_iterations;
}

long double Burn::get_H_target()
{
return peg4_H;
}


long double Burn::get_delta_v_prograde()
{
return delta_v_prograde;
}

long double Burn::get_delta_v_radial()
{
return delta_v_radial;
}

long double Burn::get_delta_v_normal()
{
return delta_v_normal;
}

long double Burn::get_delta_v_prograde2()
{
return lambert2_delta_v_prograde;
}

long double Burn::get_delta_v_radial2()
{
return lambert2_delta_v_radial;
}

long double Burn::get_delta_v_normal2()
{
return lambert2_delta_v_normal;
}


long double Burn::get_c1()
{
return peg4_c1;
}

long double Burn::get_c2()
{
return peg4_c2;
}

long double Burn::get_thetaT()
{
return peg4_thetaT;
}

long double Burn::get_H()
{
return peg4_H;
}


long double Burn::get_propellant()
{
return craft_propellant_mass;
}

long double Burn::get_tig()
{
return ignition_time;
}


long double Burn::get_tig2()
{
if (burn_type == 5)
	{
	return lambert_t2;
	}
else
	{
	cout << "The selected burn has no second ignition time - exiting..." << endl;
	exit(0);
	}
}


int Burn::get_strategy()
{
if (burn_finished)
	{
	return fit_strategy;
	}
else
	{
	return -1;
	}
}


int Burn::get_ops()
{
return peg4_ops;
}

std::string Burn::get_name()
{
return designation;
}

bool Burn::is_active (long double time)
{

if ((time > ignition_time) && (!burn_finished))
	{return true;}
else
	{return false;}

}

bool Burn::is_finished ()
{
return burn_finished;
}

bool Burn::tig_changed ()
{
return fit_new_tig;
}

bool Burn::is_initialized()
{
return fit_initialized;
}

bool Burn::is_converged ()
{
return fit_converged;
}


bool Burn::is_dry_run()
{
return fit_lambert_dry_run;
}

bool Burn::is_yfit_active()
{
return fit_lambert_y;
}


bool Burn::check_signflip(long double arg, long double lastarg)
{
if ((arg > 0.0) && (lastarg < 0.0))
	{return true;}
else if ((arg < 0.0) && (lastarg > 0.0))
	{return true;}
else
	{return false;}
}

void Burn::set_fit_log(bool flag)
{
fit_log = flag;
}

void Burn::start ()
{

if (burn_type == 0)
	{
	fit_initialized = true;
	}

if (delta_v_tot > 0.0)
	{
	burn_active = true;
	}
}


void Burn::start (long double vx, long double vy, long double vz)
{
if (burn_type == 3)
	{
	fit_initialized = true;
	}

long double res = vx * tgt_x + vy * tgt_y + vz * tgt_z;

if ((delta_v_tot > 0.0) && (res > tgt_threshold))
	{
	burn_active = true;
	}
else
	{
	burn_active = false;
	}

}


