// epidemic.cxx -- a simple program to simulate the spread of an epidemic
//
// Written by Thorsten Renk, started 2019
//
// Copyright (C) 2019  Thorsten Renk - thorsten@science-and-fiction.org 
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301


#include <iostream>
#include <fstream>
#include <string>
#include <sstream>
#include <cstdlib>
#include <math.h>

using namespace std;

#include "carrier.hxx"
#include "plot.hxx"
#include "options.hxx"
#include "measure.hxx"
#include "seed.hxx"

Options options;
Measure active_measure;
Measure *measures;

Seed *seeds;

#include "parser.cxx"

string commandline_parser (int argc, char *argv[]) 
{ 
string config_file;

if (argc != 2)
	{
	cout <<"Please specify a configuration file!" << endl;
	exit(0);
	}
else
	{
	config_file = argv[1];
	cout << "Reading " << config_file << " for configuration." << endl;
	}

return config_file;

}


double random(double min, double max)
{
double average = 0.5 * (max + min);
double variance = max - average;

return average + variance * 2.0 * ( rand()/(RAND_MAX +1.0) - 0.5);
}






int main (int argc, char *argv[] ) 
{
string config_file;

Carrier **carrier;
double *plot_array;
double **plot_array_gnu;

PlotTools plot;

config_file = commandline_parser (argc, argv);
file_parser(config_file);


if (options.random_seed_set) {srand(options.random_seed);}
else {srand(time(0));}

int timestep = 0;
int current_measure = -1;
int snapshot_counter = 0;
int grid_row = options.grid_size;
int grid_col = options.grid_size;


int max_mobility = 2;
double mobility_probability = 1.0;


// test declaration of active measure

//active_measure.init(300, 90, 4, 0.12, 1.0);



if (options.num_measures > 0)
	{
	active_measure = measures[0];
	current_measure = 0;
	}


// declare the carrier array

carrier = new Carrier *[grid_row];
plot_array = new double [grid_row * grid_col];
plot_array_gnu = new double *[grid_row];

for (int i=0; i< grid_row; i++)
	{
	carrier[i] = new Carrier [grid_col];
	plot_array_gnu[i] = new double [grid_col];
	}

int * timeline_x = new int[options.num_timesteps];
int * timeline_y = new int[options.num_timesteps];
int * timeline_y2 = new int[options.num_timesteps];
int * timeline_y3 = new int[options.num_timesteps];
int * timeline_y4 = new int[options.num_timesteps];
int * timeline_y5 = new int[options.num_timesteps];
double * growthline_x = new double[options.num_timesteps];
double * growthline_y = new double[options.num_timesteps];

// initialize the carrier array


for (int i=0; i< grid_row; i++)
	{
	for (int j=0; j< grid_col; j++)
		{
		//int actual_mobility = static_cast<int>(max_mobility * random(0,1));

		int class_id = 0;
		double rn = random(0,1);
		for (int k=0; k<options.num_classes;k++)
			{
			if (rn < options.intfraction[k])
				{
				class_id = k; break;
				}
			}


		int recovery_time = options.recovery_time + static_cast<int> (random(-1.0,1.0) * (options.recovery_time_uncertainty+1));
		int incubation_time = options.incubation_time +  static_cast<int> (random(-1.0,1.0) * (options.incubation_time_uncertainty+1));

		carrier[i][j].init(options.mobility[class_id], options.p_mobility[class_id], options.recovery_time, options.incubation_time, options.transmission_probability);

		if (random(0,1) < options.immune_fraction)
			{
			carrier[i][j].force_immune();
			}

		}
	}
	

for (int i=0; i<10; i++)
	{
	int seed_x = static_cast<int>(random(0,1) * grid_col);
	int seed_y = static_cast<int>(random(0,1) * grid_row);
	carrier[seed_x][seed_y].force_infected();
	}

// process time evolution



while (timestep < options.num_timesteps)
	{

	if (current_measure > -1) {active_measure.check(timestep);}
	if (active_measure.get_expired())
		{
		if (current_measure < options.num_measures -1)
			{
			current_measure++;
			active_measure = measures[current_measure];
			}
		}
	
	// do vaccination of active measure

	if (active_measure.get_active() && (active_measure.get_num_vaccinations() > 0))
		{
		//cout << "Doing " << active_measure.get_num_vaccinations() << " vaccinations" << endl;
		for (int i=0; i< active_measure.get_num_vaccinations(); i++)
			{
			int vacc_x = static_cast<int>(random(0,1) * grid_col);
			int vacc_y = static_cast<int>(random(0,1) * grid_row);
			carrier[vacc_x][vacc_y].vaccinate();
			}
		}

	// check for new seedings

	for (int i=0; i< options.num_seedings; i++)
		{


		if (seeds[i].check(timestep))
			{

			//cout << seeds[i].get_number() << endl;
			//cout << seeds[i].get_p_transmission() << endl;

			for (int j=0; j< seeds[i].get_number(); j++)
				{
				int seed_x = static_cast<int>(random(0,1) * grid_col);
				int seed_y = static_cast<int>(random(0,1) * grid_row);
				carrier[seed_x][seed_y].force_infected();
				carrier[seed_x][seed_y].set_p_transmission (seeds[i].get_p_transmission());
				carrier[seed_x][seed_y].set_tag_index (seeds[i].get_tag_index());
				//cout << seeds[i].get_p_transmission() << endl;
				}
			}

		}


	for (int i=0; i< grid_row; i++)
		{
		for (int j=0; j< grid_col; j++)
			{
			// process infected
			if (carrier[i][j].get_infected() == true)
				{
				carrier[i][j].progress_sickness();

				if (carrier[i][j].get_contagious() == true)
					{

					int mobility = carrier[i][j].get_mobility();

					if ((active_measure.get_active()) && (active_measure.get_mobility_restrict()))
						{
						if (mobility > active_measure.get_mobility())
							{
							mobility = active_measure.get_mobility();
							}
						}


					int move_x = static_cast<int>(random(-mobility, mobility));
					int move_y = static_cast<int>(random(-mobility, mobility));				
	


					int cand_x = j + move_x;
					int cand_y = i + move_y;

					if ((cand_x > 0) && (cand_x < grid_col) && (cand_y > 0) && (cand_y < grid_row))
						{
	
						double transmission_probability = carrier[i][j].get_transmission();
						double p_factor = 1.0;
				

						if ((active_measure.get_active()) && (active_measure.get_transmission_restrict()))
							{
							if (transmission_probability > active_measure.get_p_transmission())
								{transmission_probability = active_measure.get_p_transmission();}

							p_factor = active_measure.get_p_factor();

							}

						if (random(0.0,1.0) < transmission_probability * p_factor)
							{
							carrier[cand_y][cand_x].set_infected();
							carrier[cand_y][cand_x].set_p_transmission(carrier[i][j].get_transmission());
							carrier[cand_y][cand_x].set_tag_index(carrier[i][j].get_tag_index());
							}
						}
					}

				}
			}
		}
	

	int counter = 0;
	int counter_immune = 0;
	int counter_vaccinated = 0;
	int counter_vaccinated_prior = 0;


	for (int i=0; i< grid_row; i++)
		{
		for (int j=0; j< grid_col; j++)
			{
			if ((carrier[i][j].get_infected() == true) || ((carrier[i][j].get_immune() == true) && carrier[i][j].get_prior_vaccinated() == false))
				{counter++;}

			if ((carrier[i][j].get_immune() == true) && (carrier[i][j].get_prior_vaccinated() == false))
				{counter_immune++;}

			
			if (carrier[i][j].get_vaccinated() == true)
				{
				counter_vaccinated++;
				}

			if (carrier[i][j].get_prior_vaccinated() == true)
				{
				counter_vaccinated_prior++;
				}
			
			}
		}

	timeline_x[timestep] = timestep;
	timeline_y[timestep] = counter;	
	timeline_y2[timestep] = counter - counter_immune;
	timeline_y3[timestep] = counter_immune;
	timeline_y4[timestep] = counter_vaccinated;
	timeline_y5[timestep] = counter_vaccinated_prior;
	cout << "Time: " << timestep << " Infections: " << timeline_y[timestep] << endl;

	if (snapshot_counter == options.snapshot_interval)
		{
		stringstream ss;
		ss << timestep;

		for (int i=0; i< grid_row; i++)
			{
			for (int j=0; j< grid_col; j++)
				{
				if ((carrier[i][j].get_infected() == true) || ((carrier[i][j].get_immune() == true)&&(carrier[i][j].get_prior_vaccinated() == false)))
					{
					plot_array[i * grid_col + j] = carrier[i][j].get_tag_index();
					plot_array_gnu[i][j] = carrier[i][j].get_tag_index();
					}
				else if (carrier[i][j].get_prior_vaccinated() == true)
					{
					plot_array[i * grid_col + j] = -1.0;
					plot_array_gnu[i][j] = -1.0;
					}
				else
					{
					plot_array[i * grid_col + j] = 0.0;
					plot_array_gnu[i][j] = 0.0;
					}
				}
			}


		plot.plot_sgnu(plot_array_gnu, grid_row, grid_col, options.filename_base+"_2d"+ss.str()+".dat");
		snapshot_counter = 0;
		}

	snapshot_counter++;
	timestep++;
	}

	for (int i=0; i< grid_row; i++)
		{
		for (int j=0; j< grid_col; j++)
			{
			if ((carrier[i][j].get_infected() == true) || ((carrier[i][j].get_immune() == true) && (carrier[i][j].get_prior_vaccinated() == false)))
				{
				plot_array[i * grid_col + j] = carrier[i][j].get_tag_index();
				plot_array_gnu[i][j] = carrier[i][j].get_tag_index();
				}
			else if (carrier[i][j].get_prior_vaccinated() == true)
				{
				plot_array[i * grid_col + j] = -1.0;
				plot_array_gnu[i][j] = -1.0;
				}
			else
				{
				plot_array[i * grid_col + j] = 0.0;
				plot_array_gnu[i][j] = 0.0;
				}
			}
		}

	growthline_x[0] = 0.0;
	growthline_y[0] = 0.0;
	for (int i=1; i< options.num_timesteps; i++)
		{
		growthline_x[i] = static_cast <double> (timeline_x[i]);
		growthline_y[i] = (static_cast <double> (timeline_y[i]) / static_cast <double> (timeline_y[i-1]))-1.0;

		}


plot.plot4d (timeline_x, timeline_y, timeline_y2, timeline_y3, options.num_timesteps, options.filename_base + ".dat");
plot.plot2d (growthline_x, growthline_y, options.num_timesteps, options.filename_base + "_growth.dat");
if (options.includes_vaccination)
	{plot.plot3d (timeline_x, timeline_y4, timeline_y5, options.num_timesteps, options.filename_base + "_vaccination.dat");}
plot.plot_sgnu(plot_array_gnu, grid_row, grid_col, options.filename_base+"_2d.dat");





}
