// unit.cxx -- class to define a military unit
//
// Written by Thorsten Renk, started 2022
//
// Copyright (C) 2022 Thorsten Renk - thorsten@science-and-fiction.org 
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301

#include <math.h>
#include <cmath>
#include <iostream>

#include "unit.hxx"

using namespace std;

void Unit::init (string name_in, string type_in, double speed_in, double attack_in, double damage_in, double armor_in)
{
name = name_in;
type = type_in;
speed = speed_in;
attack = attack_in;
damage = damage_in;
armor = armor_in;
ranged_defense = attack_in;


ranged_attack = 0.0;
range = 0.0;
ranged_damage = 0.0;
shot_interval = 0.0;
projectile_drag = 0.1;

stamina = 1.0;
fraction_untouched = 1.0;
fraction_injured = 0.0;
fraction_incapacitated = 0.0;
fraction_dead = 0.0;
fraction_fighting = 1.0;

file_spacing = 1.5;
rank_spacing = 1.5;

current_order = "advance";
secondary_order = "none";
order_upon_contact = "attack";
order_upon_victory = "seek_enemy";
order_upon_position_reached = "halt";
order_upon_ammo_finished = "seek_enemy";
order_upon_target_destroyed = "fire_upon";

morale_status = "intact";
projectile_type = "none";

slow_speed = 0.25;
advance_speed = 0.5;
melee_speed = 0.65;
rush_speed = 1.0;
run_speed = 1.5;
next_speed = 0.0;

dist_to_engage = 30.0;
dist_to_disengage = 0.0;
dist_to_fire = 50.0;

current_speed = advance_speed;

num_contacts_last = 0;
num_contacts = 0;
num_friendly_contacts = 0;
num_shots = 0;
max_num_shots = 0;
tgt_id = -1;

range_flag = 0;

soldiers_per_mount = 0;

mount_mass = 0.0;
soldier_mass = 100.0;

penalty_flank_attack = 20.0;
penalty_rear_attack = 40.0;
penalty_without_formation = 0.0;

bonus_mounted_charge = 0.0;

contact_density = 0.0;
firing_timer = 0.0;
charge_veto_timer = 0.0;
under_fire_timer = 1.0;
speed_last = 0.0;
forced_motion = 0.0;


dir_x = 0.0;
dir_y = 1.0;
tgt_dir_x = 0.0;
tgt_dir_y = 1.0;

assembly_pos_x = 0.0;
assembly_pos_y = 0.0;

ang = 0.0;
tgt_ang = 0.0;


terrain_slope_along = 0.0;
terrain_slope_across = 0.0;
terrain_limit_speed = 10000.0;
terrain_slowdown_factor = 1.0;
terrain_surface_slowdown_factor = 1.0;
terrain_surface_limit_speed = 10000.0;

turn_rad_per_min = 3.1415926535; 

is_in_formation = false;
has_formation = false;
has_ranged = false;
has_mounts = false;
has_assembly = false;
is_inactive = false;
is_fled = false;
is_surrendered = false;
is_destroyed = false;

message_flag_morale = false;
flag_hit_and_run = false;
flag_charge_veto = false;

}


double Unit::rnd(double min, double max)
{
double average = 0.5 * (max + min);
double variance = max - average;

return average + variance * 2.0 * ( rand()/(RAND_MAX +1.0) - 0.5);
}

void Unit::assign_formation (int file_in, int rank_in)
{
num_files = file_in;
num_ranks = rank_in;
num_soldiers = num_files * num_ranks;

file_spacing = 1.0;
rank_spacing = 1.5;

size_file = num_files * file_spacing;
size_rank = num_ranks * rank_spacing;

is_in_formation = true;
has_formation = true;
penalty_without_formation = 20.0;

density = num_soldiers * soldier_mass / (size_file * size_rank);
push_density = soldier_mass * (0.75/file_spacing) * (1.0/rank_spacing) * pow(num_ranks,0.9);

equiv_num_close_combat = num_soldiers;
equiv_num_ranged_attack = num_soldiers;
equiv_num_ranged_damage = num_soldiers;

}

void Unit::assign_number (int num_in)
{
num_soldiers = num_in;

size_file = sqrt(num_soldiers) * file_spacing;
size_rank = sqrt(num_soldiers) * rank_spacing;

num_ranks = static_cast<int>(size_rank/ rank_spacing);

density = num_soldiers * 100.0 / (size_file * size_rank);
push_density = soldier_mass * (0.75/file_spacing) * (1.0/rank_spacing) *  pow(num_ranks,0.9); 

equiv_num_close_combat = num_soldiers;
equiv_num_ranged_attack = num_soldiers;
equiv_num_ranged_damage = num_soldiers; 
}

void Unit::set_file_spacing(double spacing)
{
file_spacing = spacing;
}

void Unit::set_rank_spacing(double spacing)
{
rank_spacing = spacing;
}

void Unit::set_spacings(double file, double rank)
{
file_spacing = file;
rank_spacing = rank;
}

void Unit::set_tag(std::string tag_in)
{
tag = tag_in;
}

void Unit::set_position(double x, double y)
{
pos_x = x;
pos_y = y;
}

void Unit::set_init_direction(double vx, double vy)
{
double norm;

dir_x = vx;
dir_y = vy;

norm = sqrt(vx*vx + vy*vy);

dir_x/=norm;
dir_y/=norm;

tgt_dir_x = dir_x;
tgt_dir_y = dir_y;

vec_to_ang();
//cout << "Unit: " << tag << "orientation tgt: " << tgt_ang * 180./3.1415926  << "orientation: " << ang * 180./3.1415926 << endl; 
}

void Unit::set_terrain_effects(double along, double across)
{
double terrain_slope_max;
double caloric_factor;



terrain_slope_along = along;
terrain_slope_across = across;

caloric_factor = 1.0 + 2.8 * terrain_slope_along;
if (caloric_factor < 0.75) {caloric_factor = 0.75;}
terrain_slowdown_factor = 1.0/ caloric_factor;
if (terrain_slowdown_factor > 1.5) {terrain_slowdown_factor = 1.5;}


terrain_slope_max = terrain_slope_along;
if (terrain_slope_max < 0.0) {terrain_slope_max = -terrain_slope_max;}

if (terrain_slope_across > terrain_slope_max) {terrain_slope_max = terrain_slope_across;}
else if (-terrain_slope_across > terrain_slope_max) {terrain_slope_max = -terrain_slope_across;}

terrain_limit_speed = 800.0;
if (terrain_slope_max > 0.5)
	{
	terrain_limit_speed = 800.0 - pow((terrain_slope_max - 0.5), 0.25) * 900.0;
	}

if (terrain_limit_speed < 1.0) {terrain_limit_speed = 1.0;}


}


void Unit::set_surface_effects(double fact, double max)
{
terrain_surface_slowdown_factor = fact;
terrain_surface_limit_speed = max;
}

void Unit::set_direction(double vx, double vy)
{
double norm;

norm = sqrt(vx*vx + vy*vy);

if (norm == 0.0) {cout << "Warning! Zero detected when setting directions!" << endl; return;}

tgt_dir_x = vx;
tgt_dir_y = vy;

tgt_dir_x/=norm;
tgt_dir_y/=norm;

vec_to_ang();

}

void Unit::set_morale(double morale_in)
{
morale = morale_in;
}

void Unit::set_parameter(string par, int value)
{
if (par == "tgt_id")
	{
	tgt_id = value;
	}
else if (par == "range_flag")
	{
	range_flag = value;
	}

}

void Unit::set_parameter(string par, double value)
{
if (par == "dist_to_engage")
	{
	dist_to_engage = value;
	}
else if (par == "dist_to_disengage")
	{
	dist_to_disengage = value;
	}
else if (par == "dist_to_fire")
	{
	dist_to_fire = value;
	}
else if (par == "ranged_defense")
	{
	ranged_defense = value;
	}
else if (par == "turn_rad_per_min")
	{
	turn_rad_per_min = value;
	}
else if (par == "penalty_flank_attack")
	{
	penalty_flank_attack = value;
	}
else if (par == "penalty_rear_attack")
	{
	penalty_rear_attack = value;
	}
else if (par == "bonus_mounted_charge")
	{
	bonus_mounted_charge = value;
	}
else if (par == "projectile_drag")
	{
	projectile_drag = value;
	}
}


void Unit::set_ranged_combat(double ranged_attack_in, double ranged_damage_in, double range_in, double shots_per_min_in, int num_shots_in, std::string ptype)
{
has_ranged = true;

ranged_attack = ranged_attack_in;
ranged_damage = ranged_damage_in;
range = range_in;
shot_interval = 1.0/shots_per_min_in;
firing_timer = shot_interval;
num_shots = num_shots_in;
max_num_shots = num_shots;
dist_to_fire = range;
projectile_type = ptype;
if ((projectile_type == "spear") || (projectile_type == "javelin")) {projectile_drag = 0.05;}
else if (projectile_type == "arrow") {projectile_drag = 0.1;}
else if (projectile_type == "bolt") {projectile_drag = 0.2;}
else if ((projectile_type == "lead") || (projectile_type == "bullet")) {projectile_drag = 0.3;}
else if (projectile_type == "stone") {projectile_drag = 0.4;}
}


void Unit::set_mounted_combat(int num, int equiv, double mass, string name)
{
double size_mount;

soldiers_per_mount = num;
mount_mass = mass;
mount_description = name;
has_mounts = true;

slow_speed = 0.1;
advance_speed = 0.2;
rush_speed = 0.4;
run_speed = 1.0;

file_spacing = mount_mass / 200.0;
rank_spacing = mount_mass / 200.0;

size_mount = 0.5 * pow(mount_mass,0.2);

size_file = sqrt(num_soldiers) * file_spacing;
size_rank = sqrt(num_soldiers) * rank_spacing;

density = num_soldiers * (soldier_mass + mount_mass/soldiers_per_mount) / (size_file * size_rank);

push_density = (soldier_mass * soldiers_per_mount + mount_mass) * (1.0/size_mount); 

equiv_num_close_combat = num_soldiers * equiv;
equiv_num_ranged_attack = num_soldiers * soldiers_per_mount;
equiv_num_ranged_damage = num_soldiers * static_cast<int>((soldier_mass * soldiers_per_mount + mount_mass) / 100.0);
}

void Unit::assign_to_army(int army)
{
army_id = army;
}

void Unit::register_contact(int id, double pos2_x, double pos2_y)
{
num_contacts ++;
contact_array[num_contacts-1] = id;

register_event("contact");

set_direction (pos2_x - pos_x, pos2_y - pos_y);
}


void Unit::register_friendly_contact(int id, double density)
{

num_friendly_contacts++;

contact_density+= density;

//cout << "Unit " << tag << " has " << num_friendly_contacts << " friendly contacts" << endl;
//cout << "Contact density " << contact_density << endl;

}

void Unit::register_event(std::string event)
{
if (event == "victory")
	{
	current_order = order_upon_victory;
	}
else if (event == "position_reached")
	{
	current_order = order_upon_position_reached;
	//cout << tag << ":position reached!" << endl;	
	}
else if (event == "contact")
	{
	current_order = order_upon_contact;
	}
else if (event == "contact_broken")
	{
	if (current_order != order_upon_victory)
		{
		current_order = order_upon_victory;
		//cout << "Unit " << tag << " is now free of contacts!" << endl;
		}
	}
else if ((event == "target_engage") || (event == "line_reached"))
	{
	current_order = "seek_enemy";
	current_speed = melee_speed;
	//if (event == "line_reached") {cout << tag << " has reached line " << endl;}
	}
else if (event == "out_of_ammo") 
	{
	//cout << "No ammunition left!" << endl;
	current_order = order_upon_ammo_finished;
	if (current_order == "seek_enemy") {current_speed = melee_speed;}
	else if (current_order == "retreat") 
		{
		current_order = "rush_to";
		order_upon_position_reached = "halt";
		tgt_pos_x = assembly_pos_x;
		tgt_pos_y = assembly_pos_y;
		secondary_order = "none";
		}
	}
else if (event == "target_destroyed")
	{
	if (secondary_order == "fire_at")
		{
		cout << "Unit " << tag  << ": Target destroyed" << endl;
		if (order_upon_target_destroyed == "fire_upon"){secondary_order = order_upon_target_destroyed;}
		else {current_order = order_upon_target_destroyed; secondary_order = "none";}
		}
	}
else if (event == "charge_successful")
	{
	//cout << "Unit " << tag << " charge successful!" << endl;
	flag_charge_veto = true;
	next_speed = rush_speed;
	current_order = "seek_enemy";
	secondary_order = "none";
	}
else if (event == "under_fire")
	{
	under_fire_timer = 0.0;
	}
else if (event == "charge_partial")
	{
	//cout << "Unit " << tag << " charge partially successful!" << endl;
	flag_charge_veto = true;
	next_speed = advance_speed;
	current_order = "seek_enemy";
	secondary_order = "none";
	}
else if (event == "retreat_to_assembly")
	{
	//cout << "Retreat to assembly" << endl;
	current_order = "rush_to";
	order_upon_position_reached = "halt";
	tgt_pos_x = assembly_pos_x;
	tgt_pos_y = assembly_pos_y;
	}
}

void Unit::set_initial_orders(std::string order, std::string uc, std::string uv, std::string up, std::string secondary)
{
current_order = order;
order_upon_contact = uc;
order_upon_victory = uv;
order_upon_position_reached = up;
secondary_order = secondary;

if (order == "halt") {current_speed = 0.0;}
else if (order == "rush") {current_speed = rush_speed;}
else if (order == "run") {current_speed = run_speed;}
else if (order == "advance") {current_speed = advance_speed;}
else if (order == "slow") {current_speed = slow_speed;}

if (has_mounts)
	{
	if (order == "canter") {current_speed = rush_speed;}
	else if (order == "charge") {current_speed = run_speed;}
	else if (order == "trot") {current_speed = advance_speed;}
	else if (order == "walk") {current_speed = slow_speed;}
	}

}

void Unit::set_initial_ranged_orders(std::string ua, std::string ut)
{
order_upon_ammo_finished = ua;
order_upon_target_destroyed = ut;

}

void Unit::set_initial_order_params(double val1, double val2)
{
if ((current_order == "slow_to") ||(current_order == "advance_to") || (current_order == "rush_to"))
	{

	//cout << "Hello - parameter setting to " << val1 << " " << val2 << endl;
	tgt_pos_x = val1;
	tgt_pos_y = val2;
	}
else if (current_order == "engage_upon")
	{
	dist_to_engage = val1;
	}
else if (current_order == "engage")
	{
	tgt_id = static_cast<int>(val1);
	}
}

void Unit::set_order(std::string condition, std::string order)
{
if (condition == "upon_contact") 
	{order_upon_contact = order;}
else if (condition == "upon_victory")
	{order_upon_victory = order;}
else if (condition == "upon_position_reached")
	{order_upon_position_reached = order;}
else if (condition == "upon_ammo_finished")
	{order_upon_ammo_finished = order;}
else if (condition == "upon_target_destroyed")
	{order_upon_target_destroyed = order;}
else if (condition == "secondary")
	{secondary_order = order;}
}

void Unit::set_assembly_point(double x, double y)
{
assembly_pos_x = x;
assembly_pos_y = y;
has_assembly = true;

}

void Unit::give_order(std::string order)
{


if (has_mounts)
	{
	if (order == "canter") {current_speed = rush_speed;  current_order = order;}
	else if (order == "charge") {current_speed = run_speed;  current_order = order; }
	else if (order == "trot") {current_speed = advance_speed;  current_order = order;}
	else if (order == "walk") {current_speed = slow_speed;  current_order = order;}
	}

if (order == "surrender") {is_surrendered = true; is_inactive = true; return;}


if (order == "halt") {current_speed = 0.0; current_order = order;}
else if (order == "rush") {current_speed = rush_speed; current_order = order;}
else if (order == "run") {current_speed = run_speed; current_order = order;}
else if (order == "advance") {current_speed = advance_speed; current_order = order;}
else if (order == "slow") {current_speed = slow_speed; current_order = order;}
else if ((order == "guard")||(order == "engage_upon")||(order == "fire_at") || (order == "fire_upon")) {secondary_order = order;}
else if ((order == "seek_enemy") && (current_speed == 0.0)) {current_speed = advance_speed; current_order = order; }
else if (order == "hold_fire") {secondary_order = order;}
else {current_order = order;}
}

void Unit::give_order(std::string order, double tgt_x, double tgt_y)
{

//cout << "Give order: " << tgt_x << " " << tgt_y << endl;

if (order == "hit_and_run")
	{
	dist_to_disengage = tgt_x;
	//cout << "Disengage: " << dist_to_disengage << endl;
	dist_to_engage = tgt_y;
	current_order =  order;
	}
else if ((order == "slow_for") || (order == "advance_for") || (order == "rush_for"))
	{
	tgt_pos_x = pos_x + sin(tgt_y * 0.017453) * tgt_x;
	tgt_pos_y = pos_y - cos(tgt_y * 0.017453) * tgt_x;
	
	if (order == "slow_for") {current_order = "slow_to";}
	else if (order == "advance_for") {current_order = "advance_to";}
	else {current_order = "rush_to";} 
	}
else
	{
	current_order = order;
	tgt_pos_x = tgt_x;
	tgt_pos_y = tgt_y;
	}
}

void Unit::give_order(std::string order, double dir)
{



if ((order == "slow") ||(order == "advance") || (order == "rush") || (order == "fall_back"))
	{
	tgt_dir_x = sin(dir);
	tgt_dir_y = -cos(dir);
	current_order = order;

	vec_to_ang();
	}
else if (has_mounts && ((order == "walk") ||(order == "trot") || (order == "canter") || (order == "charge")))
	{
	tgt_dir_x = sin(dir);
	tgt_dir_y = -cos(dir);
	current_order = order;

	//cout << "Order received: " << current_order << " to direction " << dir * 180./3.1415 << endl;
	vec_to_ang();
	}
else if (order == "engage_upon")
	{
	dist_to_engage = dir; 
	secondary_order = order;
	}
else if (order == "fire_upon")
	{
	dist_to_fire = dir;
	if  (num_shots == 0) {cout << "Unit " << tag << " has no ammunition left." << endl;}
	secondary_order = order;
	}
else if (order == "hit_and_retreat")
	{
	dist_to_disengage = dir;
	dist_to_engage = 2.0 * dir;
	current_order = order;
	}
else
	{
	current_order = order;
	}
}

void Unit::give_order(std::string order, int id)
{

tgt_id = id;

if (order == "fire_at")  
	{
	secondary_order = order;
	if (num_shots == 0) {cout << "Unit " << tag << " has no ammunition left." << endl;}
	}
else if (order == "engage")  
	{
	current_order = order;
	if (current_speed == 0.0) {current_speed = advance_speed;}
	}
else
	{
	current_order = order;
	}

}


void Unit::scatter()
{
is_in_formation = false;
}

void Unit::clear_contacts()
{
num_contacts_last = num_contacts;
num_contacts = 0;

num_friendly_contacts = 0;
contact_density = 0.0;
}



void Unit::check_contact_broken()
{
if ((num_contacts == 0) && (num_contacts_last > 0))
	{
	register_event("contact_broken");
	}
}

int Unit::get_army_id()
{
return army_id;
}

int Unit::get_tgt_id()
{
return tgt_id;
}

int Unit::get_num_contacts()
{
return num_contacts;
}


int Unit::get_num_shots()
{
return num_shots;
}

int Unit::get_max_num_shots()
{
return max_num_shots;
}



int Unit::get_nominal_num_soldiers()
{
return num_soldiers;
}

int Unit::get_current_num_soldiers()
{
return static_cast<int> (num_soldiers * (fraction_untouched + fraction_injured) );
}


int Unit::get_equiv_num_soldiers_cc()
{
return equiv_num_close_combat;
}

int Unit::get_equiv_num_soldiers_ra()
{
return equiv_num_ranged_attack;
}

int Unit::get_equiv_num_soldiers_rd()
{
return equiv_num_ranged_damage;
}

double Unit::get_speed()
{
return stamina * speed;
}

double Unit::get_current_speed()
{
return speed_last;
}

double Unit::get_push_density()
{
return (fraction_untouched + fraction_injured) * push_density;
}

double Unit::get_density()
{
return density;
}

double Unit::get_attack()
{
double strength;

strength = attack * stamina;
if ((has_formation == true) && (is_in_formation == false))
	{
	strength -=20.0;
	if (strength < 0.0) {strength = 0.0;}
	}

if ((has_mounts) && (current_speed == run_speed))
	{
	strength += bonus_mounted_charge;
	}

if (under_fire_timer < 1.0)
	{
	strength -= 10;
	if (strength < 0.0) {strength = 0.0;}
	}

return strength;
}

double Unit::get_damage()
{
return damage;
}

double Unit::get_armor()
{
return armor;
}

double Unit::get_pos_x()
{
return pos_x;
}

double Unit::get_pos_y()
{
return pos_y;
}

double Unit::get_dir_x()
{
return dir_x;
}

double Unit::get_dir_y()
{
return dir_y;
}

double Unit::get_dir()
{
return ang;
}

double Unit::get_contact_distance()
{
return 0.25 * (size_file + size_rank);
}

double Unit::get_contact_distance(double pos_x2, double pos_y2)
{
double norm, dir2x, dir2y, cos_ang, sin_ang;

norm  = sqrt((pos_x2 - pos_x) * (pos_x2 - pos_x) + (pos_y2 - pos_y) * (pos_y2 - pos_y));

if (norm == 0) {return 0.25 * (size_file + size_rank);}

dir2x = (pos_x2 - pos_x)/norm;
dir2y = (pos_y2 - pos_y)/norm;

cos_ang = (dir_x * dir2x + dir_y * dir2y);
sin_ang = sqrt(1.0 - cos_ang * cos_ang);

//cout << "Size file: " << size_file << " size rank: " << size_rank << endl;
//cout << "cos_ang: " << cos_ang << " sin_ang " << sin_ang << endl;

return sqrt(0.25 *size_rank * size_rank * cos_ang * cos_ang + 0.25 * size_file * size_file * sin_ang * sin_ang);
}

double Unit::get_dist_to_engage()
{
return dist_to_engage;
}

double Unit::get_dist_to_disengage()
{
return dist_to_disengage;
}

double Unit::get_dist_to_fire()
{
return dist_to_fire;
}

double Unit::get_health()
{
return 1.0 - (fraction_dead + fraction_incapacitated); 
}

double Unit::get_ranged_damage()
{
return ranged_damage;
}

double Unit::get_projectile_drag()
{
return projectile_drag;
}

double Unit::get_ranged_attack()
{
return ranged_attack;
}

double Unit::get_ranged_defense()
{
return ranged_defense;
}

double Unit::get_range()
{
return range;
}

double Unit::get_penalty_flank_attack()
{
return penalty_flank_attack;
}

double Unit::get_penalty_rear_attack()
{
return penalty_rear_attack;
}

double Unit::get_forced_motion()
{
return forced_motion;
}

std::string Unit::get_name()
{
return name;
}

std::string Unit::get_type()
{
return type;
}

std::string Unit::get_tag()
{
return tag;
}

std::string Unit::get_order()
{
return current_order;
}

std::string Unit::get_secondary_order()
{
return secondary_order;
}


std::string Unit::get_morale()
{
return morale_status;
}

bool Unit::get_inactive()
{
return is_inactive;
}

bool Unit::get_fled()
{
return is_fled;
}

bool Unit::get_mounted()
{
return has_mounts;
}

bool Unit::check_range()
{
if (range_flag == 0) {return false;}

return true;
}

bool Unit::get_flag(std::string flag)
{
if (flag == "morale") {return message_flag_morale;}
else if (flag == "hit_and_run") {return flag_hit_and_run;}
else if (flag == "charge_veto") {return flag_charge_veto;}

return false;
}

void Unit::set_flag(std::string flag, bool value)
{
if (flag == "morale") {message_flag_morale = value;}
else if (flag == "hit_and_run") {flag_hit_and_run = value;}
}


int Unit::get_ready_to_fire(double dt)
{

int i = 0;


if (has_ranged == false) {return 0;}
else if (num_shots == 0) {return 0;}
else if (num_contacts > 0) {return 0;}


firing_timer += dt;

//cout << "Firing timer is now: " << firing_timer << endl;


while (firing_timer > shot_interval)
	{
	i++;
	firing_timer -= shot_interval;
	}

num_shots -= i;

if (num_shots == 0) {register_event("out_of_ammo");}

//cout << "Firing " << i << " shots, " << num_shots << " remaining." << endl;

return i;

}


double Unit::normalize_angle_180(double ang_in)
{

while (ang_in < -3.1415926535)
	{
	ang_in += 2.0 *3.1415926535;
	}
while (ang_in > 3.1415926535)
	{
	ang_in -= 2.0 * 3.1415926535;
	}
return ang_in;
}

double Unit::normalize_angle_360 (double ang_in)
{

//cout <<  " Norm360 " << ang_in << endl; 
while (ang_in < 0.0)
	{
	ang_in += 2.0 * 3.1415926535;
	}

while (ang_in > 2.0 * 3.1415926535)
	{
	ang_in -= 2.0 * 3.1415926535;
	}

return ang_in;
}



void Unit::move (double t)
{
double delta_angle, corr_angle, factor, speed_factor;

if (is_inactive) {return;}

// update timers, here is where we receive timing information


if (flag_charge_veto)
	{
	charge_veto_timer += t;
	if (charge_veto_timer > 0.1)
		{
		//cout << "Unsetting charge veto timer" << endl;
		flag_charge_veto = false;
		charge_veto_timer = 0.0;
		current_speed = next_speed;
		}
	}

if (under_fire_timer < 1.0)
	{
	under_fire_timer += t;
	}
else if (under_fire_timer > 1.0)
	{
	}


// first turn the unit around

//cout << "Directions of " << tag << " " << dir_x << " " << dir_y << endl;
//cout << "Angles of " << tag << " " << tgt_ang << " " << ang << endl;
delta_angle = normalize_angle_180(tgt_ang - ang);

factor = 0.0;

if (num_contacts == 0) {factor = 1.0;}
else if (num_contacts == 1) {factor = 0.1;}

corr_angle = 0.0;

if (delta_angle > 0.0)
	{
	if (delta_angle > t * factor * turn_rad_per_min) {corr_angle = t * factor * turn_rad_per_min;}
	else {corr_angle = delta_angle;}
	}
else if (delta_angle < 0.0)
	{
	if (delta_angle < -t * factor * turn_rad_per_min) {corr_angle = -t * factor * turn_rad_per_min;}
	else {corr_angle = delta_angle;}
	}

ang += corr_angle;
ang = normalize_angle_360(ang);

ang_to_vec();


// slow down unit when moving through own units dependent on density

speed_factor = 1.0;

if (contact_density > 150.0)
	{speed_factor = 0.0;}
else if (contact_density > 100.0)
	{speed_factor *=0.1;}
else if (contact_density > 50.0)
	{speed_factor *= 0.2;}


if (num_contacts > 0) 
	{return;}





if (current_order == "halt") {current_speed = 0.0; return;}
else if ((current_order == "slow") || (current_order == "slow_to")) {current_speed = slow_speed;}
else if ((current_order == "advance") || (current_order == "advance_to")) {current_speed = advance_speed;}
else if ((current_order == "rush")|| (current_order == "rush_to")) {current_speed = rush_speed; }
else if ((current_order == "run")|| (current_order == "run_to")) {current_speed = run_speed; }
else if (current_order == "fall_back") {current_speed = -0.2;}

if (has_mounts)
	{
	if (current_order == "canter") {current_speed = rush_speed;}
	else if (current_order == "charge") {current_speed = run_speed;}
	else if (current_order == "trot") {current_speed = advance_speed;}
	else if (current_order == "walk") {current_speed = slow_speed; }
	}

speed_factor *= terrain_slowdown_factor * terrain_surface_slowdown_factor;

if (speed * current_speed * speed_factor > terrain_limit_speed)
	{
	speed_factor = terrain_limit_speed / (speed * current_speed);
	}

if (speed * current_speed * speed_factor > terrain_surface_limit_speed)
	{
	speed_factor = terrain_surface_limit_speed / (speed * current_speed);
	}





if ((current_order == "slow_to")||(current_order == "advance_to")|| (current_order == "rush_to"))	
	{
	if ((pos_x > 0.999 * tgt_pos_x) && (pos_x < 1.001 * tgt_pos_x) && (pos_y > 0.999 * tgt_pos_y) && (pos_y < 1.001 * tgt_pos_y))
		{
		register_event("position_reached");
		//cout << "Position reached" << endl;
		current_speed = melee_speed;
		return;
	
		}
	else
		{
		set_direction(tgt_pos_x - pos_x, tgt_pos_y - pos_y);
		double dist = sqrt( (tgt_pos_x - pos_x) * (tgt_pos_x - pos_x) + (tgt_pos_y - pos_y) * (tgt_pos_y - pos_y));
		//cout << "Dist: " << dist << endl;
		if ((speed * current_speed * speed_factor * t) > dist)
			{current_speed = dist / (t* speed * speed_factor);}
		}

	}

//cout << tag << endl;
//cout << "Speed factor " << speed_factor << " t: " << t << endl;
//cout << "Speed: " << speed << " Current: " << current_speed << endl;

speed_last = speed * current_speed * speed_factor;

pos_x += dir_x * speed * current_speed * speed_factor * t;
pos_y += dir_y * speed * current_speed * speed_factor * t;

//cout << "Directions of " << tag << " " << dir_x << " " << dir_y << " ang: " << ang << endl;

return;
}

void Unit::contact_move(double t, double dir_x_f, double dir_y_f, double speed_f)
{


speed_last = 0.0;

forced_motion = speed_f * t;

pos_x += dir_x_f * speed_f * t;
pos_y += dir_y_f * speed_f * t;


}

void Unit::do_damage (double l, double h, double d)
{
double fraction_counter;

fraction_counter = 1.0;

fraction_dead += d;
if (fraction_dead > fraction_counter) {fraction_dead = fraction_counter;}
fraction_counter -= fraction_dead;

fraction_incapacitated += h;
if (fraction_incapacitated > fraction_counter) {fraction_incapacitated = fraction_counter;}
fraction_counter -= fraction_incapacitated;

fraction_injured += l;
if (fraction_injured > fraction_counter) {fraction_injured = fraction_counter;}
fraction_counter -= fraction_injured;

fraction_untouched = fraction_counter;

fraction_fighting = fraction_untouched + fraction_injured;

if (fraction_dead > 0.5)
	{
	is_destroyed = true;
	is_inactive = true;
	}
} 

void Unit::check_morale(double difficulty)
{
double value;

//cout << "Unit " << name << " morale check with difficulty " << difficulty << " current " << morale << endl;

value = rnd(0.0, 10.0) + difficulty;

if (value > (morale + 4.0))
	{
	morale_status = "destroyed";
	is_fled = true;
	is_inactive = true;
	//cout << "Failed" << endl;
	}

else if (value > morale)
	{
	if (morale_status == "intact")
		{
		morale_status = "shaken";
		if ((current_order == "assault") || (current_order == "attack") || (current_order == "hold_line")) 
			{
			current_order = "yield";
			order_upon_contact = "yield";
			}
		}
	else if (morale_status == "shaken")
		{
		morale_status = "destroyed";
		is_fled = true;
		is_inactive = true;
		}
	//cout << "Failed" << endl;
	}
}

void Unit::vec_to_ang()
{
ang = normalize_angle_360(atan2(dir_x, -dir_y));
tgt_ang = normalize_angle_360(atan2(tgt_dir_x, -tgt_dir_y));
}

void Unit::ang_to_vec()
{
dir_x = sin(ang);
dir_y = -cos(ang);

tgt_dir_x = sin(tgt_ang);
tgt_dir_y = -cos(tgt_ang);
}


void Unit::list()
{
cout << name << " of type " << type << endl; 
cout << num_soldiers << " soldiers";
if (num_files > 0)
	{cout << " in " << num_ranks << " ranks and " << num_files << " files" << endl;}
else
	{cout << endl;}
cout << "serving in army:      " << army_id << endl;
cout << "Movement speed:       " << speed << endl;
cout << "Fighting strength:    " << attack << endl;
cout << "Weapons damage:       " << damage << endl;
cout << "Armor strength:       " << armor << endl;
if (has_ranged)
	{
	cout << "Ranged strength:      " << ranged_attack << endl;
	cout << "Ranged damage:        " << ranged_damage << endl;
	cout << "Weapons range:        " << range << endl;
	cout << "Dist. to fire:        " << dist_to_fire << endl;
	cout << "Firing interval:      " << shot_interval << endl;
	cout << "Number of shots:      " << num_shots << endl;
	}
if (has_mounts)
	{
	cout << "Mounted on:           " << mount_description << endl;
	}
}

void Unit::list_damage()
{
cout << name << " status report:" << endl;
cout << "Untouched:            " << fraction_untouched << endl;
cout << "Injured:              " << fraction_injured << endl;
cout << "Incapacitated:        " << fraction_incapacitated << endl;
cout << "Dead:                 " << fraction_dead << endl;
if (is_fled)
	{cout << "Unit has fled the battlefield." << endl;}
}

void Unit::report_orders()
{
cout << "Current order:   " << current_order << endl;
cout << "Secondary order: " << secondary_order << endl;
cout << "Upon contact:    " << order_upon_contact << endl;
cout << "Upon victory:    " << order_upon_victory << endl;
cout << "Upon position:   " << order_upon_position_reached << endl;
if ((secondary_order == "guard") || (secondary_order == "engage_upon") || (current_order == "hit_and_run")||(current_order == "hit_and_retreat") )
	{
	cout << "Dist. to engage: " << dist_to_engage << endl;
	}
if ((current_order == "hit_and_run")|| (current_order == "hit_and_retreat"))
	{
	cout << "   to disengage: " << dist_to_disengage << endl;
	}
if ((secondary_order == "fire_at") || (secondary_order == "fire_upon"))
	{
	cout << "Dist. to fire  : " << dist_to_fire << endl;
	} 

}

void Unit::report_position()
{
cout << name << " position report:" << endl;
cout << "Pos x:       " << pos_x << endl;
cout << "Pos y:       " << pos_y << endl;
}

void Unit::report_internal()
{
cout << "Current speed: " << current_speed << endl;
cout << "Contacts:      " << num_contacts << endl;
cout << "Density:       " << density << endl;
cout << "Push density:  " << push_density << endl;
cout << "Ang:           " << ang * 180.0/3.1415926535 << endl;
cout << "Ang_tgt:       " << tgt_ang * 180.0/3.1415926535 << endl;
cout << "vx:            " << dir_x << endl;
cout << "vy:            " << dir_y << endl;
cout << "vx_target:     " << tgt_dir_x << endl;
cout << "vy_target:     " << tgt_dir_y << endl;
cout << "pos_x:	        " << pos_x << endl;
cout << "pos_y:	        " << pos_y << endl;
if (tgt_id != -1)
	{cout << "tgt_id:        " << tgt_id << endl;}
if (has_assembly)
	{
	cout << "assembly_x:    " << assembly_pos_x << endl;
	cout << "assembly_y:    " << assembly_pos_y << endl;
	}
}

