// terrain.hxx -- provides altitudes, slopes and terrain type for battle simulation
//
// Written by Thorsten Renk, started 2022
//
// Copyright (C) 2022  Thorsten Renk - thorsten@science-and-fiction.org 
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301

#ifndef TERRAIN_H
#define TERRAIN_H

#include "plot.hxx"
#include "unit.hxx"
#include "formation.hxx"

using namespace std;


class TerrainSurface {

	double movement_slowdown_factor;
	double movement_max;
	double soft_cover;
	double hard_cover;
	string symbol;


	public:

	friend class Terrain;

	TerrainSurface();
};


class Terrain {

	double **alt_grid;
	double **slope_grid;
	TerrainSurface **surface_grid;
	TerrainSurface *surface_atlas;

	double *elev_point_x;
	double *elev_point_y;
	double *elev_point_value;

	double grid_scale;
	double grid_size_x;
	double grid_size_y;
	double interpolation_smoothing;
	double random_overlay;
	double random_jitter;
	double alt_multiplier;
	double slope_resolution;

	int grid_nx;
	int grid_ny;
	int surface_grid_nx;
	int surface_grid_ny;
	int num_ipoints;

	bool elevation_map_generated;
	bool surface_map_generated;
	
	PlotTools plot;

	void read_elevation_points(std::string);
	void read_surface_atlas(std::string);
	void read_surface_map(std::string, int, int);

	void interpolate_to_grid(void);
	
	double distance(double x1, double y1, double x2, double y2);
	double rnd(double min, double max);

	public:


	Terrain();
	Terrain(int nx, int ny, double res);
	
	double get_elevation(double x, double y);
	double get_slope(double x, double y, double dir);
	double get_slope(double x, double y);

	bool has_elevation_map();

	void generate_elevation_map(std::string filename);
	void generate_slope_map(void);
	void save_grid(std::string filename);
	void load_grid(std::string filename);
	void plot_elevation_map(std::string filename);
	void plot_positions(Unit *unit_list, Formation form);

	string get_surface_symbol(double, double);
	double get_movement_slowdown(double, double);
	double get_movement_max(double, double);
	double get_soft_cover(double, double);
	double get_hard_cover(double, double);

	void generate_surface_map(std::string, std::string, int, int);
	void set_surface_grid_size(double, double);
	void add_to_surface_atlas(int, string, double, double, double, double);
	
	bool has_surface_map();

};



#endif
