// terrain.cxx -- provides altitudes, slopes and terrain type for battle simulation
//
// Written by Thorsten Renk, started 2022
//
// Copyright (C) 2022  Thorsten Renk - thorsten@science-and-fiction.org 
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301

#include <stdlib.h>
#include <iostream>
#include <fstream>
#include <string>
#include <math.h>

#include "terrain.hxx"


TerrainSurface::TerrainSurface()
{
movement_slowdown_factor = 1.0;
movement_max = 1e6;
soft_cover = 0.0;
hard_cover = 0.0;
symbol = ".";
}



Terrain::Terrain()
{
elevation_map_generated = false;
surface_map_generated = false;

cout << "Terrain empty init." << endl;
}

Terrain::Terrain(int nx, int ny, double scale)
{

grid_nx = nx;
grid_ny = ny;
grid_scale = scale;
grid_size_x = nx * scale;
grid_size_y = ny * scale;

interpolation_smoothing = 4.0;
alt_multiplier = 1.0;
random_overlay = 0.0;
random_jitter = 0.0;
slope_resolution = 20.0;

elevation_map_generated = false;
surface_map_generated = false;

// dynamical memory allocation for arrays

alt_grid = new double *[nx];

for (int i=0; i< nx; i++)
	{
	alt_grid[i] = new double [ny];
	}

// initialize altitude grid

for (int i=0; i< nx; i++)
	{
	for (int j=0; j< ny; j++)
		{
		alt_grid[i][j] = 0.0;
		}
	}



}


double Terrain::rnd(double min, double max)
{
double average = 0.5 * (max + min);
double variance = max - average;

return average + variance * 2.0 * ( rand()/(RAND_MAX +1.0) - 0.5);
}



void Terrain::generate_elevation_map(std::string filename)
{
read_elevation_points(filename);
interpolate_to_grid();

generate_slope_map();
}

void Terrain::plot_elevation_map(std::string filename)
{
plot.plot_sgnu(alt_grid, grid_scale, grid_nx, grid_ny, filename); 

}


void Terrain::plot_positions(Unit *unit_list, Formation form)
{
int num_points, id;
double *x, *y, *z;


num_points = form.get_num_units();

x = new double[num_points];
y = new double[num_points];
z = new double[num_points];

for (int i=0; i< num_points; i++)
	{
	id = form.get_unit_id(i);
	x[i] = unit_list[id].get_pos_x();
	y[i] = unit_list[id].get_pos_y();
	z[i] = get_elevation(x[i], y[i]);
	}

plot.plot3d(y, x, z, num_points, form.get_name()+".dat");

}


double Terrain::distance(double x1, double y1, double x2, double y2)
{
double delta_x, delta_y;

delta_x = x2 - x1;
delta_y = y2 - y1;

return sqrt(delta_x * delta_x + delta_y * delta_y);

}

void Terrain::interpolate_to_grid(void)
{
double gx, gy, ix, iy;
double sum_value, sum_weight;
double d;

for (int i=0; i< grid_nx; i++)
	{
	for (int j=0; j< grid_ny; j++)
		{
		sum_value = 0.0;
		sum_weight = 0.0;
		gx = i * grid_scale;
		gy = j * grid_scale;

		for (int k=0; k< num_ipoints; k++)
			{

			ix = elev_point_x[k] * grid_size_x;
			iy = elev_point_y[k] * grid_size_y;
			d = distance (gx, gy, ix, iy);
			if (d< grid_scale) {d = grid_scale;} 
			sum_value += alt_multiplier * elev_point_value[k] / pow(d, interpolation_smoothing);
			sum_weight += 1.0/pow(d,interpolation_smoothing);
			}
		alt_grid[i][j] = sum_value/sum_weight;
		alt_grid[i][j] += rnd(-0.5 * random_overlay, 0.5 * random_overlay);
		}
	}


}

void Terrain::generate_slope_map(void)
{
double grad_1, grad_2;


slope_grid = new double *[grid_nx];

for (int i=0; i< grid_nx; i++)
	{
	slope_grid[i] = new double [grid_ny];
	}

for (int i=0; i< (grid_nx-1); i++)
	{
	for (int j=0; j< (grid_ny-1); j++)
		{
		grad_1 = abs(alt_grid[i+1][j] - alt_grid[i][j]);
		grad_2 = abs(alt_grid[i][j+1] - alt_grid[i][j]);

		if (grad_1 > grad_2)
			{slope_grid[i][j] = grad_1 / grid_scale;}
		else 
			{slope_grid[i][j] = grad_2 / grid_scale;}
		}
	}

for (int i=0; i< grid_nx; i++)
	{
	slope_grid[i][grid_ny - 1] = slope_grid[i][grid_ny-2];
	}

for (int i=0; i< grid_ny; i++)
	{
	slope_grid[grid_nx-1][i] = slope_grid[grid_nx - 2][i];
	}

//plot.plot_sgnu(slope_grid, grid_scale, grid_nx, grid_ny, "slope.dat"); 

}

void Terrain::save_grid(std::string filename)
{
ofstream resfile (filename.c_str());
if (resfile.is_open())
	{
  	for (int i=0; i< grid_nx; i++)
		{
  		for (int j=0; j< grid_ny; j++)
			{      
			resfile << alt_grid[i][j] << endl;
			}
		}
	resfile.close();
    	cout << "Elevation grid written to " << filename << "." << endl;
	}
else 
	{cout << "Unable to open file " << filename << " !" << endl;}

}

void Terrain::load_grid(std::string filename)
{
ifstream resfile (filename.c_str());
if (resfile.is_open())
	{
  	for (int i=0; i< grid_nx; i++)
		{
  		for (int j=0; j< grid_ny; j++)
			{
			resfile >> alt_grid[i][j];
			}
		}
	resfile.close();
    	cout << "Elevation grid loaded. " << endl;
	generate_slope_map();
	}
else 
	{cout << "Unable to open file " << filename << " !" << endl;}

}


void Terrain::read_elevation_points(std::string filename)
{
int counter, n_points;

std::string element;


element = "";
counter = 0;
n_points = 0;


ifstream elev_file (filename.c_str());

if (elev_file.is_open())
	{
    	while (element !="end")
	    	{
	     	elev_file >> element;

		if (element == "smoothing") {counter = 4;}
		else if (element == "random") {counter = 4;}
		else if (element == "jitter") {counter = 4;}
		else if (element == "multiply") {counter = 4;}
		if (element == "end") {counter = 5;}


		if (counter == 0)
			{
			counter ++;
			}
		else if (counter == 1)
			{
			counter ++;
			}
		else if (counter == 2)
			{
			n_points++;
			counter = 0;
			}
		else if ((counter == 4) && (element != "smoothing") && (element != "random") && (element != "jitter") && (element != "multiply"))
	      		{
			counter = 0;
			}
	
	    	}

	//cout << n_points << " points found for interpolation, proceeding..." << endl;

	elev_point_x = new double [n_points];	
	elev_point_y = new double [n_points];	
	elev_point_value = new double [n_points];	

	element = "";
	n_points = 0;	
	counter = 0;
	elev_file.seekg( 0, elev_file.beg);

	while (element !="end")
	    	{
	     	elev_file >> element;
		if (element == "smoothing") {counter = 4;}
		if (element == "random") {counter = 5;}
		if (element == "jitter") {counter = 6;}
		if (element == "multiply") {counter = 7;}
		if (element == "end") {counter = 10;}

		if (counter == 0)
			{
	 		elev_point_x[n_points] = atof (element.c_str());
			counter ++;
			}
		else if (counter == 1)
			{
			elev_point_y[n_points] = atof (element.c_str());
			counter ++;
			}
		else if (counter == 2)
			{
	 		elev_point_value[n_points] =  atof (element.c_str()) + rnd(-0.5 * random_jitter, 0.5 * random_jitter);
			n_points++;
			counter = 0;
			}
		else if ((counter == 4) && (element != "smoothing"))
	      		{
			counter = 0;
			interpolation_smoothing = atof (element.c_str());
			}
		else if ((counter == 5) && (element != "random"))
	      		{
			counter = 0;
			random_overlay = atof (element.c_str());
			}
		else if ((counter == 6) && (element != "jitter"))
	      		{
			counter = 0;
			random_jitter = atof (element.c_str());
			}
		else if ((counter == 7) && (element != "multiply"))
	      		{
			counter = 0;
			alt_multiplier = atof (element.c_str());
			}
	      
	
	    	}
	elev_file.close(); 

	if (n_points == 0)
		{
		cout << "Reading elevation file " << filename << " has failed, aborting." << endl;
		exit(0);
		}
	else
		{
		num_ipoints = n_points;
		interpolate_to_grid();
		elevation_map_generated = true;
		cout << "Read " << n_points << " elevation interpolation points from file " << filename << endl;
		}
	}
else
	{
	cout << "Reading elevation file " << filename << " has failed, aborting." << endl;
	exit(0);
	}



}

double Terrain::get_elevation(double x, double y)
{
int i_min, j_min, i, j;

i_min = 0;
j_min = 0;

if (x > 0.5 * grid_size_x) {i_min = static_cast <int> (0.5 * grid_nx);}
if (y > 0.5 * grid_size_y) {j_min = static_cast <int> (0.5 * grid_ny);}

for (i=i_min; i< (grid_nx-1); i++)
	{
	if ((x >= i * grid_scale) && (x < (i+1) * grid_scale)) {break;}
	}

for (j=j_min; j< (grid_ny-1); j++)
	{
	if ((y >= j * grid_scale) && (y < (j+1) * grid_scale)) {break;}
	}

return alt_grid[i][j];

}

double Terrain::get_slope(double x, double y, double dir)
{
double x1, y1;



x1 = x + sin(dir) * slope_resolution;
y1 = y - cos(dir) * slope_resolution;


return (get_elevation(x1,y1) - get_elevation(x,y))/slope_resolution;
}


double Terrain::get_slope(double x, double y)
{
int i, j;

i = static_cast<int>(x/grid_scale);
j = static_cast<int>(y/grid_scale);

//cout << i << " " << j << endl;

return slope_grid[i][j];
}


void Terrain::generate_surface_map(string atlas_filename, string map_filename, int nx, int ny)
{


// dynamical memory allocation for arrays

surface_grid = new TerrainSurface *[nx];

for (int i=0; i< nx; i++)
	{
	surface_grid[i] = new TerrainSurface [ny];
	}

surface_grid_nx = nx;
surface_grid_ny = ny;

//surface_grid[1][1].symbol = "#";
//surface_grid[2][2].symbol = ":";
//surface_grid[3][3].symbol = "*";

surface_map_generated = true;

read_surface_atlas(atlas_filename);
read_surface_map(map_filename, nx, ny);
}


void Terrain::add_to_surface_atlas(int index, string symbol, double move_fact, double move_max, double soft, double hard)
{
//cout << "Adding index " << index << " symbol: " << symbol << endl;

surface_atlas[index].symbol = symbol;
surface_atlas[index].movement_slowdown_factor = move_fact;
surface_atlas[index].movement_max = move_max;
surface_atlas[index].soft_cover = soft;
surface_atlas[index].hard_cover = hard;
}


void Terrain::read_surface_atlas(string filename)
{
string element, name, key, symbol;

double value;
double hard_cover, soft_cover, move_fact, move_max;

int key_flag = 0; // initially expect a keyword
int surface_counter = -1;
int n_surface, index;

bool index_set;

name ="";
symbol = ".";


// preparse to see memory requirement

n_surface = 0;
index = -1;
index_set = false;

  ifstream cfgfile (filename.c_str());
  if (cfgfile.is_open())
  {
    while (element !="end")
    {
      cfgfile >> element;
	if (element == "surface") {n_surface++;}
    }
  }


cout << "Found " << n_surface << " surface definitions in " << filename << "." << endl;

surface_atlas =  new TerrainSurface[n_surface];

element = "";
cfgfile.seekg( 0, cfgfile.beg);


  //ifstream cfgfile (filename.c_str());
  if (cfgfile.is_open())
  {
    while (element !="end")
    {
      cfgfile >> element;
      //cout << element << " " << key_flag << '\n';

      	if (key_flag == 1)
		{
		value = atof (element.c_str());
		}

   	if (key_flag == 0)
		{
		//cout << "parsing key: " << element << endl;
		if ((element == "surface")  || (element == "end"))
			{
			surface_counter++; 
			if (index_set == false) {index = surface_counter;}
			add_to_surface_atlas(index, symbol, move_fact, move_max, soft_cover, hard_cover);
			}

		else if (element == "name") {key = element; key_flag = 2;}
		else if (element == "symbol") {key = element; key_flag = 2;}
		else {key = element; key_flag = 1;}
		}
	else if (key_flag == 1)
		{
		if (key == "soft_cover") {soft_cover = value;}
		else if (key == "hard_cover") {hard_cover = value;}
		else if (key == "movement_slowdown") {move_fact = value;}
		else if (key == "movement_max") {move_max = value;}
		else if (key == "index") {index = static_cast <int>(value); index_set = true;}
		key_flag = 0;
		}
	else if (key_flag == 2)
		{
		if (key == "name") {name = element;  }
		else if (key == "symbol") {symbol = element;  }
		key_flag = 0;
		}
    }
    cfgfile.close();
  }
  else 
	{
	cout << "Unable to open surface atlas file " << filename << " !" << endl;
	exit(0);
	}


}



void Terrain::read_surface_map(string filename, int nx, int ny)
{

int i,j;
int index;

ifstream mapfile (filename.c_str());
if (mapfile.is_open())
	{
	for (i=0; i< ny; i++)
		{
		for (j=0; j < nx; j++)
			{
      			mapfile >> index;
			//if (index == 1) {cout << "Index 1" << surface_atlas[index].symbol << endl;}
			surface_grid[j][i].symbol = surface_atlas[index].symbol;
			surface_grid[j][i].movement_slowdown_factor = surface_atlas[index].movement_slowdown_factor;
			surface_grid[j][i].movement_max = surface_atlas[index].movement_max;
			surface_grid[j][i].hard_cover = surface_atlas[index].hard_cover;
			surface_grid[j][i].soft_cover = surface_atlas[index].soft_cover;
			}

		}
	 mapfile.close();
}
else 
	{
	cout << "Unable to open surface map file " << filename << " !" << endl; 
	exit(0);
	}


}




string Terrain::get_surface_symbol(double x, double y)
{
int i,j;

i = static_cast<int>(x/grid_size_x * surface_grid_nx);
j = static_cast<int>(y/grid_size_y * surface_grid_ny);

return surface_grid[i][j].symbol;
}


double Terrain::get_movement_slowdown(double x, double y)
{
int i,j;

if ((x < 0) || (x > grid_size_x)) {return 1.0;}
if ((y < 0) || (y > grid_size_y)) {return 1.0;}

i = static_cast<int>(x/grid_size_x * surface_grid_nx);
j = static_cast<int>(y/grid_size_y * surface_grid_ny);

return surface_grid[i][j].movement_slowdown_factor;
}

double Terrain::get_movement_max(double x, double y)
{
int i,j;

if ((x < 0) || (x > grid_size_x)) {return 1e6;}
if ((y < 0) || (y > grid_size_y)) {return 1e6;}

i = static_cast<int>(x/grid_size_x * surface_grid_nx);
j = static_cast<int>(y/grid_size_y * surface_grid_ny);

return surface_grid[i][j].movement_max;
}

double Terrain::get_hard_cover(double x, double y)
{
int i,j;

if ((x < 0) || (x > grid_size_x)) {return 0.0;}
if ((y < 0) || (y > grid_size_y)) {return 0.0;}

i = static_cast<int>(x/grid_size_x * surface_grid_nx);
j = static_cast<int>(y/grid_size_y * surface_grid_ny);

return surface_grid[i][j].hard_cover;
}

double Terrain::get_soft_cover(double x, double y)
{
int i,j;

if ((x < 0) || (x > grid_size_x)) {return 0.0;}
if ((y < 0) || (y > grid_size_y)) {return 0.0;}

i = static_cast<int>(x/grid_size_x * surface_grid_nx);
j = static_cast<int>(y/grid_size_y * surface_grid_ny);

return surface_grid[i][j].soft_cover;
}


void Terrain::set_surface_grid_size(double size_x, double size_y)
{
grid_size_x = size_x;
grid_size_y = size_y;
}


bool Terrain::has_elevation_map()
{
return elevation_map_generated;
}

bool Terrain::has_surface_map()
{
return surface_map_generated;
}
