// events.cxx -- provides event management for battle simulation
//
// Written by Thorsten Renk, started 2022
//
// Copyright (C) 2023  Thorsten Renk - thorsten@science-and-fiction.org 
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301

#include <stdlib.h>
#include <iostream>
#include <string>
#include <sstream>
#include <math.h>

#include "events.hxx"

using namespace std;

/* event trigger code ==================================================================== */

EventTrigger::EventTrigger()
{
trigger_condition = false;
check_formation_destroyed = false;
check_unit_destroyed = false;
check_line_crossed = false;
check_unit_engaged = false;
logic = "AND";
}


void EventTrigger::initialize(int n_fd, int n_ud, int n_lc, int n_ue)
{
i_formation_destroyed = 0;
n_formation_destroyed = n_fd;
fd_array = new string[n_fd];
fd_percentage_array = new double[n_fd];
formation_destroyed = new bool[n_fd];

i_unit_destroyed = 0;
n_unit_destroyed = n_ud;
ud_array = new string[n_ud];
ud_percentage_array = new double[n_ud];
unit_destroyed = new bool[n_ud];

i_line_crossed = 0;
n_line_crossed = n_lc;
lc_array = new string[n_lc];
lc_sign_array = new int[n_lc];
line_crossed = new bool[n_lc];
lc_pos_array = new double[n_lc];

i_unit_engaged = 0;
n_unit_engaged = n_ue;
ue_array = new string[n_ue];
unit_engaged = new bool[n_ue];
}

void EventTrigger::set_formation_destroyed(string name)
{
//cout << "i is now: " << i_formation_destroyed << endl;
fd_array[i_formation_destroyed] = name;
fd_percentage_array[i_formation_destroyed] = 1.0;
formation_destroyed[i_formation_destroyed] = false;
i_formation_destroyed++;
check_formation_destroyed = true;
}


void EventTrigger::set_unit_destroyed(string name)
{

ud_array[i_unit_destroyed] = name;
ud_percentage_array[i_unit_destroyed] = 1.0;
unit_destroyed[i_unit_destroyed] = false;
i_unit_destroyed++;
check_unit_destroyed = true;
}

void EventTrigger::set_line_crossed(string name, double pos, int sign)
{
lc_array[i_line_crossed] = name;
lc_pos_array[i_line_crossed] = pos;
lc_sign_array[i_line_crossed] = sign;
line_crossed[i_line_crossed] = false;
i_line_crossed++;
check_line_crossed = true;

}

void EventTrigger::set_unit_engaged(string name)
{
ue_array[i_unit_engaged] = name;
unit_engaged[i_unit_engaged] = false;
i_unit_engaged++;
check_unit_engaged = true;
}


void EventTrigger::set_id(string name)
{
id = name;
}


void EventTrigger::set_ud_fraction(double frac)
{
for (int i=0; i< n_unit_destroyed; i++)
	{
	ud_percentage_array[i] = frac;
	}
}

void EventTrigger::set_fd_fraction(double frac)
{
for (int i=0; i< n_formation_destroyed; i++)
	{
	fd_percentage_array[i] = frac;
	}
}

void EventTrigger::set_logic(string operation)
{
if ((operation == "AND") || (operation == "OR") || (operation == "NOT") || (operation == "XOR"))
	{logic = operation;}
else
	{cout << operation << " is not a valid trigger logic definition, exiting..." << endl; exit(0);}
}


string EventTrigger::get_id()
{
return id;
}

bool EventTrigger::check_logic(int n, bool array[])
{
int counter;

counter = 0;

for (int i=0; i< n; i++)
	{
	if (array[i] == true) {counter++;}
	}

if ((logic == "AND") && (counter == n))
	{return true;}
else if ((logic == "OR") && (counter > 0))
	{return true;}
else if ((logic == "NOT") && (counter == 0))
	{return true;}
else if ((logic == "XOR") && (counter == 1))
	{return true;}
else
	{return false;}

}

void EventTrigger::check_conditions(Unit unit_array[], int num_units, Formation formation_array[], int num_formations)
{

int formation_id;
int unit_orig_counter, unit_current_counter;
double fraction;

string name;


if (trigger_condition == true) {return;}

// formation destroyed

for (int i=0; i< n_formation_destroyed; i++)
	{
	name = fd_array[i];

	// find the formation by name

	for (int j=0; j< num_formations; j++)
		{
		if (name == formation_array[j].get_name())
			{
			formation_id = j; break;
			}
		}
	// check its units
	
	unit_orig_counter = formation_array[formation_id].get_num_units();
	unit_current_counter = 0;

	for (int j = 0; j< unit_orig_counter; j++)
		{
		if (unit_array[formation_array[formation_id].get_unit_id(j)].get_inactive() == false)
			{
			unit_current_counter++;
			}

		}


	fraction = 1.0 * unit_current_counter / unit_orig_counter;


	if (fraction <= (1.0 - fd_percentage_array[i]))
		{
		formation_destroyed[i] = true;
		}  

	}

if (check_formation_destroyed)
	{trigger_condition = check_logic(n_formation_destroyed, formation_destroyed);}

// unit destroyed

for (int i=0; i< n_unit_destroyed; i++)
	{

	name = ud_array[i];	

	//cout << "Trigger looking for name " << name << endl;

	for (int j=0; j< num_units; j++)
		{
		if (unit_array[j].get_tag() == name)
			{
			if (1.0 * unit_array[j].get_current_num_soldiers() / unit_array[j].get_nominal_num_soldiers() <= (1.0 - ud_percentage_array[i]))
				{
				unit_destroyed[i] = true; break;
				}
			if (unit_array[j].get_inactive() == true)
				{
				unit_destroyed[i] = true; break;
				} 

			}
		}

	}

if (check_unit_destroyed)
	{trigger_condition = check_logic(n_unit_destroyed, unit_destroyed);}

// line crossed

for (int i=0; i < n_line_crossed; i++)
	{
	name = lc_array[i];

	for (int j=0; j< num_units; j++)
		{
		if (unit_array[j].get_tag() == name)
			{
			if (lc_sign_array[i] > 0)
				{
				if (unit_array[j].get_pos_y() > lc_pos_array[i])
					{
					line_crossed[i] = true; break;
					}
				}
			else
				{
				if (unit_array[j].get_pos_y() < lc_pos_array[i])
					{
					line_crossed[i] = true; break;
					}

				}
			}
		

		}

	}


if (check_line_crossed)
	{trigger_condition = check_logic(n_line_crossed, line_crossed);}

// unit engaged

for (int i=0; i < n_unit_engaged; i++)
	{
	name = ue_array[i];	


	for (int j=0; j< num_units; j++)
		{
		if (unit_array[j].get_tag() == name)
			{
			if (unit_array[j].get_num_contacts() > 0)
				{
				unit_engaged[i] = true; break;
				}
			}
		}
	}


}

bool EventTrigger::get_trigger_condition()
{
return trigger_condition;
}


/* event result code ==================================================================== */

EventResult::EventResult()
{
command_to_formation = false;
command_to_unit = false;
event_executed = false;
event_triggered = false;
formation_target = "";
unit_target = "";
order = "";
trigger_id = "";
order_par_1 = -1.0;
order_par_2 = -1.0;
}


void EventResult::set_formation_target(string name)
{
formation_target = name;
command_to_formation = true;
}

void EventResult::set_unit_target(string name)
{
unit_target = name;
command_to_unit = true;
}

void EventResult::set_order(string ostring)
{
order = ostring;
}

void EventResult::set_trigger_id(string id)
{
trigger_id = id;
}

void EventResult::set_order_par_1(double par)
{
order_par_1 = par;
}

void EventResult::set_order_par_2(double par)
{
order_par_2 = par;
}


void EventResult::trigger()
{
if (event_triggered) {return;}

stringstream ss;

if (command_to_formation)
	{
	ss<< "formation ";
	ss<< formation_target;
	}
else if (command_to_unit)
	{
	ss << "unit ";
	ss <<  unit_target;
	}

ss << " ";
ss << order;
if (order_par_1 != -1.0)
	{
	ss << " ";
	ss << order_par_1;
	}
if (order_par_2 != -1.0)
	{
	ss << " ";
	ss << order_par_2;
	}


command_string = ss.str();
event_triggered = true;
}

string EventResult::get_command()
{
if (event_executed) {return "void";}

event_executed = true;
return command_string; 
}

string EventResult::get_id()
{
return trigger_id;
}


/* event logic code ==================================================================== */

EventLogic::EventLogic()
{
trigger_condition = false;

num_slots = 5;
num_slots_set = 0;

trigger_array = new bool[num_slots];
trigger_id_array = new string[num_slots];

logic == "AND";
}


bool EventLogic::check_logic(int n, bool array[])
{
int counter;

counter = 0;

for (int i=0; i< n; i++)
	{
	if (array[i] == true) {counter++;}
	}


if ((logic == "AND") && (counter == n))
	{return true;}
else if ((logic == "OR") && (counter > 0))
	{return true;}
else if ((logic == "NOT") && (counter == 0))
	{return true;}
else if ((logic == "XOR") && (counter == 1))
	{return true;}
else
	{return false;}

}


void EventLogic::set_logic(string operation)
{
if ((operation == "AND") || (operation == "OR") || (operation == "NOT") || (operation == "XOR"))
	{logic = operation;}
else
	{cout << operation << " is not a valid logic definition, exiting..." << endl; exit(0);}

}

void EventLogic::set_id(string name)
{
logic_id = name;
}

void EventLogic::add_trigger_id(string name)
{
trigger_id_array[num_slots_set] = name;
trigger_array[num_slots_set] = false;

num_slots_set++;
}

void EventLogic::set_condition(int id, bool value)
{

if (trigger_condition) {return;}

if (id > num_slots) 
	{cout << "Insufficient trigger slots, exiting..." << endl; exit(0);}
trigger_array[id] = value;
if (value == true) {trigger_condition = check_logic(num_slots_set, trigger_array);}
}




int EventLogic::get_num_conditions()
{
return num_slots_set;
}

string EventLogic::get_condition_id(int id)
{
return trigger_id_array[id];
}

string EventLogic::get_id()
{
return logic_id;
}

bool EventLogic::get_trigger_condition()
{
return trigger_condition;
}

/* event manager code ==================================================================== */

void EventManager::initialize(int num_triggers, int num_results, int num_logics)
{

n_trigger = num_triggers;
i_trigger = 0;

n_result = num_results;
i_result = 0;

n_logic = num_logics;
i_logic = 0;

trigger_array = new EventTrigger[num_triggers];
logic_array = new EventLogic[num_logics];
result_array = new EventResult[num_results];
command_buffer = new string[num_results];

}


void EventManager::create_result_formation(string trigger, string name, string order)
{
result_array[i_result].set_trigger_id(trigger);
result_array[i_result].set_formation_target(name);
result_array[i_result].set_order(order); 

i_result++;
}

void EventManager::create_result_formation(string trigger, string name, string order, double par1)
{
result_array[i_result].set_trigger_id(trigger);
result_array[i_result].set_formation_target(name);
result_array[i_result].set_order(order); 
result_array[i_result].set_order_par_1(par1);
i_result++;
}

void EventManager::create_result_formation(string trigger, string name, string order, double par1, double par2)
{
result_array[i_result].set_trigger_id(trigger);
result_array[i_result].set_formation_target(name);
result_array[i_result].set_order(order); 
result_array[i_result].set_order_par_1(par1);
result_array[i_result].set_order_par_2(par2);
i_result++;
}

void EventManager::create_result_unit(string trigger, string name, string order)
{
result_array[i_result].set_trigger_id(trigger);
result_array[i_result].set_unit_target(name);
result_array[i_result].set_order(order); 

i_result++;
}

void EventManager::init_trigger(int nf, int nu, int nl, int ne)
{
trigger_array[i_trigger].initialize(nf, nu, nl, ne);
}

void EventManager::set_trigger_id(string id)
{
trigger_array[i_trigger].set_id(id);
i_trigger++;
}

void EventManager::add_trigger_fd(string name)
{
trigger_array[i_trigger].set_formation_destroyed(name);
} 

void EventManager::add_trigger_ud(string name)
{
trigger_array[i_trigger].set_unit_destroyed(name);
} 

void EventManager::add_trigger_lc(string name, double pos, int sign)
{
trigger_array[i_trigger].set_line_crossed(name, pos, sign);
}

void EventManager::add_trigger_ue(string name)
{
trigger_array[i_trigger].set_unit_engaged(name);
}

void EventManager::set_trigger_logic(string type)
{
trigger_array[i_trigger].set_logic(type);
}

void EventManager::set_trigger_ud_fraction(double frac)
{
trigger_array[i_trigger].set_ud_fraction(frac);
}

void EventManager::set_trigger_fd_fraction(double frac)
{
trigger_array[i_trigger].set_fd_fraction(frac);
}

void EventManager::add_logic_input(string name)
{
logic_array[i_logic].add_trigger_id(name);

}

void EventManager::set_logic_type(string type)
{
logic_array[i_logic].set_logic(type);
}

void EventManager::set_logic_id(string name)
{
logic_array[i_logic].set_id(name);
i_logic++;
}


string EventManager::get_commandline(int index)
{
return command_buffer[index];
}

int EventManager::manage(Unit unit_array[], int num_units, Formation formation_array[], int num_formations)
{

int command_counter;
command_counter = 0;


for (int i=0; i< n_trigger; i++)
	{
	trigger_array[i].check_conditions(unit_array, num_units,  formation_array,  num_formations);

	if (trigger_array[i].get_trigger_condition() == true)
		{
		//cout << "Trigger is now true" << endl;
		for (int j=0; j< n_logic; j++)
			{
			for (int k=0; k< logic_array[j].get_num_conditions(); k++)
				{
				if (trigger_array[i].get_id() == logic_array[j].get_condition_id(k))
					{logic_array[j].set_condition(k, true);}
				}
			}
		for (int j=0; j< n_result; j++)
			{
			if (trigger_array[i].get_id() == result_array[j].get_id())
				{result_array[j].trigger();}
			}
		}
	}


for (int i=0; i < n_logic; i++)
	{
	
	if (logic_array[i].get_trigger_condition() == true)
		{
		//cout << "Logic is now true" << endl;
		for (int j=0; j< n_logic; j++)
			{
			for (int k=0; k< logic_array[j].get_num_conditions(); k++)
				{
				if (logic_array[i].get_id() == logic_array[j].get_condition_id(k))
					{logic_array[j].set_condition(k, true);}
				}
			}
		}
		for (int j=0; j< n_result; j++)
			{
			if ((logic_array[i].get_id() == result_array[j].get_id()) && (logic_array[i].get_trigger_condition() == true))
				{result_array[j].trigger();}
			}
	}

for (int i=0; i< n_result; i++)
	{
	if ((result_array[i].event_triggered) && (result_array[i].event_executed == false))
		{
		command_buffer[command_counter] = result_array[i].get_command();
		//cout << "Command issued: " <<   command_buffer[command_counter] << endl;
		command_counter++;
		}
	}




return command_counter;
}
