#include <iostream>
#include <fstream>
#include <string>
#include <sstream>
#include <stdlib.h>
#include <math.h>

using namespace std;

#include "options.hxx"
#include "unit.hxx"
#include "formation.hxx"
#include "terrain.hxx"
#include "events.hxx"

void show_battlefield();
void zoom_battlefield(double, double, double, double, double);
void zoom_battlefield(double);
void zoom_battlefield_reset(void);
void crop_battlefield(double, double, double, double);

double unit_dist(int, int);

Options options_list; 
Terrain terrain;  
Unit *unit_array;
Formation *formation_array;
Command cmd_input;
EventManager events;

int num_battles;
int* battle_array_1;
int* battle_array_2;
int num_charges;
int* charge_array_1;
int* charge_array_2;

string color_red, color_green, color_blue, color_magenta, color_bred, color_bgreen, color_bblue, color_bmagenta, color_normal, color_bold;
string color_rbg, color_brbg, color_ybg, color_bybg, color_rybg, color_brybg;

#include "commandline_parser.hxx"
#include "parser.hxx"

string commandline_parser (int argc, char *argv[]) 
{ 
string config_file;

if (argc != 2)
	{
	cout <<"Please specify a configuration file!" << endl;
	}
else
	{
	config_file = argv[1];
	cout << "Reading " << config_file << " for configuration." << endl;
	}

return config_file;

}

double rnd(double min, double max)
{
double average = 0.5 * (max + min);
double variance = max - average;

return average + variance * 2.0 * ( rand()/(RAND_MAX +1.0) - 0.5);
}

string get_color(string bold, string fore, string back)
{
string code;

code = "\033[";


if (bold == "bold")
	{code = code + "1;";} 


if (fore == "red")
	{code = code + "31;";}
else if (fore == "yellow")
	{code = code + "33;";}

if (back == "red")
	{code = code + "41;";}
else if (back == "yellow")
	{code = code + "43;";}


code = code + "m";
//cout << "Code: " << code << endl;

return code;
}

double unit_dist(int id1, int id2)
{

double p1x, p1y, p2x, p2y;

p1x = unit_array[id1].get_pos_x();
p1y = unit_array[id1].get_pos_y();
p2x = unit_array[id2].get_pos_x();
p2y = unit_array[id2].get_pos_y();

return sqrt( (p1x - p2x) * (p1x - p2x) + (p1y - p2y) * (p1y - p2y));
}

double unit_place_dist(int id, double x, double y)
{
double px, py;

px = unit_array[id].get_pos_x();
py = unit_array[id].get_pos_y();

return sqrt( (px - x) * (px - x) + (py - y) * (py - y));
}


int find_nearest_enemy(int id1)
{
int id_min;

double distance, distance_min;

id_min = -1;

distance_min = 1e6;

for (int i=0; i< options_list.num_units; i++)
	{
	if ((unit_array[id1].get_army_id() != unit_array[i].get_army_id()) && (unit_array[i].get_inactive() == false))
		{
		distance = unit_dist(id1, i);
		if (distance < distance_min)
			{
			distance_min = distance;
			id_min = i;
			}
		}
	}

return id_min;

}

void do_morale_check(int id, double value)
{
if (options_list.config_check_morale)
	{
	unit_array[id].check_morale(value);
	}
}


void apply_damage_table(int id, double value, double multiplier)
{
double l, h, d, m;

if (value < 1.0)
	{l = 0.0; h = 0.0; d = 0.0; m = -1.0;}
else if (value < 2.0)
	{l = 5.0; h = 5.0; d = 0.0; m = -1.0;}
else if (value < 3.0)
	{l = 10.0; h = 5.0; d = 0.0; m = -1.0;}
else if (value < 4.0)
	{l = 10.0; h = 10.0; d = 5.0; m = -1.0;}
else if (value < 5.0)
	{l = 10.0; h = 15.0; d = 5.0; m = -1.0;}
else if (value < 6.0)
	{l = 10.0; h = 15.0; d = 10.0; m = 0.0;}
else if (value < 7.0)
	{l = 15.0; h = 20.0; d = 10.0; m = 0.0;}
else if (value < 8.0)
	{l = 15.0; h = 20.0; d = 15.0; m = 0.0;}
else if (value < 9.0)
	{l = 15.0; h = 20.0; d = 20.0; m = 1.0;}
else if (value < 10.0)
	{l = 20.0; h = 25.0; d = 20.0; m = 1.0;}
else if (value < 11.0)
	{l = 20.0; h = 25.0; d = 25.0; m = 2.0;}
else if (value < 12.0)
	{l = 25.0; h = 30.0; d = 30.0; m = 2.0;}
else if (value < 13.0)
	{l = 25.0; h = 35.0; d = 35.0; m = 3.0;}
else 
	{l = 20.0; h = 40.0; d = 40.0; m = 4.0;}


// overall normalization per minute


l*=options_list.config_damage_multiplier * 0.5;
h*=options_list.config_damage_multiplier * 0.5;
d*=options_list.config_damage_multiplier * 0.5;

// move from percentage to fractions

multiplier *=0.01;

l*=multiplier;
h*=multiplier;
d*=multiplier;

unit_array[id].do_damage(l,h,d);

// we should not do full morale checks for low multipliers

if (multiplier < 0.05) {m -=4;}
else if (multiplier < 0.1) {m-=3;}
else if (multiplier < 0.25) {m-=2;}
else if (multiplier < 0.5) {m -=1;}


if (m >= 0.0) {do_morale_check(id, m); }

}


bool check_line_of_sight(int id1, int id2)
{
double x, y, vx, vy, z, vz;
double step, dist, test_dist, alt_above_terrain;


step = 10.0;

if (options_list.config_check_los == false) {return true;}



dist = unit_dist(id1, id2);

if (dist == 0) {return true;}

x = unit_array[id1].get_pos_x();
y = unit_array[id1].get_pos_y();

vx = (unit_array[id2].get_pos_x() - x)/dist;
vy = (unit_array[id2].get_pos_y() - y)/dist;



if (options_list.terrain_defined)
	{
	z = terrain.get_elevation(x,y) + 2.0;
	vz = (terrain.get_elevation(unit_array[id2].get_pos_x(), unit_array[id2].get_pos_y())+ 2.0 - z)/dist;
	z += step * vz;

	}

x += step * vx;
y += step * vy;



while (unit_place_dist(id2, x,y) > 10.0)
	{
	

	alt_above_terrain = 1.0;

	if (options_list.terrain_defined)
		{
		alt_above_terrain = z - terrain.get_elevation(x,y);
		//cout << x << " " << y << " " << z << " " << terrain.get_elevation(x,y) << endl;
		//cout << "Alt above terrain: " << alt_above_terrain << endl;
		if (alt_above_terrain < 0.0) {return false;}
		}

	if (alt_above_terrain < 2.0)
		{
		for (int i=0; i< options_list.num_units; i++)
			{
			if ((i!= id1) && (i!=id2))
				{
				test_dist = unit_place_dist(i, x,y);
				if (test_dist < 10.0) 
					{
					return false;
					}

				}
			}
		}
	x += step * vx;
	y += step * vy;

	if (options_list.terrain_defined)
		{
		z += step * vz;
		}
	}



return true;
}


double get_forced_movement (string strategy_own, string strategy_other)
{

if (strategy_own == "assault")
	{
	if (strategy_other == "assault") {return 0.0;}
	else if (strategy_other == "attack") {return 4.0;}
	else if (strategy_other == "yield") {return 8.0;}
	else if (strategy_other == "hold_line") {return 3.0;}	
	else if (strategy_other == "skirmish") {return 6.0;}	
	}
else if (strategy_own == "attack")
	{
	if (strategy_other == "assault") {return 0.0;}
	else if (strategy_other == "attack") {return 0.0;}
	else if (strategy_other == "yield") {return 4.0;}
	else if (strategy_other == "hold_line") {return 0.0;}	
	else if (strategy_other == "skirmish") {return 0.0;}	
	}
else if (strategy_own == "yield")
	{
	return 0.0;
	}
else if (strategy_own == "hold_line")
	{
	return 4.0;
	}
else if (strategy_own == "skirmish")
	{
	return 2.0;
	}

return 0.0;

}


double get_damage_modifier (string strategy_own, string strategy_other)
{

if (strategy_own == "assault")
	{
	if (strategy_other == "assault") {return 3.0;}
	else if (strategy_other == "attack") {return 2.0;}
	else if (strategy_other == "yield") {return -1.0;}
	else if (strategy_other == "hold_line") {return 3.0;}
	else if (strategy_other == "skirmish") {return 1.0;}
	}
else if(strategy_own == "attack")
	{
	if (strategy_other == "assault") {return 1.0;}
	else if (strategy_other == "attack") {return 0.0;}
	else if (strategy_other == "yield") {return -1.0;}
	else if (strategy_other == "hold_line") {return 0.0;}
	else if (strategy_other == "skirmish") {return -1.0;}
	}
else if (strategy_own == "yield")
	{
	if (strategy_other == "assault") {return -2.0;}
	else if (strategy_other == "attack") {return -2.0;}
	else if (strategy_other == "yield") {return -4.0;}
	else if (strategy_other == "hold_line") {return -3.0;}
	else if (strategy_other == "skirmish") {return -4.0;}
	}
else if (strategy_own == "hold_line")
	{
	if (strategy_other == "assault") {return 2.0;}
	else if (strategy_other == "attack") {return 0.0;}
	else if (strategy_other == "yield") {return -3.0;}
	else if (strategy_other == "hold_line") {return -3.0;}
	else if (strategy_other == "skirmish") {return -4.0;}
	}
else if (strategy_own == "skirmish")
	{
	if (strategy_other == "assault") {return 2.0;}
	else if (strategy_other == "attack") {return -2.0;}
	else if (strategy_other == "yield") {return -4.0;}
	else if (strategy_other == "hold_line") {return -4.0;}
	else if (strategy_other == "skirmish") {return -4.0;}
	}


return 0.0;
}

void process_ranged_combat(int idf, int idt)
{

bool line_of_sight;

double attack_strength, defense_strength;
double tactical_advantage_attack, tactical_advantage_defense, speed;
double damage, distance, range, dratio, damage_mult, armour_mult;
double alt_f, alt_t, alt_difference, alt_max;

distance = unit_dist(idf, idt);

attack_strength = rnd(0, options_list.config_random_range) + unit_array[idf].get_ranged_attack();
defense_strength = rnd(0, options_list.config_random_range) + unit_array[idt].get_ranged_defense();

line_of_sight = check_line_of_sight(idf, idt);

//cout << "LOS is now " << line_of_sight << endl;

if (line_of_sight == false) 
	{attack_strength = 0.2 * attack_strength;}



//cout << "Range: " << unit_array[idf].get_range() << " speed: " << unit_array[idf].get_current_speed() << " " << unit_array[idt].get_current_speed() << " dist: " << distance << endl;


range = unit_array[idf].get_range();


// terrain effects

if (options_list.terrain_defined)
	{
	alt_f = terrain.get_elevation(unit_array[idf].get_pos_x(), unit_array[idf].get_pos_y());
	alt_t = terrain.get_elevation(unit_array[idt].get_pos_x(), unit_array[idt].get_pos_y());
	
	alt_difference = alt_t - alt_f;
	alt_max = range/2.0;

	if (alt_max < alt_difference)
		{range = 0.0;}
	else
		{
		range = range/sqrt(alt_max) * sqrt(alt_max - alt_difference); 
		}
	}



speed = unit_array[idf].get_current_speed() + unit_array[idt].get_current_speed() * distance/range;

if ((speed > 50.0) && (unit_array[idf].get_mounted() == false))
	{
	speed = 0.5 * (speed - 50);
	attack_strength -= 0.5 * (speed - 50.0); 
	}
else if (speed > 150.0)
	{
	speed = 0.25 * (speed - 150);
	attack_strength -= 0.25 * (speed - 150.0); 
	}
else {speed = 0.0;}



tactical_advantage_attack = 0.1 * (attack_strength - defense_strength);
if (tactical_advantage_attack < 0.0) {tactical_advantage_attack = 0.0;}

tactical_advantage_defense = 0.1 * (defense_strength - attack_strength);
if (tactical_advantage_defense < 0.0) {tactical_advantage_defense = 0.0;}

damage_mult = 1.0;
armour_mult = 1.0;




if (distance > range)
	{
	damage_mult = 0.0;	
	}
else
	{
	dratio = distance/range;
	//damage_mult = 1.0 - 0.2 * dratio;
	damage_mult = 1.0 - (1.0 - pow((1.0 - unit_array[idf].get_projectile_drag()),2.0)) * dratio;
	armour_mult = 1.0 + 0.4 * dratio * dratio;
	}

if (options_list.terrain_defined)
	{
	if (alt_difference < alt_max)
		{damage_mult *= 1./alt_max * (alt_max - alt_difference);}
	else
		{damage_mult = 0.0;}
	}


damage = rnd(0,10.0) + (tactical_advantage_attack - tactical_advantage_defense) + unit_array[idf].get_ranged_damage() - unit_array[idt].get_armor() * armour_mult;
if (damage < 0.0) {damage = 0.0;}

// adjust damage multiplier for the relative number of soldiers


apply_damage_table (idt, damage, damage_mult * (1.0 * unit_array[idf].get_equiv_num_soldiers_ra()/ unit_array[idt].get_equiv_num_soldiers_rd()));
unit_array[idt].register_event("under_fire");

if (options_list.config_events_verbose)
	{
	if (unit_array[idf].get_num_shots() == 1)
		{
		cout << "Unit " << unit_array[idf].get_tag() << " out of ammunition." << endl;
		} 
	}

if (options_list.config_battle_verbose)
	{
	cout << "Ranged attack of " << unit_array[idf].get_tag() << " on " << unit_array[idt].get_tag() << endl;
	cout << "Current range is " << distance << " with damage factor " << damage_mult << " and armour factor " << armour_mult << endl;
	cout << "Movement reduces attack strength by " << speed << " points." << endl;
	if (options_list.terrain_defined)
		{
		cout << "Altitude difference is " << alt_difference << " m, altering range to " << range << " m." << endl;
		}
	cout << "Tactical advantage " << tactical_advantage_attack << " doing " << damage << " points of damage." << endl;  
	}
else if (options_list.config_battle_terse)
	{
	cout << "Ranged attack of " << unit_array[idf].get_tag() << " on " << unit_array[idt].get_tag() << " doing " << damage << " points of damage." << endl;
	}

if (unit_array[idt].get_inactive() == true)
	{
	unit_array[idf].register_event("target_destroyed");
	if (options_list.config_events_verbose) 
		{	
		cout << "Unit " << unit_array[idt].get_name() << " is defeated." << endl;
		}
	}


if ((options_list.config_events_verbose) && (unit_array[idt].get_fled() == true))
	{
	cout << "Unit " << unit_array[idt].get_tag() << " flees." << endl;
	}

}



void process_close_combat (int id1, int id2)
{
double fighting_strength_A, fighting_strength_B;
double tactical_advantage_A, tactical_advantage_B;
double damage_A, damage_B, dmod_A, dmod_B, dmult_A, dmult_B, nmult_A, nmult_B;
double move_A, move_B, impact_A, impact_B;
double dist_norm, ang_A, ang_B, vx1, vy1, vx2, vy2;
double alt_A, alt_B, slope, slip_A, slip_B;

dist_norm = unit_dist(id1, id2);

if (dist_norm == 0.0) {cout << "Close combat: Zero unit distance detected!" << endl; dist_norm = 0.0001;}

vx1 = unit_array[id2].get_pos_x() - unit_array[id1].get_pos_x();
vy1 = unit_array[id2].get_pos_y() - unit_array[id1].get_pos_y();

vx1/=dist_norm;
vy1/=dist_norm;

ang_A = unit_array[id1].get_dir_x() * vx1 + unit_array[id1].get_dir_y() * vy1;

vx2=-vx1;
vy2=-vy1;

ang_B = unit_array[id2].get_dir_x() * vx2 + unit_array[id2].get_dir_y() * vy2;

//cout << unit_array[id1].get_tag() << " angle " << ang_A << endl;
//cout << unit_array[id2].get_tag() << " angle " << ang_B << endl;

fighting_strength_A = rnd(0, options_list.config_random_range) + unit_array[id1].get_attack();
fighting_strength_B = rnd(0, options_list.config_random_range) + unit_array[id2].get_attack();

if (ang_B < -0.5) {fighting_strength_A +=  unit_array[id2].get_penalty_rear_attack();}
else if (ang_B < 0.5) {fighting_strength_A += unit_array[id2].get_penalty_flank_attack();}

if (ang_A < -0.5) {fighting_strength_B +=  unit_array[id1].get_penalty_rear_attack();}
else if (ang_A < 0.5) {fighting_strength_B += unit_array[id1].get_penalty_flank_attack();}

if (unit_array[id1].get_num_contacts() > 1)
	{fighting_strength_A -= 20.0;}

if (unit_array[id2].get_num_contacts() > 1)
	{fighting_strength_B -= 20.0;}

tactical_advantage_A = 0.1 * (fighting_strength_A - fighting_strength_B);
if (tactical_advantage_A < 0.0) {tactical_advantage_A = 0.0;}

tactical_advantage_B = 0.1 * (fighting_strength_B - fighting_strength_A);
if (tactical_advantage_B < 0.0) {tactical_advantage_B = 0.0;}

// forced movement

move_A = tactical_advantage_A;
move_B = tactical_advantage_B;

move_A += get_forced_movement(unit_array[id1].get_order(), unit_array[id2].get_order());
move_B += get_forced_movement(unit_array[id2].get_order(), unit_array[id1].get_order());

impact_A = unit_array[id1].get_current_speed() * unit_array[id1].get_push_density();
impact_B = unit_array[id2].get_current_speed() * unit_array[id2].get_push_density();

//cout << "Impact A: " << impact_A << endl;
//cout << "Impact B: " << impact_B << endl;

if ((impact_A == 0) && (impact_B == 0)) // pushing match
	{
	impact_A = unit_array[id1].get_push_density();
	impact_B = unit_array[id2].get_push_density();


	if (options_list.terrain_defined)
		{
		alt_A = terrain.get_elevation(unit_array[id1].get_pos_x(), unit_array[id1].get_pos_y());
		alt_B = terrain.get_elevation(unit_array[id2].get_pos_x(), unit_array[id2].get_pos_y());
		slope = (alt_B - alt_A)/dist_norm;

		if (slope > 0.0) // A is facing uphill
			{
			slip_A = 1.0 - pow(slope, 2.0);
			if (slip_A < 0.0) {slip_A = 0.0;}
			impact_A *= slip_A;

			slip_B = 1.25 - pow((0.5 - slope),2.0);
			if (slip_B < 0.0) {slip_B = 0.0;}
			impact_B *= slip_B;

			//cout << "Slip A: " << slip_A << " slip_B: " << slip_B << endl;
			}
		else // B is facing uphill
			{
			slip_B = 1.0 - pow(slope, 2.0);
			if (slip_B < 0.0) {slip_B = 0.0;}
			impact_B *= slip_B;

			slip_A = 1.25 - pow((0.5 - slope),2.0);
			if (slip_A < 0.0) {slip_A = 0.0;}
			impact_A *= slip_A;

			//cout << "Slip A: " << slip_A << " slip_B: " << slip_B << endl;
			}
		}

	//cout << "Push A: " << impact_A << endl;
	//cout << "Push B: " << impact_B << endl;

	if (impact_A > impact_B) 
		{
		move_A += (impact_A - impact_B)/400.0;
		}
	else
		{
		move_B += (impact_B - impact_A)/400.0;
		}
	}

else
	{
	if (impact_A > impact_B) // running crash
		{
		move_A += (impact_A - impact_B)/40000.0;
		}
	else
		{
		move_B += (impact_B - impact_A)/40000.0;
		}
	}
	



dmult_A = 1.0;
dmult_B = 1.0;
nmult_A = 1.0;
nmult_B = 1.0;

if (move_A > move_B)
	{
	move_A = move_A - move_B; 
	move_B = 0.0;

	if ((unit_array[id1].get_order() == "yield") || (unit_array[id1].get_order() == "hold_line")) {move_A = 0.0;}

	if (move_A > 0.25 * unit_array[id2].get_speed()) 
		{
		dmult_B = 3.0; 
		unit_array[id2].scatter();
		do_morale_check(id2, 8.0);

		if (options_list.config_events_verbose) 
			{
			cout << "Unit " << unit_array[id2].get_tag() << " is broken." << endl;

			if (unit_array[id2].get_morale() == "destroyed")
				{
				cout << "Unit " << unit_array[id2].get_tag() << " flees." << endl;
				}
			else if (unit_array[id2].get_morale() == "shaken")
				{
				if (unit_array[id2].get_flag("morale") == false)
					{
					cout << "Unit " << unit_array[id2].get_tag() << " shaken." << endl;
					unit_array[id2].set_flag("morale", true);
					}
				}
			}
		}

	else if (move_A > 0.2 * unit_array[id2].get_speed()) 
		{
		dmult_B = 2.0; 
		unit_array[id2].scatter();
		do_morale_check(id2, 5.0);
		if (options_list.config_events_verbose) 
			{
			cout << "Unit " << unit_array[id2].get_tag() << " is overrun." << endl;

			if (unit_array[id2].get_morale() == "destroyed")
				{
				cout << "Unit " << unit_array[id2].get_tag() << " flees." << endl;
				}
			else if (unit_array[id2].get_morale() == "shaken")
				{
				if (unit_array[id2].get_flag("morale") == false)
					{
					cout << "Unit " << unit_array[id2].get_tag() << " shaken." << endl;
					unit_array[id2].set_flag("morale", true);
					}
				}
			}
		}

	if (options_list.config_battle_verbose)
		{
		cout << "Unit " << unit_array[id1].get_tag() << " pushes " << move_A << endl; 
		}
	}
else
	{
	move_B = move_B - move_A;
	move_A = 0.0;

	if ((unit_array[id2].get_order() == "yield")||(unit_array[id2].get_order() == "hold_line")) {move_B = 0.0;}

	if (move_B > 0.25 * unit_array[id1].get_speed()) 
		{
		dmult_A = 3.0; 
		unit_array[id1].scatter();
		do_morale_check(id1, 8.0);
		if (options_list.config_events_verbose) 
			{
			cout << "Unit " << unit_array[id1].get_tag() << " is broken." << endl;

			if (unit_array[id1].get_morale() == "destroyed")
				{
				cout << "Unit " << unit_array[id1].get_tag() << " flees." << endl;
				}
			else if (unit_array[id1].get_morale() == "shaken")
				{
				if (unit_array[id1].get_flag("morale") == false)
					{
					cout << "Unit " << unit_array[id1].get_tag() << " shaken." << endl;
					unit_array[id1].set_flag("morale", true);
					}
				}
			}
		}
	else if (move_B > 0.2 * unit_array[id1].get_speed()) 
		{
		dmult_A = 2.0; 
		unit_array[id1].scatter();
		do_morale_check(id1, 5.0);
		if (options_list.config_events_verbose) 
			{
			cout << "Unit " << unit_array[id1].get_tag() << " is overrun." << endl;

			if (unit_array[id1].get_morale() == "destroyed")
				{
				cout << "Unit " << unit_array[id1].get_tag() << " flees." << endl;
				}
			else if (unit_array[id1].get_morale() == "shaken")
				{
				if (unit_array[id1].get_flag("morale") == false)
					{
					cout << "Unit " << unit_array[id1].get_tag() << " shaken." << endl;
					unit_array[id1].set_flag("morale", true);
					}
				}
			}
		}

	if (options_list.config_battle_verbose)
		{
		cout << "Unit " << unit_array[id2].get_tag() << " pushes " << move_B << endl; 
		}
	}



if (move_A > 0.0)
	{
	if (unit_array[id1].get_order() == "skirmish")
		{
		unit_array[id1].contact_move(1.0, vx1, vy1,  -move_A);
		unit_array[id2].contact_move(1.0, vx1, vy1,   move_A);
		}
	else
		{ 
		unit_array[id1].contact_move(1.0, vx1, vy1,  move_A);
		unit_array[id2].contact_move(1.0, vx1, vy1,  move_A);
		}
	}
else
	{
	if (unit_array[id2].get_order() == "skirmish")
		{
		unit_array[id1].contact_move(1.0, vx2, vy2,   move_B);
		unit_array[id2].contact_move(1.0, vx2, vy2,  -move_B);
		}
	else
		{ 
		unit_array[id1].contact_move(1.0, vx2, vy2,  move_B);
		unit_array[id2].contact_move(1.0, vx2, vy2,  move_B);
		}
	}

// damage

dmod_A = get_damage_modifier(unit_array[id1].get_order(), unit_array[id2].get_order());
dmod_B = get_damage_modifier(unit_array[id2].get_order(), unit_array[id1].get_order());

if ((dmult_A == 2.0)&& (dmod_A < 0.0)) {dmod_A = 0.0;} // yielding does not protect when being overrun
if ((dmult_B == 2.0)&& (dmod_B < 0.0)) {dmod_B = 0.0;}

damage_A = rnd(0,10.0) + tactical_advantage_B + unit_array[id2].get_damage() - unit_array[id1].get_armor() + dmod_A;
damage_B = rnd(0,10.0) + tactical_advantage_A + unit_array[id1].get_damage() - unit_array[id2].get_armor() + dmod_B;

if (damage_A < 0.0) {damage_A = 0.0;}
if (damage_B < 0.0) {damage_B = 0.0;}

// adjust damage multiplier for the relative number of soldiers

if (unit_array[id1].get_equiv_num_soldiers_cc() > unit_array[id2].get_equiv_num_soldiers_cc())	
	{
	nmult_A = 1.0 * unit_array[id2].get_equiv_num_soldiers_cc()/ unit_array[id1].get_equiv_num_soldiers_cc();
	dmult_A *= nmult_A ;
	}
else if (unit_array[id2].get_equiv_num_soldiers_cc() > unit_array[id1].get_equiv_num_soldiers_cc())
	{
	nmult_B = 1.0 * unit_array[id1].get_equiv_num_soldiers_cc()/ unit_array[id2].get_equiv_num_soldiers_cc();
	dmult_B *=  nmult_B;
	}

apply_damage_table (id1, damage_A, dmult_A);
apply_damage_table (id2, damage_B, dmult_B);

// verbose

if (options_list.config_battle_verbose)
	{
	if (tactical_advantage_A > 0.0)
		{cout << "Tactical advantage " << unit_array[id1].get_name() << " " << tactical_advantage_A << " points" << endl;}
	else if (tactical_advantage_B > 0.0)
		{cout << "Tactical advantage " << unit_array[id2].get_name() << " " << tactical_advantage_B << " points" << endl;}
	else 
		{cout << "No advantage." << endl;}
	if (nmult_A < nmult_B)
		{
		cout << "Damage reduction by relative numbers for " << unit_array[id1].get_name() << " is " << nmult_A << endl;
		}
	else if (nmult_B < nmult_A)
		{
		cout << "Damage reduction by relative numbers for " << unit_array[id2].get_name() << " is " << nmult_B << endl;
		}
	cout << "Unit " << unit_array[id1].get_name() << " takes " << damage_A << " damage points" << endl;
	cout << "Unit " << unit_array[id2].get_name() << " takes " << damage_B << " damage points" << endl;

	}
else if (options_list.config_battle_terse)
	{
	cout << "Close combat of " << unit_array[id1].get_tag() << " taking " << damage_A << " and " << unit_array[id2].get_tag() << " taking " << damage_B << " damage points" << endl;
	}


// detect victory

if (unit_array[id1].get_inactive())
	{
	unit_array[id2].register_event("victory");
	if (options_list.config_battle_verbose)
		{
		cout << "Unit " << unit_array[id1].get_name() << " is defeated." << endl;
		}
	}

if (unit_array[id2].get_inactive())
	{
	unit_array[id1].register_event("victory");
	if (options_list.config_battle_verbose)
		{
		cout << "Unit " << unit_array[id2].get_name() << " is defeated." << endl;
		}
	}


}


void update_move_targets()
{
double tgt_dir_x, tgt_dir_y, distance;

int id;

for (int i=0; i< options_list.num_units; i++)
	{
	if (unit_array[i].get_order() == "seek_enemy")
		{
		id = find_nearest_enemy(i);

		if (id == -1) {unit_array[i].give_order("halt"); continue;}

		tgt_dir_x = unit_array[id].get_pos_x() - unit_array[i].get_pos_x();
		tgt_dir_y = unit_array[id].get_pos_y() - unit_array[i].get_pos_y();

		unit_array[i].set_direction(tgt_dir_x, tgt_dir_y);
		}
	else if ((unit_array[i].get_order() == "hit_and_run") || (unit_array[i].get_order() == "hit_and_retreat"))
		{
		id = find_nearest_enemy(i);

		if (id == -1) {unit_array[i].give_order("halt"); continue;}

		tgt_dir_x = unit_array[id].get_pos_x() - unit_array[i].get_pos_x();
		tgt_dir_y = unit_array[id].get_pos_y() - unit_array[i].get_pos_y();

		distance = unit_dist(i, id);

		if ((unit_array[i].get_flag("hit_and_run")) && (distance < unit_array[i].get_dist_to_engage()))
			{
			tgt_dir_x = -tgt_dir_x;
			tgt_dir_y = -tgt_dir_y;
			}
		else if (distance < unit_array[i].get_dist_to_disengage())
			{
			if (unit_array[i].get_order() == "hit_and_run")
				{
				unit_array[i].set_flag("hit_and_run", true);
				}
			else 		
				{
				unit_array[i].register_event("retreat_to_assembly");
				}
			}
		else if (distance > unit_array[i].get_dist_to_engage())
			{
			unit_array[i].set_flag("hit_and_run", false);
			}
		unit_array[i].set_direction(tgt_dir_x, tgt_dir_y);

		}
	else if (unit_array[i].get_order() == "engage")
		{
		id = unit_array[i].get_tgt_id();
		if (unit_array[id].get_inactive() == false)
			{
			tgt_dir_x = unit_array[id].get_pos_x() - unit_array[i].get_pos_x();
			tgt_dir_y = unit_array[id].get_pos_y() - unit_array[i].get_pos_y();

			unit_array[i].set_direction(tgt_dir_x, tgt_dir_y);

			}
		else 	
			{unit_array[i].register_event("contact_broken");}

		}

	if ((unit_array[i].get_secondary_order() == "engage_upon") || (unit_array[i].get_secondary_order() == "guard"))
		{

		id = find_nearest_enemy(i);

		if (id == -1) {unit_array[i].give_order("halt"); continue;}

		if (unit_dist(i, id) < unit_array[i].get_dist_to_engage())
			{
			tgt_dir_x = unit_array[id].get_pos_x() - unit_array[i].get_pos_x();
			tgt_dir_y = unit_array[id].get_pos_y() - unit_array[i].get_pos_y();

			unit_array[i].set_direction(tgt_dir_x, tgt_dir_y);
			unit_array[i].register_event("target_engage");
			}

		}
	else if (unit_array[i].get_secondary_order() == "fire_upon")
		{
		id = find_nearest_enemy(i);

		if (id == -1) {unit_array[i].give_order("halt"); continue;}

		if (unit_dist(i, id) < unit_array[i].get_dist_to_fire())
			{
			unit_array[i].set_parameter("tgt_id", id);
			unit_array[i].set_parameter("range_flag", 1);
			}
		else
			{
			unit_array[i].set_parameter("range_flag", 0);
			}

		}
	else if (unit_array[i].get_secondary_order() == "fire_at")
		{
		id = unit_array[i].get_tgt_id();
		if (unit_dist(i, id) < unit_array[i].get_dist_to_fire())
			{
			unit_array[i].set_parameter("range_flag", 1);
			}
		else
			{
			unit_array[i].set_parameter("range_flag", 0);
			}
		}
	}
}

void move_units(double t)
{
double terrain_slope_along, terrain_slope_across;
double terrain_slowdown, terrain_max;

for (int i=0; i< options_list.num_units; i++)
	{
	if (unit_array[i].get_inactive()) {continue;}

	if (options_list.terrain_defined)
		{
		terrain_slope_along = terrain.get_slope(unit_array[i].get_pos_x(), unit_array[i].get_pos_y(), unit_array[i].get_dir());
		terrain_slope_across = terrain.get_slope(unit_array[i].get_pos_x(), unit_array[i].get_pos_y(), unit_array[i].get_dir() + 1.57079632675);
		unit_array[i].set_terrain_effects(terrain_slope_along, terrain_slope_across);
		}

	if (options_list.terrain_surface_defined)
		{
		terrain_slowdown = terrain.get_movement_slowdown(unit_array[i].get_pos_x(), unit_array[i].get_pos_y());
		terrain_max = terrain.get_movement_max(unit_array[i].get_pos_x(), unit_array[i].get_pos_y());
		unit_array[i].set_surface_effects(terrain_slowdown, terrain_max);
		}

	unit_array[i].move(t);


	if ((unit_array[i].get_army_id() == 1) && (unit_array[i].get_pos_y() < options_list.config_y_limit_1))
		{unit_array[i].register_event("line_reached");}
	if ((unit_array[i].get_army_id() == 2) && (unit_array[i].get_pos_y() > options_list.config_y_limit_2))
		{unit_array[i].register_event("line_reached");}
	}

}

void do_battle()
{


for (int i=0; i< num_battles; i++)
	{
	process_close_combat(battle_array_1[i], battle_array_2[i]);
	}

}

void do_mounted_charges()
{
int id1, id2;
bool is_charging1, is_charging2;
double forced_motion;


for (int i=0; i< num_charges; i++)
	{
	is_charging1 = false;
	is_charging2 = false;	

	id1 = charge_array_1[i];
	id2 = charge_array_2[i];

	if (unit_array[id1].get_current_speed() > options_list.config_charge_threshold) {is_charging1 = true;}
	if (unit_array[id2].get_current_speed() > options_list.config_charge_threshold) {is_charging2 = true;}
	
	if ((is_charging1 == true) && (is_charging2 == false))
		{do_morale_check(id2, 5.0);}

	if ((is_charging2 == true) && (is_charging1 == false))
		{do_morale_check(id1, 5.0);}

	process_close_combat(charge_array_1[i], charge_array_2[i]);
	//cout << "Forced motion: " << unit_array[id1].get_forced_motion() << endl;
	forced_motion = unit_array[id1].get_forced_motion();

	if (is_charging1 && is_charging2)
		{
		unit_array[id1].register_event("charge_successful");
		unit_array[id2].register_event("charge_successful");
		}
	else if (forced_motion > 15.0)
		{
		if ((is_charging1 == true) && (is_charging2 == false))
			{
			//cout << "Charge success!" << endl;
			unit_array[id1].register_event("charge_successful");
			}
		else if ((is_charging1 == false) && (is_charging2 == true))
			{
			unit_array[id2].register_event("charge_successful");
			}
		}
	else if ((forced_motion > 12.0) && (is_charging1 == true) && (unit_array[id2].get_push_density() < 150.0))
		{
		//cout << "Charge partial success!" << endl;
		unit_array[id1].register_event("charge_partial");
		}
	else if ((forced_motion > 12.0) && (is_charging2 == true) && (unit_array[id1].get_push_density() < 150.0))
		{
		unit_array[id2].register_event("charge_partial");
		}
	else if ((is_charging1 == true) && (unit_array[id1].get_push_density() < 100.0))
		{
		unit_array[id1].register_event("charge_partial");
		}
	else if ((is_charging2 == true) && (unit_array[id1].get_push_density() < 100.0))
		{
		unit_array[id2].register_event("charge_partial");
		}
	else // the charge has been stopped
		{
		double pos1_x = unit_array[id1].get_pos_x();
		double pos1_y = unit_array[id1].get_pos_y();
		double pos2_x = unit_array[id2].get_pos_x();
		double pos2_y = unit_array[id2].get_pos_y();

		battle_array_1[num_battles] = id1;
		battle_array_2[num_battles] = id2;
		unit_array[id1].register_contact(id2, pos2_x, pos2_y);
		unit_array[id2].register_contact(id1, pos1_x, pos1_y);
		num_battles++;
		}

	}

}


void do_ranged_attacks()
{
int num_shots_fired;


for (int i=0; i< options_list.num_units; i++)
	{
	if (unit_array[i].get_inactive() == true) {continue;}

	if ((unit_array[i].get_secondary_order() == "fire_at") && (unit_array[unit_array[i].get_tgt_id()].get_inactive() == false)&& (unit_array[i].check_range() == true))
		{
			
		num_shots_fired = unit_array[i].get_ready_to_fire(options_list.config_internal_timestep);

		for (int j=0; j< num_shots_fired ; j++)
			{
			process_ranged_combat(i, unit_array[i].get_tgt_id());
			}
		}
	else if ((unit_array[i].get_secondary_order() == "fire_at")  && (unit_array[unit_array[i].get_tgt_id()].get_inactive() == true))
		{
		unit_array[i].register_event("target_destroyed"); 
		}
	else if  ((unit_array[i].get_secondary_order() == "fire_upon") && (unit_array[i].check_range() == true))
		{

		if (unit_array[unit_array[i].get_tgt_id()].get_inactive() == false)
			{

			num_shots_fired = unit_array[i].get_ready_to_fire(options_list.config_internal_timestep);

			for (int j=0; j< num_shots_fired; j++)
				{
				process_ranged_combat(i, unit_array[i].get_tgt_id());
				}
			}

		}

	}
}


void clear_contacts()
{
for (int i=0; i< options_list.num_units; i++)
	{
	unit_array[i].clear_contacts();
	}
}

void find_contacts()
{
double pos1_x, pos1_y, pos2_x, pos2_y, cdist;
double dist, angle;

for (int i=0; i< options_list.num_units; i++)
	{
	for (int j=i; j< options_list.num_units; j++)
		{
		if (unit_array[i].get_inactive()) {continue;}
		if (unit_array[j].get_inactive()) {continue;}

		pos1_x = unit_array[i].get_pos_x();
		pos1_y = unit_array[i].get_pos_y();
		pos2_x = unit_array[j].get_pos_x();
		pos2_y = unit_array[j].get_pos_y();

		cdist = unit_array[i].get_contact_distance() + unit_array[j].get_contact_distance();
		
		if (i!=j)
			{
			dist = sqrt((pos1_x - pos2_x) * (pos1_x - pos2_x) + (pos1_y - pos2_y) * (pos1_y - pos2_y));
			if (dist < cdist)
				{
				if (dist > unit_array[i].get_contact_distance(pos2_x, pos2_y) + unit_array[j].get_contact_distance(pos1_x, pos1_y) ){continue;}	

				// we register charge when one unit is fast-moving
				if ((unit_array[i].get_army_id() != unit_array[j].get_army_id()) && ((unit_array[i].get_current_speed() > options_list.config_charge_threshold) || (unit_array[j].get_current_speed() > options_list.config_charge_threshold)))				{
					if ((unit_array[i].get_flag("charge_veto") == false) && (unit_array[i].get_flag("charge_veto") == false))
						{
						charge_array_1[num_charges] = i;
						charge_array_2[num_charges] = j;
						num_charges++;
						}
					}
				// we register a contact when slow-moving units meet
				else if (unit_array[i].get_army_id() != unit_array[j].get_army_id())
					{
					battle_array_1[num_battles] = i;
					battle_array_2[num_battles] = j;
					unit_array[i].register_contact(j, pos2_x, pos2_y);
					unit_array[j].register_contact(i, pos1_x, pos1_y);
					num_battles++;
					}
				else
					{
					// we register a friendly contact only for the unit which moves towards the other unit
					
					if (dist != 0.0)
						{
						angle = (pos2_x - pos1_x)/dist * unit_array[i].get_dir_x() +  (pos2_y - pos1_y)/dist * unit_array[i].get_dir_y();	
						if (angle > 0.5)
							{unit_array[i].register_friendly_contact(j, unit_array[j].get_density());}			
						else
							{unit_array[j].register_friendly_contact(i, unit_array[i].get_density());}
						}
					else 	 {
						cout << "Warning: zero unit contact distance found." << endl; 
						cout << unit_array[i].get_tag() << " " << unit_array[j].get_tag() << endl;
						}

					}
				}

			}
		}
	}

for (int i=0; i< options_list.num_units; i++)
	{
	if (unit_array[i].get_inactive()) {continue;}
	unit_array[i].check_contact_broken();
	}

}


void do_global_morale_checks()
{
for (int i=0; i< options_list.num_global_morale_checks_destruction; i++)
	{
	for (int j=0; j<options_list.num_units; j++)
		{
		if (options_list.global_morale_check_destruction[i] == unit_array[j].get_tag())
			{
			if (unit_array[j].get_inactive())
				{
				options_list.global_morale_check_destruction[i] = "";
				//cout << "Unit " << unit_array[j].get_tag() << " is dead, long live the unit!" << endl;

				for (int k=0; k< options_list.num_units; k++)
					{
					if ((unit_array[k].get_army_id() == unit_array[j].get_army_id()) && (unit_array[k].get_inactive() == false))
						{
						do_morale_check(k, options_list.global_morale_check_difficulty[i]);
						if ((unit_array[k].get_morale() == "destroyed") && options_list.config_events_verbose)
							{
							cout << "Unit " << unit_array[k].get_tag() << " flees." << endl;
						}
						}


					}
				}
			break;
			}
		}
	
	}
	
}


void formation_housekeeping()
{
double sum_x, sum_y, max_x, max_y, min_x, min_y;
double x_tmp, y_tmp;
int counter;

for (int i=0; i< options_list.num_formations; i++)
	{
	sum_x = 0;
	sum_y = 0;
	max_x = -1e6;
	max_y = -1e6;
	min_x = 1e6;
	min_y = 1e6;
	counter = 0;

	if (formation_array[i].get_inactive()) {continue;}

	for (int j=0; j< formation_array[i].get_num_units(); j++)
		{
		if (unit_array[formation_array[i].get_unit_id(j)].get_inactive() == false)
			{
			x_tmp = unit_array[formation_array[i].get_unit_id(j)].get_pos_x();
			y_tmp = unit_array[formation_array[i].get_unit_id(j)].get_pos_y();	
			sum_x += x_tmp;
			sum_y += y_tmp;

			if (x_tmp > max_x) {max_x = x_tmp;}
			if (y_tmp > max_y) {max_y = y_tmp;}
			if (x_tmp < min_x) {min_x = x_tmp;}
			if (y_tmp < min_y) {min_y = y_tmp;}
	
			counter++;
			}
		}
	if (counter > 0)
		{
		formation_array[i].set_shape(sum_x/counter, sum_y/counter, max_x, min_x, max_y, min_y);
		}
	else
		{
		formation_array[i].set_active(false);
		}
		
	}

}


void cleanup()
{
double limit_x;
double limit_y;


limit_x = options_list.field_size_x * options_list.field_scale;
limit_y = options_list.field_size_y * options_list.field_scale;

for (int i=0; i< options_list.num_units; i++)
	{
	if ((unit_array[i].get_pos_x() < - limit_x) || (unit_array[i].get_pos_x() > 2.0 * limit_x))
		{
		//cout << "Cleanup!" << endl;
		unit_array[i].check_morale(100.0);
		}
	else if ((unit_array[i].get_pos_y() < - limit_y) || (unit_array[i].get_pos_y() > 2.0 * limit_y))
		{
		//cout << "Cleanup!" << endl;
		unit_array[i].check_morale(100.0);
		}

	}

}



void zoom_battlefield_reset(void)
{
options_list.field_size_x = options_list.field_size_x_orig;
options_list.field_size_y = options_list.field_size_y_orig;
options_list.field_scale = options_list.field_scale_orig;
options_list.field_show_x_low = 0;
options_list.field_show_x_high = options_list.field_size_x;
options_list.field_show_y_low = 0;
options_list.field_show_y_high = options_list.field_size_y;
options_list.field_xl_current = 0.0;
options_list.field_xh_current = 1.0;
options_list.field_yl_current = 0.0;
options_list.field_yh_current = 1.0;
show_battlefield();
}

void zoom_battlefield(double scale, double xfrac_low, double xfrac_high, double yfrac_low, double yfrac_high)
{
if (scale != options_list.field_scale)
	{
	options_list.field_size_x = static_cast<int>(options_list.field_size_x * options_list.field_scale/scale);
	options_list.field_size_y = static_cast<int>(options_list.field_size_y * options_list.field_scale/scale);
	options_list.field_scale = scale;
	}

options_list.field_show_x_low = static_cast<int>(xfrac_low * options_list.field_size_x);
options_list.field_show_x_high = static_cast<int>(xfrac_high * options_list.field_size_x);
options_list.field_show_y_low = static_cast<int>(yfrac_low * options_list.field_size_y);
options_list.field_show_y_high = static_cast<int>(yfrac_high * options_list.field_size_y);

options_list.field_xl_current = xfrac_low;
options_list.field_xh_current = xfrac_high;
options_list.field_yl_current = yfrac_low;
options_list.field_yh_current = yfrac_high;

//cout << "x_low: " <<  options_list.field_show_x_low << " x_high: " << options_list.field_show_x_high << endl;
//cout << "y_low: " <<  options_list.field_show_y_low << " y_high: " << options_list.field_show_y_high << endl;

show_battlefield();
}

void crop_battlefield(double xfrac_low, double xfrac_high, double yfrac_low, double yfrac_high)
{
double delta_x, delta_y;
double xl_new, xh_new, yl_new, yh_new;

delta_x = options_list.field_xh_current - options_list.field_xl_current;
delta_y = options_list.field_yh_current - options_list.field_yl_current;

xl_new = options_list.field_xl_current + xfrac_low * delta_x;
xh_new = options_list.field_xl_current + xfrac_high * delta_x;
yl_new = options_list.field_yl_current + yfrac_low * delta_y;
yh_new = options_list.field_yl_current + yfrac_high * delta_y;

//cout << "xfrac_low " << xl_new << "xfrac_high" << xh_new << endl;
//cout << "yfrac_low " << yl_new << "yfrac_high" << yh_new << endl;

options_list.field_show_x_low = static_cast<int>(xl_new * options_list.field_size_x);
options_list.field_show_x_high = static_cast<int>(xh_new * options_list.field_size_x);
options_list.field_show_y_low = static_cast<int>(yl_new * options_list.field_size_y);
options_list.field_show_y_high = static_cast<int>(yh_new * options_list.field_size_y);

options_list.field_xl_current = xl_new;
options_list.field_xh_current = xh_new;
options_list.field_yl_current = yl_new;
options_list.field_yh_current = yh_new;


show_battlefield();
}


void zoom_battlefield(double scale)
{


if (scale != options_list.field_scale)
	{
	options_list.field_size_x = static_cast<int>(options_list.field_size_x * options_list.field_scale/scale);
	options_list.field_size_y = static_cast<int>(options_list.field_size_y * options_list.field_scale/scale);
	options_list.field_scale = scale;
	}

options_list.field_show_x_low = static_cast<int>(options_list.field_xl_current * options_list.field_size_x);
options_list.field_show_x_high = static_cast<int>(options_list.field_xh_current* options_list.field_size_x);
options_list.field_show_y_low = static_cast<int>(options_list.field_yl_current * options_list.field_size_y);
options_list.field_show_y_high = static_cast<int>(options_list.field_yh_current * options_list.field_size_y);


show_battlefield();
}


void show_battlefield()
{
double fraction;
int counter, num_cells, num_points;
string c1,  bold,  fore,  back;
string terrain_surface_symbol;

terrain_surface_symbol = ".";

counter = 0;

num_cells = 0;
num_points = options_list.config_tag_length;

//cout << options_list.config_tag_length << " " << options_list.field_aspect_factor << endl;

while (num_points >= options_list.field_aspect_factor)
	{
	num_cells++;
	num_points -= options_list.field_aspect_factor;
	}


//cout << "Num cells: " << num_cells << " Num points: " << num_points << endl; 

if (num_points > 0) {num_cells++; num_points = options_list.field_aspect_factor - num_points; }


//cout << "Num cells: " << num_cells << " Num points: " << num_points << endl; 

for (int i=options_list.field_show_y_low; i< options_list.field_show_y_high; i++)
	{
	for (int j=options_list.field_show_x_low; j< options_list.field_show_x_high; j++)
		{
		//cout << j * options_list.field_scale << " " << unit_array[0].get_pos_x() << endl;
			for (int k=0; k<options_list.num_units; k++)
				{
				if (unit_array[k].get_inactive()) {continue;}

				if ((j * options_list.field_scale < unit_array[k].get_pos_x()) && ((j+1) * options_list.field_scale >= unit_array[k].get_pos_x()) && (i * options_list.field_scale < unit_array[k].get_pos_y()) && ((i+1) * options_list.field_scale >= unit_array[k].get_pos_y()) && (counter == 0) )
					{
					//cout << "####";

					if (options_list.field_color_coding == 0)
						{
						if ((unit_array[k].get_health() < 0.5) && (unit_array[k].get_num_contacts() > 0)) {c1 = color_bred;}
						else if (unit_array[k].get_health() < 0.5) {c1 = color_red;}
						else if (unit_array[k].get_num_contacts() > 0) {c1 = color_bold;}
						else {c1 = color_normal;}
						}
					else if (options_list.field_color_coding == 1) // ammo
						{
						fraction = -1.0;
						if (unit_array[k].get_max_num_shots() > 0)
							{fraction = 1.0 * unit_array[k].get_num_shots()/unit_array[k].get_max_num_shots();}
						
						if (fraction >  0.75) {c1 = color_bgreen;}
						else if (fraction >  0.25) {c1 = color_bblue;}
						else if (fraction > 0.0) {c1 = color_bmagenta;}
						else if (fraction == 0.0) {c1 = color_bred;}
						else {c1 = color_normal;}
						}
					else if (options_list.field_color_coding == 2) // unit type
						{
						if ((unit_array[k].get_max_num_shots() > 0) && (unit_array[k].get_mounted()) && (unit_array[k].get_armor() > 5.0)) {c1 = color_bmagenta;}
						else if ((unit_array[k].get_max_num_shots() > 0) && (unit_array[k].get_mounted())) {c1 = color_magenta;}
						else if ((unit_array[k].get_max_num_shots() > 0) && (unit_array[k].get_armor() > 5.0)) {c1 = color_bblue;}
						else if (unit_array[k].get_max_num_shots() > 0) {c1 = color_blue;}
						else if (unit_array[k].get_mounted() && (unit_array[k].get_armor() > 5.0)) {c1 = color_bred;}
						else if (unit_array[k].get_mounted()) {c1 = color_red;}
						else if (unit_array[k].get_armor() > 5.0) {c1 = color_bold;}
						else {c1 = color_normal;}
						}
					else if (options_list.field_color_coding == 3) // slope
						{
						bold = "normal"; 
						fore = "normal";
						back = "normal";

						if (terrain.get_slope(j* options_list.field_scale , i* options_list.field_scale) > 1.0) {back = "red";}
						else if (terrain.get_slope(j* options_list.field_scale , i* options_list.field_scale) > 0.5) {back = "yellow";}


						//if ((unit_array[k].get_health() < 0.5) && (unit_array[k].get_num_contacts() > 0)) {fore = "red", bold = "bold";}
						//else if (unit_array[k].get_health() < 0.5) {fore = "red", bold = "normal";}
						//else if (unit_array[k].get_num_contacts() > 0) {fore = "normal", bold = "bold";}
						//else {c1 = color_normal;}
				

						//cout << c1 << endl;

						if ((unit_array[k].get_health() < 0.5) && (unit_array[k].get_num_contacts() > 0)) {c1 = color_bred;}
						else if (unit_array[k].get_health() < 0.5) {c1 = color_red;}
						else if (unit_array[k].get_num_contacts() > 0) 
							{
							if (back == "normal") {c1 = color_bold;}
							else if (back == "yellow") {c1 = color_bybg;}
							else if (back == "red") {c1 = color_brybg;}
							}
						else {c1 = color_normal;}
						}

					cout << c1 << unit_array[k].get_tag() << color_normal;
					counter = 1;
					}
				

		
				}
				if (counter == 0)
					{
					c1 = color_normal;
					if (options_list.field_color_coding == 3) // slope
						{
						if (terrain.get_slope(j* options_list.field_scale , i* options_list.field_scale) > 1.0) {c1 = color_rbg;}
						else if (terrain.get_slope(j* options_list.field_scale , i* options_list.field_scale) > 0.5) {c1 = color_ybg;}
						else {c1 = color_normal;}
						}
					
					if (options_list.terrain_surface_defined)
						{
						terrain_surface_symbol = terrain.get_surface_symbol(j* options_list.field_scale , i* options_list.field_scale);
						}

					for (int l=0; l<options_list.field_aspect_factor; l++)
						{cout << c1<< terrain_surface_symbol << color_normal;}
					}
				else 
					{
					counter ++;
					if (counter == num_cells+1) 
						{
						counter = 0;

						c1 = color_normal;
						if (options_list.field_color_coding == 3) // slope
							{
							if (terrain.get_slope(j* options_list.field_scale , i* options_list.field_scale) > 1.0) {c1 = color_rbg;}
							else if (terrain.get_slope(j* options_list.field_scale , i* options_list.field_scale) > 0.5) {c1 = color_ybg;}
							else {c1 = color_normal;}
							}
					
						if (options_list.terrain_surface_defined)
							{
							terrain_surface_symbol = terrain.get_surface_symbol(j* options_list.field_scale , i* options_list.field_scale);
							}

						for (int l=0; l<num_points; l++)
							{cout << c1<< terrain_surface_symbol << color_normal;}
						}
					}

		}
		cout << endl;
	}

	cout << endl;
}


void report_battle_summary()
{
int counter_A, counter_B, counter_A_fled, counter_B_fled;

counter_A = 0; 
counter_B = 0;
counter_A_fled = 0;
counter_B_fled = 0;

for (int i=0; i< options_list.num_units; i++)
	{
	if (unit_array[i].get_army_id() == 1) 
		{
		if (unit_array[i].get_fled() == true)
			{
			counter_A_fled += unit_array[i].get_current_num_soldiers();
			}
		else
			{
			counter_A += unit_array[i].get_current_num_soldiers();
			}
		}
	else if (unit_array[i].get_army_id() == 2) 
		{
		if (unit_array[i].get_fled() == true)
				{
				counter_B_fled += unit_array[i].get_current_num_soldiers();
				}
			else
				{
				counter_B += unit_array[i].get_current_num_soldiers();
				}
		}
	}

cout << "Army " << options_list.config_army_designation_1 << " currently has " << counter_A << " soldiers fighting and " << counter_A_fled << " soldiers routed." << endl;
cout << "Army " << options_list.config_army_designation_2 << " currently has " << counter_B << " soldiers fighting and " << counter_B_fled << " soldiers routed." << endl;
}


void report_army_composition()
{
int* counter_A;
int* counter_B;


counter_A = new int[options_list.num_unit_types];
counter_B = new int[options_list.num_unit_types];


for (int i=0; i< options_list.num_unit_types; i++)
	{
	counter_A[i] = 0;
	counter_B[i] = 0;
	}


for (int i=0; i< options_list.num_units; i++)
	{
	if (unit_array[i].get_army_id() == 1) 
		{
		for (int j=0; j<options_list.num_unit_types; j++)
			{
			if (unit_array[i].get_type() == options_list.unit_type_description[j]) {counter_A[j]+= unit_array[i].get_nominal_num_soldiers() ; break;}
			}
		}
	else if (unit_array[i].get_army_id() == 2) 
		{
		for (int j=0; j<options_list.num_unit_types; j++)
			{
			if (unit_array[i].get_type() == options_list.unit_type_description[j]) {counter_B[j]+= unit_array[i].get_nominal_num_soldiers(); break;}
			}
		}
	}


cout << "Army " << options_list.config_army_designation_1 << ":" << endl;

for (int i=0; i<  options_list.num_unit_types; i++)
	{
	cout << counter_A[i] << " soldiers of type " << options_list.unit_type_description[i] << endl;
	}

cout << "Army " << options_list.config_army_designation_2 << ":" << endl;

for (int i=0; i<  options_list.num_unit_types; i++)
	{
	cout << counter_B[i] << " soldiers of type " << options_list.unit_type_description[i] << endl;
	}

}

void report_unit_summary(bool flag)
{
int unit_counter_A = 0;
int unit_counter_B = 0;
int unit_orig_A = 0;
int unit_orig_B = 0;
int soldier_counter_A = 0;
int soldier_orig_A = 0;
int soldier_fled_A = 0;
int soldier_counter_B = 0;
int soldier_orig_B = 0;	
int soldier_fled_B = 0;
			
for (int i=0; i< options_list.num_units; i++)
	{
	if ((unit_array[i].get_army_id() == 1))
		{
		unit_orig_A++;
		if (flag) {soldier_orig_A += unit_array[i].get_nominal_num_soldiers();}
		if (unit_array[i].get_inactive() == false)
			{
			unit_counter_A++;
			if (flag) {soldier_counter_A += unit_array[i].get_current_num_soldiers();}
			}
		else
			{
			if (flag) {soldier_fled_A += unit_array[i].get_current_num_soldiers();}
			}
		}
	else if (unit_array[i].get_army_id() == 2) 
		{
		unit_orig_B++;
		if (flag) {soldier_orig_B += unit_array[i].get_nominal_num_soldiers();}
		if (unit_array[i].get_inactive() == false)
			{
			unit_counter_B++;
			if (flag) {soldier_counter_B += unit_array[i].get_current_num_soldiers();}
			}
		else
			{
			if (flag) {soldier_fled_B += unit_array[i].get_current_num_soldiers();}
			}
		}

	}

cout << "Army " << options_list.config_army_designation_1 << ":" << endl;
cout << unit_counter_A << " out of " << unit_orig_A << " units remain active." << endl;
if (flag) {cout << soldier_counter_A << " out of " << soldier_orig_A << " soldiers remain active, " << soldier_fled_A << " have fled or surrendered." << endl;}


cout << "Army " << options_list.config_army_designation_2 << ":" << endl;
cout << unit_counter_B << " out of " << unit_orig_B << " units remain active." << endl;
if (flag) {cout << soldier_counter_B << " out of " << soldier_orig_B << " soldiers remain active, " << soldier_fled_B << " have fled or surrendered." << endl;}
}




int main (int argc, char *argv[] ) 
{
string config_file, keyboard_input;

int unit_counter_A, unit_counter_B;
int iteration, victory_counter, draw_counter;

double elapsed_time, evolve_timer, battle_timer;



config_file = commandline_parser (argc, argv);

iteration = 0;
victory_counter = 0;
draw_counter = 0;

keyboard_input = "";
elapsed_time = 0.0;
evolve_timer = 0.0;
battle_timer = 0.0;


options_list = file_parser(config_file);

if (options_list.config_random_seed_set) {srand(options_list.config_random_seed);}
else {srand(time(0));}




if (options_list.terrain_defined)
	{
	Terrain terrain_gen (options_list.field_size_x, options_list.field_size_y, options_list.field_scale);
	terrain = terrain_gen;

	if (options_list.terrain_elevation_map_filename != "")
		{
		terrain.generate_elevation_map(options_list.terrain_elevation_map_filename);
		terrain.plot_elevation_map(options_list.terrain_plot_filename);
		}
	else if (options_list.terrain_elevation_grid_filename != "")
		{
		terrain.load_grid(options_list.terrain_elevation_grid_filename);
		terrain.plot_elevation_map(options_list.terrain_plot_filename);
		}

	if ((options_list.terrain_surface_map_filename != "") && (options_list.terrain_surface_atlas_filename != ""))
		{
		if ((options_list.terrain_surface_map_size_x > 0) && (options_list.terrain_surface_map_size_y > 0))
			{	
			terrain.generate_surface_map(options_list.terrain_surface_atlas_filename, options_list.terrain_surface_map_filename, options_list.terrain_surface_map_size_x, options_list.terrain_surface_map_size_y);
			options_list.terrain_surface_defined = true;
			}
		else 
			{
			cout << "No size of surface map specified, exiting..." << endl; exit(0);
			}
		} 



	}


zoom_battlefield(options_list.config_initial_zoom);


while (cmd_input.command != "end")
	{

	if (options_list.config_n_iterations == 1)
		{
		cout << "Command:";
		getline (cin, keyboard_input);
		input_parser(keyboard_input);
		}
	else 
		{cmd_input.command = "reset";}



	if (cmd_input.command == "reset")
		{
		options_list = file_parser(config_file);
		elapsed_time = 0.0;
		evolve_timer = 0.0;
		battle_timer = 0.0;


		if (options_list.config_n_iterations > 1) 
			{
			cout << "Doing iteration " << iteration + 1 << " of " <<  options_list.config_n_iterations << "..." << endl;
			cmd_input.command = "evolve";
			cmd_input.timestep = 500.0;


			}
		}

	if (cmd_input.command == "evolve")
		{
		evolve_timer = 0;
		while (evolve_timer < cmd_input.timestep)
			{
			num_battles = 0;
			num_charges = 0;
			update_move_targets();
			move_units(options_list.config_internal_timestep);
			clear_contacts();
			find_contacts();

			do_mounted_charges();
			
			do_ranged_attacks();

			if (battle_timer > options_list.config_battle_timestep)
				{
				do_battle();

				if (options_list.global_morale_checks)
					{
					do_global_morale_checks();
					}

				battle_timer = 0.0;	

				unit_counter_A = 0;
				unit_counter_B = 0;
				for (int i=0; i< options_list.num_units; i++)
					{
					if ((unit_array[i].get_army_id() == 1) && (unit_array[i].get_inactive() == false))
						{
						unit_counter_A++;
						}
					else if ((unit_array[i].get_army_id() == 2) && (unit_array[i].get_inactive() == false))
						{
						unit_counter_B++;
						}

					}
				if ((unit_counter_B == 0) && (unit_counter_A != 0))
					{
					cout << "Victory for " << options_list.config_army_designation_1 << " after " << elapsed_time << " minutes." << endl;
					cmd_input.command == "wait";
					cmd_input.timestep = evolve_timer; 
					iteration++;
					victory_counter++;

					if ((iteration == options_list.config_n_iterations) && (options_list.config_n_iterations > 1))
						{cmd_input.command = "end";}
					}
				else if ((unit_counter_A == 0) && (unit_counter_B != 0))
					{
					cout << "Victory for " << options_list.config_army_designation_2 <<  " after " << elapsed_time << " minutes." << endl;
					cmd_input.command == "wait";
					cmd_input.timestep = evolve_timer; 
					iteration++;
				
					if ((iteration == options_list.config_n_iterations) && (options_list.config_n_iterations > 1))
						{cmd_input.command = "end";}
					}
				else if ((unit_counter_A == 0) && (unit_counter_B == 0))
					{
					cout << "Draw  after " << elapsed_time << " minutes." << endl;
					cmd_input.command == "wait";
					cmd_input.timestep = evolve_timer; 
					iteration++;
					draw_counter++;
				
					if ((iteration == options_list.config_n_iterations) && (options_list.config_n_iterations > 1))
						{cmd_input.command = "end";}

					}

				formation_housekeeping();


				int num_events;
				num_events = events.manage(unit_array, options_list.num_units, formation_array, options_list.num_formations);
				//cout << "Event manager has " << num_events << " commands ready" << endl;

				for (int i=0; i< num_events; i++)
					{		
					input_parser(events.get_commandline(i));
					}


				cleanup();
			
				}
			evolve_timer += options_list.config_internal_timestep;
			elapsed_time += options_list.config_internal_timestep;
			battle_timer += options_list.config_internal_timestep;


			}

		if ((options_list.config_auto_battlefield_display) && (options_list.config_n_iterations == 1))
			{
			show_battlefield();
			}
		if (options_list.config_auto_report_units)
			{
			report_unit_summary(true);
			}


	}


	if (cmd_input.command == "show_battlefield") 
		{
		show_battlefield();
		}

	if (cmd_input.command == "report_status")
		{
		report_battle_summary();
		}
	if (cmd_input.command == "report_composition")
		{
		report_army_composition();
		}
	if (cmd_input.command == "get_elevation")
		{
		cout << "Terrain elevation is " << terrain.get_elevation(cmd_input.par_1, cmd_input.par_2) << " m" << endl;
		}
	if (cmd_input.command == "get_slope")
		{
		cout << "Terrain slope is " << terrain.get_slope(cmd_input.par_1, cmd_input.par_2, cmd_input.par_3) << endl;
		}
	if (cmd_input.command == "get_movement_factor")
		{
		cout << "Movement reduction across terrain is " << terrain.get_movement_slowdown(cmd_input.par_1, cmd_input.par_2) << endl;
		}
	if (cmd_input.command == "get_movement_max")
		{
		cout << "Movement max. cap across terrain is " << terrain.get_movement_max(cmd_input.par_1, cmd_input.par_2) << endl;
		}
	if (cmd_input.command == "get_hard_cover")
		{
		cout << "Terrain hard cover is " << terrain.get_hard_cover(cmd_input.par_1, cmd_input.par_2) << endl;
		}
	if (cmd_input.command == "get_soft_cover")
		{
		cout << "Terrain soft cover is " << terrain.get_soft_cover(cmd_input.par_1, cmd_input.par_2) << endl;
		}
	if (cmd_input.command == "get_los")
		{
		if (check_line_of_sight(cmd_input.int_1, cmd_input.int_2))
			{cout << "Line of sight is clear." << endl;}
		else
			{cout << "Line of sight is blocked." << endl;}
		}
	if (cmd_input.command == "plot_positions")
		{
		terrain.plot_positions(unit_array, formation_array[cmd_input.formation_id]);
		}
		
	}

if (options_list.config_n_iterations == 1)
	{
	for (int i=0; i< options_list.num_units;i++) {unit_array[i].list_damage();}	
	}
else 
	{
	cout << "Out of " << iteration << " battles " << victory_counter << " victories for " << options_list.config_army_designation_1 << " " << iteration-victory_counter-draw_counter << " for " << options_list.config_army_designation_2 << " and " << draw_counter << " draws." << endl;
	}

exit(0);


for (int j=0; j< options_list.num_units; j++)
		{unit_array[j].list_damage();}

/*
for (int i=0; i< 10; i++)
	{
	process_close_combat(0,1);
	unit_array[0].list_damage();
	unit_array[1].list_damage();
	}
*/

}
